import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions
from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task and prepare recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # PDDL object keys
        bottom_side_pos = positions['side-pos-bottom']
        bottom_anchor_pos = positions['anchor-pos-bottom']

        # === Compute Target Orientation ===
        # Rotate gripper around Z by 90°
        target_quat = R.from_euler('z', 90, degrees=True).as_quat()

        # === Oracle Plan Execution ===

        # Frozen Code Start
        obs, reward, done = rotate(env, task, target_quat)
        obs, reward, done = move(env, task, np.array(bottom_side_pos)
        obs, reward, done = move(env, task, np.array(bottom_anchor_pos)
        # Frozen Code End

        # Check if any early termination occurred
        if done:
            print("[Task] Terminated during initial steps, reward:", reward)
            return

        # Step 4: Pick at the anchor position of the bottom drawer
        print(f"[Task] Picking drawer at anchor position {bottom_anchor_pos}")
        obs, reward, done = pick(env, task, np.array(bottom_anchor_pos))
        if done:
            print("[Task] Terminated during pick, reward:", reward)
            return

        # Step 5: Pull the drawer to open it
        print("[Task] Pulling the drawer to open")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Task] Terminated during pull, reward:", reward)
            return

        # All steps complete
        print("[Task] Completed all plan steps successfully. Final reward:", reward)

    finally:
        # Ensure environment teardown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()