from scipy.spatial.transform import Rotation as R
import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Define required orientations ===
        # Compute a 90° rotation about the Z axis (from zero_deg to ninety_deg)
        ninety_quat = R.from_euler('z', 90, degrees=True).as_quat()

        # === Oracle Plan Execution ===

        # Step 1: rotate gripper from zero_deg to ninety_deg
        print("[Task] Step 1: Rotating gripper to ninety_deg")
        obs, reward, done = rotate(env, task, target_quat=ninety_quat)
        if done:
            print("[Task] Early termination after rotation. Reward:", reward)
            return

        # Step 2: move gripper from nowhere-pos to side-pos-bottom
        print("[Task] Step 2: Moving gripper to side position bottom")
        target_side_bottom = positions['bottom_side_pos']
        obs, reward, done = move(env, task, target_pos=target_side_bottom)
        if done:
            print("[Task] Early termination after moving to side position. Reward:", reward)
            return

        # Step 3: move gripper from side-pos-bottom to anchor-pos-bottom
        print("[Task] Step 3: Moving gripper to anchor position bottom")
        target_anchor_bottom = positions['bottom_anchor_pos']
        obs, reward, done = move(env, task, target_pos=target_anchor_bottom)
        if done:
            print("[Task] Early termination after moving to anchor position. Reward:", reward)
            return

        # Step 4: pick the bottom drawer handle at anchor position
        print("[Task] Step 4: Picking bottom drawer handle")
        obs, reward, done = pick(env, task, target_pos=target_anchor_bottom)
        if done:
            print("[Task] Early termination after pick. Reward:", reward)
            return

        # Step 5: pull the bottom drawer to open it
        print("[Task] Step 5: Pulling bottom drawer to open")
        pull_distance = 0.1  # meters
        obs, reward, done = pull(env, task, pull_distance=pull_distance)
        if done:
            print("[Task] Drawer opened successfully! Reward:", reward)
        else:
            print("[Task] Drawer pull executed but done=False. Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()