import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running the open-drawer task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Expected keys include:
        #   'bottom_side_pos', 'bottom_anchor_pos', etc.
        # Convert to numpy arrays if not already
        bottom_side_pos = np.array(positions['bottom_side_pos'])
        bottom_anchor_pos = np.array(positions['bottom_anchor_pos'])
        
        # === Define orientation quaternions for zero_deg and ninety_deg ===
        # We assume rotation about z-axis
        zero_quat = R.from_euler('z', 0, degrees=True).as_quat()
        ninety_quat = R.from_euler('z', 90, degrees=True).as_quat()

        # === Execute Oracle Plan Steps ===

        # Step 1: rotate(gripper, zero_deg -> ninety_deg)
        print("[Plan] Step 1: rotate gripper from zero_deg to ninety_deg")
        obs, reward, done = rotate(env, task, target_quat=ninety_quat)
        if done:
            print("Task ended prematurely after rotate.")
            return

        # Step 2: move(gripper, to bottom_side_pos)
        print(f"[Plan] Step 2: move gripper to side position {bottom_side_pos}")
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("Task ended prematurely after move to side position.")
            return

        # Step 3: move(gripper, to bottom_anchor_pos)
        print(f"[Plan] Step 3: move gripper to anchor position {bottom_anchor_pos}")
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("Task ended prematurely after move to anchor position.")
            return

        # Step 4: pick(gripper, bottom drawer at anchor position)
        print(f"[Plan] Step 4: pick bottom drawer at {bottom_anchor_pos}")
        obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("Task ended prematurely after pick.")
            return

        # Step 5: pull(gripper, bottom drawer)
        print("[Plan] Step 5: pull bottom drawer to open it")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("Task ended prematurely after pull.")
            return

        # Check final success
        if not done:
            print("Task completed the open-drawer plan. Final reward:", reward)
        else:
            print("Task reports done=True. Final reward:", reward)

    except Exception as e:
        print("Exception during task execution:", str(e))
        raise
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()