import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        bottom_side_pos = np.array(positions['bottom_side_pos'])
        bottom_anchor_pos = np.array(positions['bottom_anchor_pos'])

        # === Oracle Plan Execution ===

        # Step 1: Rotate gripper from zero_deg to ninety_deg
        print("[Task] Step 1: Rotating gripper to ninety_deg")
        quat_90 = R.from_euler('xyz', [0, 0, 90], degrees=True).as_quat()
        obs, reward, done = rotate(env, task, quat_90)
        if done:
            print("[Task] Early termination after rotate")
            return

        # Step 2: Move to side position of the bottom drawer
        print("[Task] Step 2: Moving to side position of bottom drawer")
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Task] Early termination after move-to-side")
            return

        # Step 3: Move to anchor position of the bottom drawer
        print("[Task] Step 3: Moving to anchor position of bottom drawer")
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Task] Early termination after move-to-anchor")
            return

        # Step 4: Pick the bottom drawer handle
        print("[Task] Step 4: Picking bottom drawer handle")
        obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Task] Early termination after pick")
            return

        # Step 5: Pull to open the bottom drawer fully
        print("[Task] Step 5: Pulling bottom drawer to open fully")
        pull_distance = 0.2  # adjust based on drawer depth
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
        if done:
            print("[Task] Drawer opened and task completed successfully! Reward:", reward)
        else:
            print("[Task] Pull step finished but done=False. Current reward:", reward)

    except Exception as e:
        print(f"[Task] Exception during execution: {e}")
    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()