import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        side_pos_bottom = np.array(positions['bottom_side_pos'])
        anchor_pos_bottom = np.array(positions['bottom_anchor_pos'])

        # === Oracle Plan Execution ===

        # Step 1: rotate gripper from zero_deg to ninety_deg
        print("[Plan] Step 1: rotate gripper to ninety_deg")
        from scipy.spatial.transform import Rotation as R
        target_quat = R.from_euler('y', 90, degrees=True).as_quat()
        # [Frozen Code Start]
        obs, reward, done = rotate(env, task, target_quat)
        obs, reward, done = move(env, task, side_pos_bottom)
        # [Frozen Code End]
        if done:
            print("[Task] Terminated early after rotate and first move.")
            return

        # Step 2: move gripper from side-pos-bottom to anchor-pos-bottom (this is move step 3 logically)
        print("[Plan] Step 3: move gripper to anchor-pos-bottom")
        obs, reward, done = move(env, task, anchor_pos_bottom)
        if done:
            print("[Task] Terminated early after second move.")
            return

        # Step 4: pick drawer at anchor-pos-bottom
        print("[Plan] Step 4: pick drawer at anchor-pos-bottom")
        obs, reward, done = pick(env, task, anchor_pos_bottom)
        if done:
            print("[Task] Terminated early after pick.")
            return

        # Step 5: pull drawer to open it
        print("[Plan] Step 5: pull drawer to open")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Task] Drawer opened successfully! Reward:", reward)
        else:
            print("[Task] Drawer pull completed but done=False. Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()