import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        
        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)
        
        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)
        
        # === Retrieve Object Positions ===
        positions = get_object_positions()
        
        # Choose an unlocked drawer to open. 'top' and 'middle' are locked, so pick 'bottom'
        side_pos = positions['bottom_side_pos']
        anchor_pos = positions['bottom_anchor_pos']
        
        # Precompute target orientation quaternion for ninety_deg
        target_quat_90 = np.array([0.0, 0.0, 0.7071, 0.7071])
        
        # === Execute Oracle Plan ===
        # Step 1: Rotate gripper to ninety_deg orientation
        print("[Task] Rotating gripper to ninety_deg orientation")
        # Frozen Code Start
        obs, reward, done = rotate(env, task, target_quat=target_quat_90)
        # Frozen Code End
        if done:
            print("[Task] Task ended during rotate.")
            return
        
        # Step 2: Move to the side position of the drawer
        print(f"[Task] Moving to side position: {side_pos}")
        try:
            obs, reward, done = move(env, task, target_pos=np.array(side_pos))
        except Exception as e:
            print(f"[Error] move to side position failed: {e}")
            return
        if done:
            print("[Task] Task ended during move to side.")
            return
        
        # Step 3: Move to the anchor position of the drawer
        print(f"[Task] Moving to anchor position: {anchor_pos}")
        try:
            obs, reward, done = move(env, task, target_pos=np.array(anchor_pos))
        except Exception as e:
            print(f"[Error] move to anchor position failed: {e}")
            return
        if done:
            print("[Task] Task ended during move to anchor.")
            return
        
        # Step 4: Pick (grasp) the drawer at anchor position
        print(f"[Task] Picking at anchor position: {anchor_pos}")
        try:
            obs, reward, done = pick(env, task, target_pos=np.array(anchor_pos))
        except Exception as e:
            print(f"[Error] pick at anchor position failed: {e}")
            return
        if done:
            print("[Task] Task ended during pick.")
            return
        
        # Step 5: Pull the drawer to open it
        print("[Task] Pulling drawer to open it")
        try:
            obs, reward, done = pull(env, task, pull_distance=0.05, pull_axis='x')
        except Exception as e:
            print(f"[Error] pull failed: {e}")
            return
        
        if done or (reward is not None and reward > 0):
            print(f"[Task] Successfully opened drawer. Reward: {reward}")
        else:
            print(f"[Task] Drawer pull did not complete. done={done}, reward={reward}")
    
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)
    
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()