import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move  # we only need pick and place for this task

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: pick the rubbish from the table and place it into the bin.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Expecting keys: 'rubbish', 'table', 'bin', plus any other objects
        
        # Extract the three relevant positions
        rubbish_pos = positions['rubbish']
        table_pos   = positions['table']
        bin_pos     = positions['bin']

        # === Oracle Plan Execution ===

        # Step 1: pick rubbish from table
        print("[Task] Step 1: Picking 'rubbish' from 'table'")
        obs, reward, done = pick(
            env,
            task,
            target_pos=rubbish_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print(f"[Task] Ended during pick(rubbish). Reward: {reward}")
            return

        # Step 2: place rubbish into bin
        print("[Task] Step 2: Placing 'rubbish' into 'bin'")
        obs, reward, done = place(
            env,
            task,
            target_pos=bin_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print(f"[Task] Ended after place(rubbish). Reward: {reward}")
            return

        # Final status
        if reward > 0 or done:
            print(f"[Task] Plan executed successfully! Final reward: {reward}")
        else:
            print(f"[Task] Plan executed, but task not marked done. Reward: {reward}")

    except Exception as e:
        print(f"[Task] Exception during plan execution: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()