import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Define object positions
        table_pos = positions['table']
        bin_pos = positions['bin']
        rubbish_pos = positions['item1']  # Assuming item1 is the rubbish

        # === Execute the Plan ===
        # Step 1: Pull the drawer to unlock the cabinet
        print("[Task] Pulling the drawer to unlock the cabinet.")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='y')
        if done:
            print("[Task] Task ended after pulling the drawer!")
            return

        # Step 2: Rotate the gripper to the correct angle
        print("[Task] Rotating the gripper to ninety degrees.")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, np.sin(np.pi/4), np.cos(np.pi/4)]))  # Rotate to 90 degrees
        if done:
            print("[Task] Task ended after rotating the gripper!")
            return

        # Step 3: Pick up the rubbish
        print("[Task] Picking up the rubbish.")
        obs, reward, done = pick(env, task, target_pos=rubbish_pos)
        if done:
            print("[Task] Task ended after picking up the rubbish!")
            return

        # Step 4: Place the rubbish in the bin
        print("[Task] Placing the rubbish in the bin.")
        obs, reward, done = place(env, task, target_pos=bin_pos)
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()