import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *  # Importing predefined skills
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Define object positions based on the provided object list
        rubbish_pos = positions['item3']  # Assuming item3 is the rubbish
        bin_pos = positions['bin']  # Assuming bin is the drop location

        # === Execute the Plan ===
        # Step 1: Rotate the gripper to ninety degrees
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 1, 0]))  # Assuming a quaternion for 90 degrees
        if done:
            print("[Task] Gripper rotated successfully!")
        else:
            print("[Task] Failed to rotate the gripper.")

        # Step 2: Move to the side position of the bottom drawer
        obs, reward, done = move(env, task, target_pos=positions['bottom_side_pos'])
        if done:
            print("[Task] Moved to the side position of the bottom drawer successfully!")
        else:
            print("[Task] Failed to move to the side position.")

        # Step 3: Move to the anchor position of the bottom drawer
        obs, reward, done = move(env, task, target_pos=positions['bottom_anchor_pos'])
        if done:
            print("[Task] Moved to the anchor position of the bottom drawer successfully!")
        else:
            print("[Task] Failed to move to the anchor position.")

        # Step 4: Pick the bottom drawer
        obs, reward, done = pick(env, task, target_pos=positions['bottom_anchor_pos'])
        if done:
            print("[Task] Picked the bottom drawer successfully!")
        else:
            print("[Task] Failed to pick the bottom drawer.")

        # Step 5: Pull open the bottom drawer
        obs, reward, done = pull(env, task, pull_distance=0.1)  # Assuming a pull distance
        if done:
            print("[Task] Drawer pulled open successfully!")
        else:
            print("[Task] Failed to pull the drawer open.")

        # Step 6: Pick the rubbish from the table
        obs, reward, done = pick(env, task, target_pos=rubbish_pos)
        if done:
            print("[Task] Picked the rubbish successfully!")
        else:
            print("[Task] Failed to pick the rubbish.")

        # Step 7: Place the rubbish into the bin
        obs, reward, done = place(env, task, target_pos=bin_pos)
        if done:
            print("[Task] Rubbish placed in the bin successfully!")
        else:
            print("[Task] Failed to place rubbish in the bin.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()