import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Define object positions
        top_drawer_pos = positions['top_anchor_pos']
        rubbish_pos = positions['item1']  # Assuming rubbish is item1
        bin_pos = positions['bin']

        # Step 1: Pull the top drawer
        print("[Task] Pulling the top drawer.")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='y', max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Successfully pulled the drawer.")
        else:
            print("[Task] Failed to pull the drawer.")

        # Step 2: Rotate the gripper to the correct orientation
        print("[Task] Rotating the gripper to ninety degrees.")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 1, 0]), max_steps=100, threshold=0.05, timeout=10.0)
        if done:
            print("[Task] Successfully rotated the gripper.")
        else:
            print("[Task] Failed to rotate the gripper.")

        # Step 3: Move to the side position of the drawer
        print("[Task] Moving to the side position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=positions['bottom_side_pos'], max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Successfully moved to the side position.")
        else:
            print("[Task] Failed to move to the side position.")

        # Step 4: Move to the anchor position of the drawer
        print("[Task] Moving to the anchor position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=positions['bottom_anchor_pos'], max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Successfully moved to the anchor position.")
        else:
            print("[Task] Failed to move to the anchor position.")

        # Step 5: Pick the drawer
        print("[Task] Picking the bottom drawer.")
        obs, reward, done = pick_drawer(env, task, gripper='gripper', d='bottom', p='anchor-pos-bottom')
        if done:
            print("[Task] Successfully picked the drawer.")
        else:
            print("[Task] Failed to pick the drawer.")

        # Step 6: Pull the drawer open
        print("[Task] Pulling the drawer open.")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='y', max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Successfully pulled the drawer open.")
        else:
            print("[Task] Failed to pull the drawer open.")

        # Step 7: Pick the rubbish
        print("[Task] Picking the rubbish.")
        obs, reward, done = pick(env, task, target_pos=rubbish_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        if done:
            print("[Task] Successfully picked the rubbish.")
        else:
            print("[Task] Failed to pick the rubbish.")

        # Step 8: Place the rubbish in the bin
        print("[Task] Placing rubbish in the bin.")
        obs, reward, done = place(env, task, target_pos=bin_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()