import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Define object positions
        plate_pos = positions['plate']
        tomato1_pos = positions['item1']
        tomato2_pos = positions['item2']
        bottom_drawer_pos = positions['bottom_anchor_pos']

        # === Task Execution Plan ===
        # Step 1: Pull the drawer to unlock the cabinet
        print("[Task] Pulling the drawer to unlock the cabinet.")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='y')
        
        # Step 2: Rotate the gripper to the correct angle
        print("[Task] Rotating the gripper to ninety degrees.")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, np.sin(np.pi/4), np.cos(np.pi/4)]))  # Rotate to 90 degrees
        
        # Step 3: Move to the side position of the bottom drawer
        print("[Task] Moving to the side position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=bottom_drawer_pos + np.array([0, 0, 0.1]))  # Adjust height slightly
        
        # Step 4: Move to the anchor position of the bottom drawer
        print("[Task] Moving to the anchor position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=bottom_drawer_pos)
        
        # Step 5: Pick the bottom drawer
        print("[Task] Picking the bottom drawer.")
        obs, reward, done = pick(env, task, target_pos=bottom_drawer_pos)
        
        # Step 6: Pull the drawer to unlock the cabinet
        print("[Task] Pulling the drawer to unlock the cabinet.")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='y')
        
        # Step 7: Pick the first tomato from the table
        print("[Task] Picking tomato1 from the table.")
        obs, reward, done = pick(env, task, target_pos=tomato1_pos)
        
        # Step 8: Place the first tomato onto the plate
        print("[Task] Placing tomato1 onto the plate.")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        
        # Step 9: Pick the second tomato from the table
        print("[Task] Picking tomato2 from the table.")
        obs, reward, done = pick(env, task, target_pos=tomato2_pos)
        
        # Step 10: Place the second tomato onto the plate
        print("[Task] Placing tomato2 onto the plate.")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        
        print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()