import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Define object positions
        tomato1_pos = positions['item1']
        tomato2_pos = positions['item2']
        plate_pos = positions['plate']
        bottom_drawer_pos = positions['bottom_anchor_pos']  # Assuming this is the drawer to pull

        # Step 1: Rotate the gripper to the correct orientation
        print("[Task] Rotating the gripper to the correct orientation.")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 0, 1]))  # Adjust quaternion as needed
        if done:
            print("[Task] Task ended after rotating the gripper!")
            return

        # Step 2: Move to the side position of the bottom drawer
        print("[Task] Moving to the side position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=positions['bottom_side_pos'])
        if done:
            print("[Task] Task ended after moving to the side position!")
            return

        # Step 3: Move to the anchor position of the bottom drawer
        print("[Task] Moving to the anchor position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=bottom_drawer_pos)
        if done:
            print("[Task] Task ended after moving to the anchor position!")
            return

        # Step 4: Pick the bottom drawer
        print("[Task] Picking the bottom drawer.")
        obs, reward, done = pick(env, task, target_pos=bottom_drawer_pos)
        if done:
            print("[Task] Task ended after picking the bottom drawer!")
            return

        # Step 5: Pull the drawer to unlock the cabinet
        print("[Task] Pulling the drawer to unlock the cabinet.")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='y')  # Adjust distance and axis as needed
        if done:
            print("[Task] Task ended after pulling the drawer!")
            return

        # Step 6: Pick the first tomato
        print("[Task] Picking the first tomato.")
        obs, reward, done = pick(env, task, target_pos=tomato1_pos)
        if done:
            print("[Task] Task ended after picking the first tomato!")
            return

        # Step 7: Place the first tomato on the plate
        print("[Task] Placing the first tomato on the plate.")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Task ended after placing the first tomato!")
            return

        # Step 8: Pick the second tomato
        print("[Task] Picking the second tomato.")
        obs, reward, done = pick(env, task, target_pos=tomato2_pos)
        if done:
            print("[Task] Task ended after picking the second tomato!")
            return

        # Step 9: Place the second tomato on the plate
        print("[Task] Placing the second tomato on the plate.")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()