import numpy as np
from env import setup_environment, shutdown_environment
from skill_code import pick, place, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        _, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Define object positions
        plate_pos = positions['plate']
        tomato1_pos = positions['item1']
        tomato2_pos = positions['item2']

        # Step 1: Pull the top drawer to unlock the cabinet
        print("[Task] Pulling the top drawer to unlock the cabinet.")
        obs, _, done = pull(env, task, pull_distance=0.1, pull_axis='y', max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Drawer pulled successfully!")
        else:
            print("[Task] Failed to pull the drawer.")

        # Step 2: Pick the first tomato from the table
        print("[Task] Picking tomato1 from the table.")
        obs, _, done = pick(env, task, target_pos=tomato1_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        if done:
            print("[Task] Tomato1 picked successfully!")
        else:
            print("[Task] Failed to pick tomato1.")

        # Step 3: Place the first tomato onto the plate
        print("[Task] Placing tomato1 onto the plate.")
        obs, _, done = place(env, task, target_pos=plate_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        if done:
            print("[Task] Tomato1 placed successfully!")
        else:
            print("[Task] Failed to place tomato1.")

        # Step 4: Pick the second tomato from the table
        print("[Task] Picking tomato2 from the table.")
        obs, _, done = pick(env, task, target_pos=tomato2_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        if done:
            print("[Task] Tomato2 picked successfully!")
        else:
            print("[Task] Failed to pick tomato2.")

        # Step 5: Place the second tomato onto the plate
        print("[Task] Placing tomato2 onto the plate.")
        obs, _, done = place(env, task, target_pos=plate_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        if done:
            print("[Task] Tomato2 placed successfully!")
        else:
            print("[Task] Failed to place tomato2.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()