import numpy as np
from env import setup_environment, shutdown_environment
from skill_code import *  
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    env, task = setup_environment()
    try:
        obs = task.reset()[1]
        init_video_writers(obs)

        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        positions = get_object_positions()
        bottom_side_pos = positions['bottom_side_pos']
        bottom_anchor_pos = positions['bottom_anchor_pos']

        print("[Task] Rotating gripper to 90 degrees.")
        obs, _, done = rotate(env, task, target_quat=np.array([0, 0, 1, 0]), max_steps=100, threshold=0.05, timeout=10.0)
        if done:
            print("[Task] Task ended during rotation!")
            return

        print("[Task] Moving to side position of the bottom drawer.")
        obs, _, done = move(env, task, target_pos=bottom_side_pos, max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Task ended during move to side position!")
            return

        print("[Task] Moving to anchor position of the bottom drawer.")
        obs, _, done = move(env, task, target_pos=bottom_anchor_pos, max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Task ended during move to anchor position!")
            return

        print("[Task] Picking the bottom drawer.")
        obs, _, done = pick(env, task, target_pos=bottom_anchor_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        if done:
            print("[Task] Task ended during pick!")
            return

        print("[Task] Pulling the bottom drawer to open it.")
        obs, _, done = pull(env, task, pull_distance=0.1, pull_axis='y', max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Task ended during pull!")
            return

        print("[Task] Successfully opened the bottom drawer!")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()