import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Setting target positions based on observation
        anchor_pos_bottom = positions['bottom_anchor_pos']
        side_pos_bottom = positions['bottom_side_pos']
        anchor_pos_top = positions['top_anchor_pos']
        side_pos_top = positions['top_side_pos']

        # === Execute Task Plan ===
        try:
            # Step 1: Move gripper to bottom anchor position and pick drawer
            print("[Task] Moving gripper to bottom anchor position and picking drawer.")
            obs, reward, done = pick(env, task, target_pos=anchor_pos_bottom)

            if done:
                print("[Task] Task ended after picking!")
                return

            # Step 2: Pull drawer to open it
            print("[Task] Pulling drawer to open it.")
            obs, reward, done = pull(env, task, pull_distance=0.5, pull_axis='z')  # Assumed axis and distance

            if done:
                print("[Task] Task ended after pulling!")
                return

            # Step 3: Move gripper to top position after opening the drawer
            print("[Task] Moving gripper to top position.")
            obs, reward, done = move(env, task, target_pos=side_pos_top)

            if done:
                print("[Task] Task completed successfully! Reward:", reward)
                return
            else:
                print("[Task] Task not completed yet (done=False).")

        except Exception as e:
            print(f"[Task] Encountered an error: {e}")
            
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()