import json
import subprocess
import tempfile
from pathlib import Path
from typing import Tuple
import sys
sys.path.append("Input_Your_Path")
from code_gen.llm import llm_response

def print_validation_result(success: bool, title: str):
    line_width = 60
    PURPLE = "\033[95m"
    GREEN = "\033[92m"
    RED = "\033[91m"
    RESET = "\033[0m"

    print(PURPLE + "=" * line_width + RESET)
    print(title.center(line_width))
    print(PURPLE + "=" * line_width + RESET)

    result_msg = "PASS" if success else "FAIL"
    color = GREEN if success else RED
    print((color + f"Result: {result_msg}" + RESET).center(line_width))
    print(PURPLE + "=" * line_width + RESET)

def _syntax_ok(code: str, lang: str) -> Tuple[bool, str]:
    if lang == "bash":
        with tempfile.NamedTemporaryFile("w", suffix=".sh", delete=False) as f:
            f.write(code)
            fname = f.name
        try:
            subprocess.run(["shellcheck", fname], check=True,
                           stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
            return True, "ShellCheck passed."
        except subprocess.CalledProcessError as e:
            return False, e.stdout.decode()
        finally:
            Path(fname).unlink(missing_ok=True)

    if lang == "python":
        with tempfile.NamedTemporaryFile("w", suffix=".py", delete=False) as f:
            f.write(code)
            fname = f.name
        try:
            subprocess.run(["pylint", "--score", "n", fname], check=True,
                           stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
            return True, "Pylint passed."
        except subprocess.CalledProcessError as e:
            return False, e.stdout.decode()
        finally:
            Path(fname).unlink(missing_ok=True)

    return True, "No syntax checker configured."

def _prompt_func(code: str) -> str:
    sys_prompt = (
        "You are a senior software engineer. "
        "Summarize what the given source code does, focusing on its external behaviour and key steps, not on syntax. "
        "Return only plain sentences in English."
    )
    user_prompt = f"<CODE>\n{code}\n</CODE>"
    full_prompt = sys_prompt + "\n\n" + user_prompt
    return llm_response("openai", full_prompt).strip()

def _prompt_sim(task: str, code_func: str) -> dict:
    sys_prompt = (
        "You judge whether a code functionality achieves a task. "
        "Reply in JSON with keys ok (true/false) and reason (one sentence)."
    )
    user_prompt = json.dumps({"task": task, "code_func": code_func}, ensure_ascii=False)
    full_prompt = sys_prompt + "\n\n" + user_prompt
    raw = llm_response("openai", full_prompt)
    try:
        data = json.loads(raw)
        return {"ok": bool(data.get("ok")), "reason": str(data.get("reason", ""))}
    except Exception:
        return {"ok": False, "reason": "LLM JSON parsing failed."}

def _prompt_diff(task: str, code_func: str) -> str:
    sys_prompt = (
        "List *all* semantic differences between TASK and CODE_FUNC. "
        "If there is no difference, output the single token EXACT. Return a numbered markdown list or EXACT."
    )
    user_prompt = f"TASK:\n{task}\n\nCODE_FUNC:\n{code_func}"
    full_prompt = sys_prompt + "\n\n" + user_prompt
    return llm_response("openai", full_prompt).strip()

def _prompt_refine(task: str, code: str, feedback: str) -> str:
    sys_prompt = (
        "You are a senior software engineer. "
        "Refine the following code so that it fully satisfies the TASK, based on the FEEDBACK. "
        "Return only the corrected source code without any explanation."
        "Do not use code block. Output should be in plain text format."
    )
    user_prompt = (
        f"TASK:\n{task}\n\n"
        f"CODE:\n{code}\n\n"
        f"FEEDBACK:\n{feedback}"
    )
    full_prompt = sys_prompt + "\n\n" + user_prompt
    return llm_response("openai", full_prompt).strip()

def codesift_validate(
    code: str,
    task_desc: str,
    lang: str = "python",
) -> Tuple[bool, dict]:
    details = {}

    print("Step 0: Running syntax check...")
    syntax_pass, syntax_feedback = _syntax_ok(code, lang)
    details["syntax_pass"] = syntax_pass
    details["syntax_feedback"] = syntax_feedback
    print_validation_result(syntax_pass, "Syntax Check")

    print("Step 1: Summarizing code functionality...")
    code_func = _prompt_func(code)
    details["code_func"] = code_func
    print_validation_result(True, "Functionality Summary")

    print("Step 2: Checking semantic similarity...")
    sim = _prompt_sim(task_desc, code_func)
    details["sim"] = sim
    print_validation_result(sim["ok"], "Semantic Similarity")

    print("Step 3: Listing semantic differences...")
    diff = _prompt_diff(task_desc, code_func)
    details["diff"] = diff
    diff_ok = diff.strip().upper() == "EXACT"
    print_validation_result(diff_ok, "Semantic Diff")

    print("Step 4: Final decision...")
    is_correct = syntax_pass and sim["ok"] and diff_ok
    details["is_correct"] = is_correct
    print_validation_result(is_correct, "Final Validation")

    if not is_correct:
        print("Step 5: Refining code...")
        feedbacks = []
        if not syntax_pass:
            feedbacks.append(f"Syntax error: {syntax_feedback}")
        if not sim["ok"]:
            feedbacks.append(f"Similarity failed: {sim['reason']}")
        if not diff_ok:
            feedbacks.append(f"Differences found: {diff}")
        combined_feedback = "\n".join(feedbacks)
        refined_code = _prompt_refine(task_desc, code, combined_feedback)
        details["refined_code"] = refined_code
        print("Refined code generated.")

    return is_correct, details