# Copyright 2024-present the HuggingFace Inc. team.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
from __future__ import annotations

import warnings
from typing import Optional

import bitsandbytes as bnb
import torch

from peft.import_utils import is_bnb_4bit_available, is_bnb_available
from peft.tuners.tuners_utils import check_adapters_to_merge
from peft.utils.integrations import dequantize_bnb_weight
from peft.utils.other import transpose

from .layer import HOFTLayer


if is_bnb_available():

    class Linear8bitLt(torch.nn.Module, HOFTLayer):
        def __init__(
            self,
            base_layer: torch.nn.Module,
            adapter_name: str,
            r: int = 0,
            hoft_dropout: float = 0.0,
            fan_in_fan_out: bool = False,
            init_weights: bool = True,
            use_shoft: bool = False,
            **kwargs,
        ) -> None:
            super().__init__()
            HOFTLayer.__init__(self, base_layer, **kwargs)
            self.fan_in_fan_out = fan_in_fan_out
            self.use_shoft = use_shoft

            self._active_adapter = adapter_name
            self.update_layer(adapter_name, r, hoft_dropout, init_weights, use_shoft)

        def merge(self, safe_merge: bool = False, adapter_names: Optional[list[str]] = None) -> None:
            if self.merged:
                warnings.warn(
                    f"Already following adapters were merged {','.join(self.merged_adapters)}. "
                    f"You are now additionally merging {','.join(self.active_adapters)}."
                )

            adapter_names = check_adapters_to_merge(self, adapter_names)
            if not adapter_names:
                return

            for active_adapter in adapter_names:
                if active_adapter not in self.hoft_U.keys():
                    continue

                warnings.warn(
                    "Merge hoft module to 8-bit linear may get different generations due to rounding errors."
                )

                weight = self.get_base_layer().weight
                state = self.get_base_layer().state
                if state.SCB is None:
                    state.SCB = weight.SCB

                output = dequantize_bnb_weight(weight, state)
                hoft_data = self.get_delta_weight(active_adapter, output)

                w_data = output.to(hoft_data.dtype).to(hoft_data.device) + hoft_data

                if safe_merge and not torch.isfinite(w_data).all():
                    raise ValueError(
                        f"NaNs detected in the merged weights. The adapter {active_adapter} seems to be broken"
                    )

                self.get_base_layer().weight = bnb.nn.Int8Params(
                    w_data.to("cpu"), requires_grad=False, has_fp16_weights=weight.has_fp16_weights
                ).to(weight.device)
                state.reset_grads()
                self.merged_adapters.append(active_adapter)

        def unmerge(self) -> None:
            if not self.merged:
                warnings.warn("Already unmerged. Nothing to do")
                return

            while len(self.merged_adapters) > 0:
                active_adapter = self.merged_adapters.pop()
                if active_adapter not in self.hoft_U.keys():
                    continue
                warnings.warn(
                    "Unmerge hoft module to 8-bit linear may get different generations due to rounding errors."
                )
   
                weight = self.get_base_layer().weight
                state = self.get_base_layer().state
                if state.SCB is None:
                    state.SCB = weight.SCB
                output = dequantize_bnb_weight(weight, state=state)
                hoft_data = self.get_delta_weight(active_adapter, output)

                w_data = output.to(hoft_data.dtype).to(hoft_data.device) - hoft_data

                self.get_base_layer().weight = bnb.nn.Int8Params(
                    w_data.to("cpu"), requires_grad=False, has_fp16_weights=weight.has_fp16_weights
                ).to(weight.device)
                state.reset_grads()

        def get_delta_weight(self, adapter, base_weight) -> torch.Tensor:
            """
            Compute the delta weight for the given adapter.

            Args:
                adapter (str): The name of the adapter for which the delta weight should be computed.

            Returns:
                torch.Tensor: The computed delta weight for the VeRA adapter.

            Note:
                This method implements the HOFT/RHOFT-specific weight update. Unlike HOFT, RHOFT uses a shared projection
                matrix (rhoft_A) across all layers, along with per-layer trainable parameters (hoft_U).
            """
            # Retrieve shared projection matrices
            U = self.hoft_U[adapter]
            V = self.hoft_V[adapter]

            device = U.device
            dtype = U.dtype

            # In case users wants to merge the adapter weights that are in
            # (b)float16 while being on CPU, we need to cast the weights to float32, perform the merge and then cast back to
            # (b)float16 because some CPUs have slow bf16/fp16 matmuls.
            cast_to_fp32 = device.type == "cpu" and (dtype == torch.float16 or dtype == torch.bfloat16)


            if cast_to_fp32:
                U = U.float()
                V = V.float()

            WV = torch.matmul(V.T, V)
            DV = WV.diagonal().reciprocal().mul(2)
            SV = DV * WV.triu(1) * DV
            SV.diagonal().sub_(DV)
            preresult = torch.eye(self.in_features) + torch.matmul(V, torch.matmul(SV, V.T))
            result = torch.matmul(preresult, base_weight.T)
            
            if self.use_shoft:
                magnitude = self.hoft_magnitude_vector[adapter]
                if cast_to_fp32:
                    magnitude = magnitude.float()
                result = magnitude * result

            result_U = torch.matmul(result, U)
            WU = torch.matmul(U.T, U)
            DU = WU.diagonal().reciprocal().mul(2)
            SU = DU * WU.triu(1) * DU
            SU.diagonal().sub_(DU)
            
            result = result + torch.matmul(result_U, torch.matmul(SU, U.T))
            result = result - base_weight

            if cast_to_fp32:
                output_tensor = output_tensor.to(dtype=dtype)

                self.hoft_U[adapter].data = U.to(dtype=dtype)
                self.hoft_V[adapter].data = V.to(dtype=dtype)
                if self.use_shoft:
                    self.hoft_magnitude_vector[adapter].data = magnitude.to(dtype=dtype)

            return output_tensor

        def forward(self, x: torch.Tensor, *args, **kwargs) -> torch.Tensor:
            """
            Perform the forward pass using the HOFT adapter.

            Args:
                x (torch.Tensor): Input tensor.

            Returns:
                torch.Tensor: Output tensor after applying the HOFT adaptation.

            Note:
                This method implements the HOFT/RHOFT-specific forward pass.
            """

            if self.disable_adapters:
                if self.merged:
                    self.unmerge()
                result = self.base_layer(x, *args, **kwargs)
            elif self.merged:
                result = self.base_layer(x, *args, **kwargs)
            else:
                result = self.base_layer(x, *args, **kwargs)
                for active_adapter in self.active_adapters:
                    if active_adapter not in self.hoft_U.keys():
                        continue

                    U = self.hoft_U[active_adapter]
                    V = self.hoft_V[active_adapter]

                    x = x.to(U.dtype)

                    dropout = self.hoft_dropout[active_adapter]        
                    result_V = torch.matmul(dropout(x), V)

                    WV = torch.matmul(V.T, V)
                    DV = WV.diagonal().reciprocal().mul(2)
                    SV = DV * WV.triu(1) * DV
                    SV.diagonal().sub_(DV)

                    preresult = x + torch.matmul(result_V, torch.matmul(SV, V.T))
                    result = self.base_layer(preresult, *args, **kwargs)
                    
                    if self.base_layer.bias is not None:
                        result = result - self.base_layer.bias

                    if self.use_shoft:
                        magnitude = self.hoft_magnitude_vector[active_adapter]
                        result = magnitude * result 
        
                    result_U = torch.matmul(result, U)

                    WU = torch.matmul(U.T, U)
                    DU = WU.diagonal().reciprocal().mul(2)
                    SU = DU * WU.triu(1) * DU
                    SU.diagonal().sub_(DU)
                    
                    result = result + torch.matmul(result_U, torch.matmul(SU, U.T))

                    if self.base_layer.bias is not None:
                        result = result + self.base_layer.bias

            # Ensure the output tensor has the same dtype as the input tensor
            return result.to(x.dtype)

        def __repr__(self) -> str:
            rep = super().__repr__()
            return "hoft." + rep


if is_bnb_4bit_available():

    class Linear4bit(torch.nn.Module, HOFTLayer):
        def __init__(
            self,
            base_layer: torch.nn.Module,
            adapter_name: str,
            r: int = 0,
            hoft_dropout: float = 0.0,
            fan_in_fan_out: bool = False,
            init_weights: str = 'normal',
            use_shoft: bool = False,
            **kwargs,
        ) -> None:
            super().__init__()
            HOFTLayer.__init__(self, base_layer, **kwargs)
            self.fan_in_fan_out = fan_in_fan_out

            self.use_shoft = use_shoft

            self._active_adapter = adapter_name
            
            self.update_layer(adapter_name, r, hoft_dropout, init_weights, use_shoft)
            

        def merge(self, safe_merge: bool = False, adapter_names: Optional[list[str]] = None) -> None:
            if self.merged:
                warnings.warn(
                    f"Already following adapters were merged {','.join(self.merged_adapters)}. "
                    f"You are now additionally merging {','.join(self.active_adapters)}."
                )

            adapter_names = check_adapters_to_merge(self, adapter_names)
            if not adapter_names:
                return

            for active_adapter in adapter_names:
                if active_adapter not in self.hoft_U.keys():
                    continue

                warnings.warn(
                    "Merge hoft module to 4-bit linear may get different generations due to rounding errors."
                )

                weight = self.get_base_layer().weight
                kwargs = weight.__dict__
                w_data = bnb.functional.dequantize_4bit(weight.data, weight.quant_state)
                hoft_data = self.get_delta_weight(active_adapter, w_data)
                w_data = w_data + hoft_data

                if safe_merge and not torch.isfinite(w_data).all():
                    raise ValueError(
                        f"NaNs detected in the merged weights. The adapter {active_adapter} seems to be broken"
                    )

                self.get_base_layer().weight = bnb.nn.Params4bit(w_data.to("cpu"), requires_grad=False, **kwargs).to(
                    weight.device
                )
                self.merged_adapters.append(active_adapter)

        def unmerge(self) -> None:
            if not self.merged:
                warnings.warn("Already unmerged. Nothing to do")
                return

            while len(self.merged_adapters) > 0:
                active_adapter = self.merged_adapters.pop()
                if active_adapter not in self.hoft_U.keys():
                    continue
                warnings.warn(
                    "Unmerge hoft module to 4-bit linear may get different generations due to rounding errors."
                )

                weight = self.get_base_layer().weight
                kwargs = weight.__dict__
                w_data = bnb.functional.dequantize_4bit(weight.data, weight.quant_state)
                hoft_data = self.get_delta_weight(active_adapter, w_data)
                w_data = w_data - hoft_data

                self.get_base_layer().weight = bnb.nn.Params4bit(w_data.to("cpu"), requires_grad=False, **kwargs).to(
                    weight.device
                )

        def get_delta_weight(self, adapter, base_weight) -> torch.Tensor:
            # Retrieve shared projection matrices
            U = self.hoft_U[adapter]
            V = self.hoft_V[adapter]

            device = U.device
            dtype = U.dtype

            # In case users wants to merge the adapter weights that are in
            # (b)float16 while being on CPU, we need to cast the weights to float32, perform the merge and then cast back to
            # (b)float16 because some CPUs have slow bf16/fp16 matmuls.
            cast_to_fp32 = device.type == "cpu" and (dtype == torch.float16 or dtype == torch.bfloat16)


            if cast_to_fp32:
                U = U.float()
                V = V.float()

            WV = torch.matmul(V.T, V)
            DV = WV.diagonal().reciprocal().mul(2)
            SV = DV * WV.triu(1) * DV
            SV.diagonal().sub_(DV)
            preresult = torch.eye(self.in_features) + torch.matmul(V, torch.matmul(SV, V.T))
            result = torch.matmul(preresult, base_weight.T)
            
            if self.use_shoft:
                magnitude = self.hoft_magnitude_vector[adapter]
                if cast_to_fp32:
                    magnitude = magnitude.float()
                result = magnitude * result

            result_U = torch.matmul(result, U)
            WU = torch.matmul(U.T, U)
            DU = WU.diagonal().reciprocal().mul(2)
            SU = DU * WU.triu(1) * DU
            SU.diagonal().sub_(DU)
            
            result = result + torch.matmul(result_U, torch.matmul(SU, U.T))
            result = result - base_weight

            if cast_to_fp32:
                output_tensor = output_tensor.to(dtype=dtype)

                self.hoft_U[adapter].data = U.to(dtype=dtype)
                self.hoft_V[adapter].data = V.to(dtype=dtype)
                if self.use_shoft:
                    self.hoft_magnitude_vector[adapter].data = magnitude.to(dtype=dtype)

            return output_tensor

        def forward(self, x: torch.Tensor, *args, **kwargs) -> torch.Tensor:
            if self.disable_adapters:
                if self.merged:
                    self.unmerge()
                result = self.base_layer(x, *args, **kwargs)
            elif self.merged:
                result = self.base_layer(x, *args, **kwargs)
            else:
                expected_dtype = x.dtype
                for active_adapter in self.active_adapters:
                    if active_adapter not in self.hoft_U.keys():
                        continue

                    U = self.hoft_U[active_adapter].to(expected_dtype)
                    V = self.hoft_V[active_adapter].to(expected_dtype)
        

                    dropout = self.hoft_dropout[active_adapter]        
                    result_V = torch.matmul(dropout(x), V)

                    WV = torch.matmul(V.T, V)
                    DV = WV.diagonal().reciprocal().mul(2)
                    SV = DV * WV.triu(1) * DV
                    SV.diagonal().sub_(DV)

                    preresult = x + torch.matmul(result_V, torch.matmul(SV, V.T))
                    result = self.base_layer(preresult, *args, **kwargs)
                    
                    if self.base_layer.bias is not None:
                        result = result - self.base_layer.bias

                    if self.use_shoft:
                        magnitude = self.hoft_magnitude_vector[active_adapter].to(expected_dtype)
                        result = magnitude * result
                    
                    result_U = torch.matmul(result, U)

                    WU = torch.matmul(U.T, U)
                    DU = WU.diagonal().reciprocal().mul(2)
                    SU = DU * WU.triu(1) * DU
                    SU.diagonal().sub_(DU)
                    
                    result = result + torch.matmul(result_U, torch.matmul(SU, U.T))

                    if self.base_layer.bias is not None:
                        result = result + self.base_layer.bias

            # Ensure the output tensor has the same dtype as the input tensor
            return result.to(expected_dtype)

        def __repr__(self) -> str:
            rep = super().__repr__()
            return "hoft." + rep
