# main.py
import argparse
import asyncio
import importlib
import sys


async def run_task():
    """
    Parses command-line arguments to dynamically import and run the
    main async function from the specified script.
    """
    parser = argparse.ArgumentParser(
        description="Main entry point to run agentic workflow tasks."
    )
    parser.add_argument(
        "--task-type",
        "-t",
        help="The name of the task to run.",
        choices=["generalize", "solve"],
    )
    parser.add_argument(
        "--reviewer-type",
        "-e",
        help="The type of reviewer to use.",
        choices=["composite", "monolithic"],
    )
    args = parser.parse_args()

    script_prefix = "solver" if args.task_type == "solve" else "generalizer"
    module_name = f"scripts.{script_prefix}_{args.reviewer_type}"

    try:
        print(f"Attempting to run: {module_name}...")
        task_module = importlib.import_module(module_name)
    except ImportError:
        print(
            f"Error: Could not find the script for '{args.task_type}' with evaluator '{args.reviewer_type}'."
        )
        print(f"Looked for module: {module_name}")
        sys.exit(1)

    if hasattr(task_module, "run_async_workflow") and asyncio.iscoroutinefunction(
        task_module.run_async_workflow
    ):
        await task_module.run_async_workflow()
    else:
        print(
            f"Error: The module '{module_name}' does not have an async 'run_async_workflow' function."
        )
        sys.exit(1)


if __name__ == "__main__":
    asyncio.run(run_task())
