def runSimBatch(
    dataSplit="test",           # train | test
    numberSamples=None,         # number of samples to include in dataset
    sent_model_name="textattack/bert-base-uncased-yelp-polarity",
    model_name="meta-llama/Meta-Llama-3.1-8B-Instruct",
    layerIdx=2,                 # layer to index at
    w=0.35,                     # weight of index
    indexApplicationType="standard",
    base_dir="./outputs/",
    seed=0,
    max_mem=None,
    batch_size=32               # how many chats per forward-pass
):
    set_seed(seed)
    params = { ... }  # same as before
    outputFolderPath = create_param_folder(params, base_dir=base_dir)
    with open(os.path.join(outputFolderPath, "params.json"), "w") as jf:
        json.dump(params, jf, indent=4)
    outputFile = os.path.join(outputFolderPath, "data.csv")

    # 1. Load & possibly truncate dataset
    dataset = load_dataset("yelp_polarity", split=dataSplit).shuffle(seed=0)
    if numberSamples is not None:
        dataset = dataset.select(range(numberSamples))
    numberToLabel = {0: "Negative", 1: "Positive"}

    # 2. Sentiment classifier
    sc = SentimentClassification(sent_model_name=sent_model_name)

    # 3. Model & tokenizer
    tokenizer = AutoTokenizer.from_pretrained(model_name, use_fast=False)
    if max_mem is None:
        model = AutoModelForCausalLM.from_pretrained(model_name, device_map="auto")
    else:
        model = AutoModelForCausalLM.from_pretrained(
            model_name,
            torch_dtype=torch.bfloat16,
            device_map="sequential",
            max_memory=max_mem,
            low_cpu_mem_usage=True
        )
    model.eval()

    # 4. Chat pipeline with batching
    pipeline = transformers.pipeline(
        "chat",
        model=model,
        tokenizer=tokenizer,
        model_kwargs={"torch_dtype": torch.bfloat16},
    )
    pipeline.batch_size = batch_size

    # 5. Precompute index tokens
    negIndexToken = get_last_token_representation(model, tokenizer, "be extremely negative", layerIdx)
    posIndexToken = get_last_token_representation(model, tokenizer, "be extremely positive", layerIdx)

    append_list_to_csv(
        ["Orig Text","Rephrased Text","Ground Truth Label",
         "Orig Pred Label","Rephrased Pred Label",
         "Orig Pred Score","Rephrased Pred Score","Bleu"],
        outputFile
    )

    # 6. Batch-accumulate & generate
    sample_batch = []
    msgs_batch   = []
    for sample in tqdm(dataset, desc="Processing samples"):
        text, label = sample["text"].strip(), sample["label"]
        msgs = [
            {"role": "system", "content":
                "You are a helpful assistant. The user will provide you with a review, "
                "respond only with a rephrased review (with no additional commentary) "
                "while keeping the original meaning.\n\n"
            },
            {"role": "user", "content": text}
        ]
        sample_batch.append((text, label))
        msgs_batch.append(msgs)

        # once we've collected `batch_size` chats, fire off one batched call
        if len(sample_batch) == batch_size:
            # apply index injection for this batch (using first sample's label)
            idx_tok = negIndexToken if sample_batch[0][1] == 1 else posIndexToken
            orig_fwd = model.model.layers[layerIdx].forward
            if w > 0.:
                updateModelForwardPass(model, layerIdx, idx_tok, w, indexApplicationType)

            outputs = pipeline(
                msgs_batch,
                max_new_tokens=1024,
                batch_size=batch_size
            )
            model.model.layers[layerIdx].forward = orig_fwd

            # record each result
            for (txt, lbl), out in zip(sample_batch, outputs):
                rephrased = out["generated_text"][-1]["content"]
                bleu = compute_bleu4(txt, rephrased)
                oL, oS = sc.classify_sentiment(txt)
                rL, rS = sc.classify_sentiment(rephrased)
                append_list_to_csv([
                    txt, rephrased, numberToLabel[lbl],
                    oL, rL, oS, rS, bleu
                ], outputFile)

            sample_batch.clear()
            msgs_batch.clear()

    # 7. Flush any leftovers
    if sample_batch:
        idx_tok = negIndexToken if sample_batch[0][1] == 1 else posIndexToken
        orig_fwd = model.model.layers[layerIdx].forward
        if w > 0.:
            updateModelForwardPass(model, layerIdx, idx_tok, w, indexApplicationType)

        outputs = pipeline(
            msgs_batch,
            max_new_tokens=1024,
            batch_size=len(sample_batch)
        )
        model.model.layers[layerIdx].forward = orig_fwd

        for (txt, lbl), out in zip(sample_batch, outputs):
            rephrased = out["generated_text"][-1]["content"]
            bleu = compute_bleu4(txt, rephrased)
            oL, oS = sc.classify_sentiment(txt)
            rL, rS = sc.classify_sentiment(rephrased)
            append_list_to_csv([
                txt, rephrased, numberToLabel[lbl],
                oL, rL, oS, rS, bleu
            ], outputFile)

    # 8. Cleanup
    del pipeline
    del model
    del sc
    del tokenizer
    torch.cuda.empty_cache()
    gc.collect()
    torch.cuda.ipc_collect()
