from embedder import embedder
import torch.nn as nn
import torch.optim as optim
import torch.nn.functional as F
import numpy as np
from copy import deepcopy
import datetime
import utils
from tqdm import trange
import pickle
from torch_geometric.nn.models import LabelPropagation
from filling_strategies import filling
from utils import get_symmetrically_normalized_adjacency, add_labels
import torch

class LP_label_trick():
    def __init__(self, args):
        self.args = args
    
    def training(self):
        file = utils.set_filename(self.args)
        logger = utils.setup_logger('./', '-', file)

        seed_result = {}
        seed_result['acc'] = []
        seed_result['macro_F'] = []
        
        for seed in trange(0, 0+self.args.n_runs):
            print(f'============== seed:{seed} ==============')
            utils.seed_everything(seed)
            print('seed:', seed, file)
            self.args.seed = seed
            self = embedder(self.args, seed)
            # self.x = torch.nan_to_num(self.x, 0)

            filled_features_fp = filling(self.args.filling_method, self.edge_index, self.x, self.missing_feature_mask, self.args.num_iterations)
            self.x = torch.where(self.missing_feature_mask, self.x, filled_features_fp)

            edge_index, edge_weight = get_symmetrically_normalized_adjacency(self.edge_index, n_nodes=self.x.shape[0])
            self.adj = torch.sparse.FloatTensor(edge_index, values=edge_weight, size=(self.x.shape[0], self.x.shape[0])).to(edge_index.device)

            self.lp_coef = torch.zeros(self.x.shape[0], 1).to(self.x.device)
            lp_coef_train = torch.max(self.lp_mat[self.train_mask], 1)[0].unsqueeze(1).to(self.x.device)
            self.lp_coef[self.train_mask] = lp_coef_train

            model = modeler(self.args, self.adj, self.lp_coef).to(self.args.device)
            optimizer = optim.Adam(model.parameters(), lr=self.args.lr)

            # Main training
            acc_vals = []
            test_results = []
            best_metric = 0

            for epoch in range(0, self.args.epochs):
                model.train()
                optimizer.zero_grad()

                if 'OGBN' in self.args.dataset:
                    train_idx = self.train_mask
                    ogbn = True
                else:
                    train_idx = torch.arange(self.x.shape[0])[self.train_mask].to(self.train_mask.device)
                    ogbn = False

                if self.args.mask_rate == 1:
                    self.x_new = add_labels(self.x, self.labels, train_idx, self.args.n_class, ogbn=ogbn) # 1433 -> 1440 (cora 7 개 니까.)
                    self.train_mask = train_idx
                else:
                    mask = torch.rand(train_idx.shape) < self.args.mask_rate

                    train_labels_idx = train_idx[mask] # labels included for training
                    self.train_mask = train_idx[~mask] # labels for prediction

                    self.x_new = add_labels(self.x, self.labels, train_labels_idx, self.args.n_class, scale=1/self.args.mask_rate, ogbn=ogbn)
                
                loss = model(self.x_new, self.labels, self.train_mask, use_coef=self.args.use_coef)

                loss.backward()
                optimizer.step()

                #  Valid
                model.eval()
                # self.x_new = add_labels(self.x, self.labels, train_idx, self.args.n_class, ogbn=ogbn) # 1433 -> 1440 (cora 7 개 니까.)
                if self.args.mask_rate == 1.0:
                    output = model.linear(model.buff)
                else:
                    out = torch.cat([model.buff, model.propagate(self.x_new[:, -self.args.n_class:])], dim=-1)
                    output = model.linear(out)

                acc_val, macro_F_val = utils.performance(output[self.val_mask], self.labels[self.val_mask], pre='valid', evaluator=self.evaluator)
                acc_vals.append(acc_val)
                max_idx = acc_vals.index(max(acc_vals))

                if best_metric <= acc_val:
                    best_metric = acc_val
                    best_output = output[:]

                # Test
                acc_test, macro_F_test = utils.performance(output[self.test_mask], self.labels[self.test_mask], pre='test', evaluator=self.evaluator)

                test_results.append([acc_test, macro_F_test])
                best_test_result = test_results[max_idx]

                if epoch % self.args.print_result == 0:
                    model_name = self.args.embedder + '-' + self.args.filling_method + f'_mask_rate_{self.args.mask_rate}'
                    st = "[seed {}][{}-{}][{}][Epoch {}]".format(seed, self.args.dataset, self.args.missing_rate, model_name, epoch)
                    st += "[Val] ACC: {:.2f}, Macro-F1: {:.2f}|| ".format(acc_val, macro_F_val)
                    st += "[Test] ACC: {:.2f}, Macro-F1: {:.2f}\n".format(acc_test, macro_F_test)
                    st += "  [*Best Test Result*][Epoch {}] ACC: {:.2f}, Macro-F1: {:.2f}".format(max_idx, best_test_result[0], best_test_result[1])
                    print(st)
                      
                if (epoch - max_idx > self.args.patience) or (epoch+1 == self.args.epochs):
                    if epoch - max_idx > self.args.patience:
                        print("Early stop")
                    output = best_output
                    best_test_result[0], best_test_result[1] = utils.performance(output[self.test_mask], self.labels[self.test_mask], pre='test', evaluator=self.evaluator)
                    print("[Best Test Result] ACC: {:.2f}, Macro-F1: {:.2f}".format(best_test_result[0], best_test_result[1]))
                    torch.cuda.empty_cache()
                    break

            seed_result['acc'].append(float(best_test_result[0]))
            seed_result['macro_F'].append(float(best_test_result[1]))

        acc = seed_result['acc']
        f1 = seed_result['macro_F']

        print('[Averaged result] ACC: {:.2f}+{:.2f}, Macro-F: {:.2f}+{:.2f}'.format(np.mean(acc), np.std(acc), np.mean(f1), np.std(f1)))
        print('{:.2f}+{:.2f} {:.2f}+{:.2f}'.format(np.mean(acc), np.std(acc), np.mean(f1), np.std(f1)))

        logger.info('')
        logger.info(datetime.datetime.now())
        logger.info(file)
        logger.info(f'----------- missing rate: {self.args.missing_rate} -----------')
        logger.info('{:.2f}+{:.2f} {:.2f}+{:.2f}'.format(np.mean(acc), np.std(acc), np.mean(f1), np.std(f1)))
        logger.info('{:.2f}+{:.2f}'.format(np.mean(acc), np.std(acc)))
        logger.info('{:.2f}+{:.2f}'.format(np.mean(f1), np.std(f1)))
        logger.info(self.args)
        logger.info(f'=================================')


class modeler(nn.Module):
    def __init__(self, args, adj, lp_coef):
        super(modeler, self).__init__()
        self.args = args
        self.adj = adj
        self.lp_coef = lp_coef
        self.linear = nn.Linear(args.n_feat, args.n_class, bias=False)
        self.buff = None

    def forward(self, feat, labels, idx_train, use_coef=False):
        h0 = feat
        h = h0

        if self.buff is not None:
            h = self.buff
        else:
            h = self.propagate(h)
            if use_coef:
                h[:, -self.args.n_class:] = h[:, -self.args.n_class:] - self.lp_coef * h0[:, -self.args.n_class:] 

            self.buff = h

        # if self.args.mask_rate == 1.0:
        #     if self.buff is not None:
        #         h = self.buff
        #     else:
        #         h = self.propagate(h)
        #         # if lp_coef is not None:
        #         #     h[:, -self.args.n_class:] = h[:, -self.args.n_class:] - lp_coef * h0[:, -self.args.n_class:] 

        #         self.buff = h
        # else:
        #     if self.buff is not None:
        #         h = torch.cat([self.buff, self.propagate(h[:, -self.args.n_class:])], dim=-1)
        #     else:
        #         h = self.propagate(h)

        #     self.buff = h[:, :-self.args.n_class]

        output = self.linear(h)

        if 'OGBN' in self.args.dataset: # arxiv only or producs as well?
            labels = labels.squeeze(1)
        loss_nodeclassification = F.cross_entropy(output[idx_train], labels[idx_train])

        return loss_nodeclassification

    def propagate(self, h, num_iterations=7):
        res = (1-self.args.lp_alpha) * h
        # print('Start Propagation ...!')
        for i in range(num_iterations):
            h = torch.sparse.mm(self.adj, h)
            h.mul_(self.args.lp_alpha).add_(res)
            h.clamp_(0., 1.)
        
        return h
