import datetime
from embedder import embedder
import numpy as np
import torch
import torch.nn as nn
import torch.optim as optim
import torch.nn.functional as F
from dgl import function as fn
import utils
from tqdm import trange
from utils import add_labels

class TWIRLS():
    def __init__(self, args):
        self.args = args
    
    def training(self):
        file = utils.set_filename(self.args)
        logger = utils.setup_logger('./', '-', file)

        seed_result = {}
        seed_result['acc'] = []
        seed_result['macro_F'] = []

        for seed in trange(0, 0+self.args.n_runs):
            print(f'============== seed:{seed} ==============')
            utils.seed_everything(seed)
            print('seed:', seed, file)
            self.args.seed = seed
            self = embedder(self.args, seed)


            if self.args.label_trick:
                self.args.n_feat += self.args.n_class
                if 'OGBN' in self.args.dataset:
                    train_idx = self.train_mask
                    ogbn = True
                else:
                    train_idx = torch.arange(self.x.shape[0])[self.train_mask].to(self.train_mask.device)
                    ogbn = False

                self.x_new = add_labels(self.x, self.labels, train_idx, self.args.n_class, ogbn=ogbn) # 1433 -> 1440 (cora 7 개 니까.)
                self.train_mask = train_idx

                self.dgl_graph.ndata['feat'] = self.x_new
                self.dgl_graph.ndata['feature'] = self.x_new

            # Main training
            model = modeler(self.args, self.dgl_graph.number_of_nodes()).to(self.args.device)
            optimizer = optim.Adam(model.parameters(), lr=self.args.lr)

            acc_vals = []
            test_results = []
            best_metric = 0
            self.dgl_graph.ndata['feat'] = torch.nan_to_num(self.dgl_graph.ndata['feat'], 0)
            self.dgl_graph.ndata['feature'] = torch.nan_to_num(self.dgl_graph.ndata['feature'], 0)

            for epoch in range(0, self.args.epochs):
                model.train()
                optimizer.zero_grad()

                if self.args.label_trick:
                    self.x_new = add_labels(self.x, self.labels, train_idx, self.args.n_class, ogbn=ogbn) # 1433 -> 1440 (cora 7 개 니까.)

                    self.dgl_graph.ndata['feat'] = self.x_new
                    self.dgl_graph.ndata['feature'] = self.x_new

                output = model(self.dgl_graph)
                if 'OGBN' in self.args.dataset:
                    loss = F.cross_entropy(output[self.dgl_graph.ndata['train_mask']],
                                           self.dgl_graph.ndata['label'][self.dgl_graph.ndata['train_mask']].squeeze(1))
                else:
                    loss = F.cross_entropy(output[self.dgl_graph.ndata['train_mask']],
                                           self.dgl_graph.ndata['label'][self.dgl_graph.ndata['train_mask']])

                loss.backward()
                optimizer.step()

                # Valid
                model.eval()
                output = model(self.dgl_graph)

                if (self.args.label_trick) & (self.args.n_reuse > 0):
                    if 'OGBN' not in self.args.dataset:
                        idx_val = torch.arange(self.x.shape[0])[self.val_mask].to(self.x.device)
                        idx_test = torch.arange(self.x.shape[0])[self.test_mask].to(self.x.device)
                    else:
                        idx_val = self.val_mask
                        idx_test = self.test_mask
                    unlabel_idx = torch.cat([train_idx, idx_val, idx_test])
                    for _ in range(self.args.n_reuse):
                        output = output.detach()
                        self.dgl_graph.ndata['feat'][unlabel_idx, -self.args.n_class:] = F.softmax(output[unlabel_idx], dim=-1)
                        self.dgl_graph.ndata['feature'][unlabel_idx, -self.args.n_class:] = F.softmax(output[unlabel_idx], dim=-1)
                        output = model(self.dgl_graph)

                acc_val, macro_F_val = utils.performance(output[self.dgl_graph.ndata['val_mask']],
                                                         self.dgl_graph.ndata['label'][self.dgl_graph.ndata['val_mask']],
                                                         pre='valid',
                                                         evaluator=self.evaluator)

                acc_vals.append(acc_val)

                if best_metric <= acc_val:
                    best_metric = acc_val
                    max_idx = acc_vals.index(max(acc_vals))
                    best_output = output[:]

                # Test
                acc_test, macro_F_test = utils.performance(output[self.dgl_graph.ndata['test_mask']],
                                                           self.dgl_graph.ndata['label'][self.dgl_graph.ndata['test_mask']],
                                                           pre='test',
                                                           evaluator=self.evaluator)

                test_results.append([acc_test, macro_F_test])
                best_test_result = test_results[max_idx]

                if epoch % 100 == 0:
                    st = "[seed {}][{}-{}][TWIRLS][Epoch {}]".format(seed, self.args.dataset, self.args.missing_rate, epoch)
                    st += "[Val] ACC: {:.2f}, Macro-F1: {:.2f}|| ".format(acc_val, macro_F_val)
                    st += "[Test] ACC: {:.2f}, Macro-F1: {:.2f}\n".format(acc_test, macro_F_test)
                    st += "  [*Best Test Result*][Epoch {}] ACC: {:.2f}, Macro-F1: {:.2f}".format(max_idx, best_test_result[0], best_test_result[1])
                    print(st)
                      
                if (epoch - max_idx > self.args.patience) or (epoch+1 == self.args.epochs):
                    if epoch - max_idx > self.args.patience:
                        print("Early stop")
                    output = best_output
                    best_test_result[0], best_test_result[1] = utils.performance(
                        output[self.dgl_graph.ndata['test_mask']],
                        self.dgl_graph.ndata['label'][self.dgl_graph.ndata['test_mask']],
                        pre='test',
                        evaluator=self.evaluator)
                    print("[Best Test Result] ACC: {:.2f}, Macro-F1: {:.2f}".format(best_test_result[0], best_test_result[1]))
                    break

            seed_result['acc'].append(float(best_test_result[0]))
            seed_result['macro_F'].append(float(best_test_result[1]))

        acc = seed_result['acc']
        f1 = seed_result['macro_F']

        print('[Averaged result] ACC: {:.2f}+{:.2f}, Macro-F: {:.2f}+{:.2f}'.format(np.mean(acc), np.std(acc), np.mean(f1), np.std(f1)))
        print('{:.2f}+{:.2f} {:.2f}+{:.2f}'.format(np.mean(acc), np.std(acc), np.mean(f1), np.std(f1)))

        logger.info('')
        logger.info(datetime.datetime.now())
        logger.info(file)
        logger.info(f'----------- missing rate: {self.args.missing_rate} -----------')
        logger.info('{:.2f}+{:.2f} {:.2f}+{:.2f}'.format(np.mean(acc), np.std(acc), np.mean(f1), np.std(f1)))
        logger.info('{:.2f}+{:.2f}'.format(np.mean(acc), np.std(acc)))
        logger.info('{:.2f}+{:.2f}'.format(np.mean(f1), np.std(f1)))
        logger.info(self.args)
        logger.info(f'=================================')
        
        # print(self.args)

class modeler(nn.Module):
    def __init__(self, args, n_nodes):
        super().__init__()
        self.input_d        = args.n_feat
        self.output_d       = args.n_class
        self.hidden_d       = args.n_hid
        self.prop_step      = args.prop_step
        self.num_mlp_before = args.mlp_bef
        self.num_mlp_after  = args.mlp_aft
        self.norm           = args.norm
        self.precond        = not args.no_precond
        self.attention      = args.attention
        self.alp            = args.alp
        self.lam            = args.lam
        self.tau            = args.tau
        self.T              = args.T
        self.attn_p         = args.attn_p
        self.use_eta        = args.use_eta
        self.init_att       = args.attn_bef
        self.dropout        = args.dropout
        self.attn_dropout   = args.attn_dropout
        self.inp_dropout    = args.inp_dropout
        self.learn_emb      = (n_nodes, args.learn_emb)

        # ----- initialization of some variables -----
        # where to put attention
        self.attn_aft = args.prop_step // 2 if args.attention else -1 

        # whether to learn a embedding for each node. used in amazon co-purchase dataset
        if self.learn_emb[1] > 0:
            self.node_emb = nn.Parameter(torch.randn(args.learn_emb[0] , args.learn_emb[1]))
            self.input_d  = args.learn_emb[1]
            nn.init.normal_(self.node_emb , 0 , 1e-3)
            
        # whether we can cache unfolding result
        self.cacheable = (not self.attention) and self.num_mlp_before == 0 and self.inp_dropout <= 0 and self.learn_emb[1] <= 0
        if self.cacheable:
            self.cached_unfolding = None


        # if only one layer, then no hidden size
        self.size_bef_unf = self.hidden_d
        self.size_aft_unf = self.hidden_d
        if self.num_mlp_before == 0:
            self.size_aft_unf = self.input_d  # as the input  of mlp_aft
        if self.num_mlp_after == 0:
            self.size_bef_unf = self.output_d # as the output of mlp_bef


        # ----- computational modules -----
        self.mlp_bef = MLP(self.input_d , self.hidden_d , self.size_bef_unf , self.num_mlp_before , 
                self.dropout , self.norm , init_activate = False)

        self.unfolding = UnfoldindAndAttention(self.hidden_d, self.alp, self.lam, self.prop_step, self.attn_aft, 
                self.tau, self.T, self.attn_p, self.use_eta, self.init_att, self.attn_dropout, self.precond)

        # if there are really transformations before unfolding, then do init_activate in mlp_aft
        self.mlp_aft = MLP(self.size_aft_unf , self.hidden_d , self.output_d , self.num_mlp_after  , 
            self.dropout , self.norm , 
            init_activate = (self.num_mlp_before > 0) and (self.num_mlp_after > 0) 
        )

    def forward(self , g):

         # use trained node embedding
        if self.learn_emb[1] > 0:
            x = self.node_emb
        else:
            x = g.ndata["feature"]

        if self.cacheable: 
            # to cache unfolding result becase there is no paramaters before it
            if self.cached_unfolding is None:
                self.cached_unfolding = self.unfolding(g , x)

            x = self.cached_unfolding
        else:
            if self.inp_dropout > 0:
                x = F.dropout(x, self.inp_dropout, training = self.training)
            x = self.mlp_bef(x)
            x = self.unfolding(g , x)

        x = self.mlp_aft(x)

        return x

class MLP(nn.Module):
    def __init__(self, input_d, hidden_d, output_d, num_layers, dropout, norm, init_activate) :
        super().__init__()

        self.init_activate  = init_activate
        self.norm           = norm
        self.dropout        = dropout

        self.layers = nn.ModuleList([])

        if num_layers == 1:
            self.layers.append(nn.Linear(input_d, output_d))
        elif num_layers > 1:
            self.layers.append(nn.Linear(input_d, hidden_d))
            for k in range(num_layers - 2):
                self.layers.append(nn.Linear(hidden_d, hidden_d))
            self.layers.append(nn.Linear(hidden_d, output_d))

        self.norm_cnt = num_layers-1+int(init_activate) # how many norm layers we have
        if norm == "batch":
            self.norms = nn.ModuleList([nn.BatchNorm1d(hidden_d) for _ in range(self.norm_cnt)])
        elif norm == "layer":
            self.norms = nn.ModuleList([nn.LayerNorm  (hidden_d) for _ in range(self.norm_cnt)])


        self.reset_params()

    def reset_params(self):
        for layer in self.layers:
            nn.init.xavier_normal_(layer.weight.data)
            nn.init.constant_     (layer.bias.data, 0)

    def activate(self, x):
        if self.norm != "none":
            x = self.norms[self.cur_norm_idx](x) # use the last norm layer
            self.cur_norm_idx += 1
        x = F.relu(x)
        x = F.dropout(x , self.dropout , training = self.training)
        return x 


    def forward(self, x):
        self.cur_norm_idx = 0

        if self.init_activate:
            x = self.activate(x)

        for i , layer in enumerate( self.layers ):
            x = layer(x)
            if i != len(self.layers) - 1: # do not activate in the last layer
                x = self.activate(x)

        return x
    
class UnfoldindAndAttention(nn.Module):
    def __init__(self, d, alp, lam, prop_step, attn_aft, tau, T, attn_p, use_eta, init_att , attn_dropout, precond):

        super().__init__()

        self.d      = d
        self.alp    = alp if alp > 0 else 1 / (lam + 1) # automatic set alpha
        self.lam    = lam
        self.tau    = tau
        self.attn_p    = attn_p
        self.prop_step = prop_step
        self.attn_aft  = attn_aft
        self.use_eta   = use_eta
        self.init_att  = init_att

        prop_method      = Propagate if precond else PropagateNoPrecond
        self.prop_layers = nn.ModuleList([prop_method() for _ in range(prop_step)])

        self.init_attn   = Attention(tau, T, attn_p, attn_dropout) if self.init_att      else None
        self.attn_layer  = Attention(tau, T, attn_p, attn_dropout) if self.attn_aft >= 0 else None
        self.etas        = nn.Parameter(torch.ones(d)) if self.use_eta else None

    def forward(self , g , X):
        
        Y = X


        g.edata["w"]    = torch.ones(g.number_of_edges(), 1, device = g.device)
        g.ndata["deg"]  = g.in_degrees().float()

        if self.init_att:
            g = self.init_attn(g, Y, self.etas)

        for k, layer in enumerate(self.prop_layers):

            # do unfolding
            Y = layer(g, Y, X, self.alp, self.lam)

            # do attention at certain layer
            if k == self.attn_aft - 1:
                g = self.attn_layer(g, Y, self.etas)

        return Y

class Propagate(nn.Module):
    def __init__(self):
        super().__init__()

    def prop(self, graph, Y, lam):

        Y = D_power_bias_X(graph, Y, -0.5, lam, 1 - lam)
        Y = AX(graph, Y)
        Y = D_power_bias_X(graph, Y, -0.5, lam, 1 - lam)

        return Y

    def forward(self, graph, Y, X, alp, lam):
        return (1 - alp) * Y + alp * lam * self.prop(graph, Y, lam) + alp * D_power_bias_X(graph, X, -1, lam, 1 - lam)


class PropagateNoPrecond(nn.Module):
    def __init__(self):
        super().__init__()

    def forward(self, graph, Y, X, alp, lam):

        return (1 - alp * lam - alp) * Y + alp * lam * normalized_AX(graph, Y) + alp * X

class Attention(nn.Module):
    def __init__(self, tau, T, attn_p, attn_dropout = 0.0):
        super().__init__()

        self.tau = tau
        self.T = T
        self.attn_p = attn_p
        self.attn_dropout = attn_dropout

    def reweighting(self, graph, etas):

        w = graph.edata["w"]

        # It is not activation here but to ensure w > 0.
        # w can be < 0 here because of some precision issue in dgl, which causes NaN afterwards.
        w = F.relu(w) + 1e-7

        w = torch.pow(w, 1 - 0.5 * self.attn_p)

        w[(w < self.tau)] = self.tau
        if self.T > 0:
            w[(w > self.T  )] = float("inf")

        w = 1 / w

        if not (w == w).all():
            raise "nan occured!"

        graph.edata["w"] = w + 1e-9 # avoid 0 degree

    def forward(self, graph, Y, etas = None):

        if etas is not None:
            Y = Y * etas.view(-1)

        # computing edge distance
        graph.srcdata["h"] = Y
        graph.srcdata["h_norm"] = (Y ** 2).sum(-1)
        graph.apply_edges(fn.u_dot_v("h", "h", "dot_"))
        graph.apply_edges(fn.u_add_v("h_norm", "h_norm", "norm_"))
        graph.edata["dot_"]  = graph.edata["dot_"].view(-1)
        graph.edata["norm_"] = graph.edata["norm_"].view(-1)
        graph.edata["w"]     = graph.edata["norm_"] - 2 * graph.edata["dot_"]
        
        # apply edge distance to get edge weight
        self.reweighting(graph, etas)

        # update node degrees
        graph.update_all(fn.copy_e("w", "m"), fn.sum("m", "deg"))
        graph.ndata["deg"] = graph.ndata["deg"].view(-1)

        # attention dropout. the implementation can ensure the degrees do not change in expectation.
        # FIXME: consider if there is a better way
        if self.attn_dropout > 0:
            graph.edata["w"] = F.dropout(graph.edata["w"], self.attn_dropout, training = self.training)

        return graph

def normalized_AX(graph, X):
    """Y = D^{-1/2}AD^{-1/2}X"""

    Y = D_power_X(graph, X, -0.5)  # Y = D^{-1/2}X
    Y = AX(graph, Y)  # Y = AD^{-1/2}X
    Y = D_power_X(graph, Y, -0.5)  # Y = D^{-1/2}AD^{-1/2}X

    return Y


def AX(graph, X):
    """Y = AX"""

    graph.srcdata["h"] = X
    graph.update_all(
        fn.u_mul_e("h", "w", "m"), fn.sum("m", "h"),
    )
    Y = graph.dstdata["h"]

    return Y


def D_power_X(graph, X, power):
    """Y = D^{power}X"""

    degs = graph.ndata["deg"]
    norm = torch.pow(degs, power)
    Y = X * norm.view(X.size(0), 1)
    return Y

def D_power_bias_X(graph, X, power, coeff, bias):
    """Y = (coeff*D + bias*I)^{power} X"""
    degs = graph.ndata["deg"]
    degs = coeff * degs + bias
    norm = torch.pow(degs, power)
    Y = X * norm.view(X.size(0), 1)
    return Y