from embedder import embedder
import torch.nn as nn
import torch.optim as optim
import torch.nn.functional as F
import numpy as np
from copy import deepcopy
import torch
import datetime
import utils
from tqdm import trange
from layers import GCN, GAT, SAGE, SGC
from utils import add_labels
from torch_geometric.data import NeighborSampler

class gnn():
    def __init__(self, args):
        self.args = args
    
    def training(self):
        file = utils.set_filename(self.args)
        logger = utils.setup_logger('./', '-', file)

        seed_result = {}
        seed_result['acc'] = []
        seed_result['macro_F'] = []
        
        for seed in trange(0, 0+self.args.n_runs):
            print(f'============== seed:{seed} ==============')
            utils.seed_everything(seed)
            print('seed:', seed, file)
            self.args.seed = seed
            self = embedder(self.args, seed)
            
            # Main training
            model = modeler(self.args).to(self.args.device)
            optimizer = optim.Adam(model.parameters(), lr=self.args.lr)

            acc_vals = []
            test_results = []
            best_metric = 0

            for epoch in range(0, self.args.epochs):
                model.train()
                if self.args.label_trick:
                    if 'OGBN' in self.args.dataset:
                        train_idx = self.train_mask
                        ogbn = True
                    else:
                        train_idx = torch.arange(self.x.shape[0])[self.train_mask].to(self.train_mask.device)
                        ogbn = False

                    if self.args.mask_rate == 1:
                        self.x_new = add_labels(self.x, self.labels, train_idx, self.args.n_class, ogbn=ogbn) # 1433 -> 1440 (cora 7 개 니까.)
                        self.train_mask = train_idx
                    else:
                        mask = torch.rand(train_idx.shape) < self.args.mask_rate

                        train_labels_idx = train_idx[mask] # labels included for training
                        self.train_mask = train_idx[~mask] # labels for prediction

                        self.x_new = add_labels(self.x, self.labels, train_labels_idx, self.args.n_class, scale=1/self.args.mask_rate, ogbn=ogbn)
                    
                        self.train_loader = (
                            NeighborSampler(
                                self.edge_index,
                                node_idx=self.train_mask,
                                sizes=[15, 10, 5][:self.args.n_layer],
                                batch_size=self.args.batch_size,
                                shuffle=True,
                                num_workers=12,
                            )
                            if self.args.graph_sampling
                            else None
                        )

                    x = self.x_new
                else:
                    x = self.x
                
                # Prepare loader for Large Datasets
                if self.train_loader:
                    total_loss = 0
                    idx = 0
                    for batch_size, n_id, adjs in self.train_loader:
                        optimizer.zero_grad()
                        idx += 1
                        # `adjs` holds a list of `(edge_index, e_id, size)` tuples.
                        adjs = [adj.to(self.args.device) for adj in adjs]
                        x_batch = x[n_id]

                        y_pred = model.classifier(x_batch, adjs, sample=True)
                        y_true = self.labels[n_id[:batch_size]].squeeze()
                        loss = F.cross_entropy(y_pred, y_true)

                        loss.backward()
                        optimizer.step()
                        total_loss += loss.item()

                        # logger.debug(f"Batch loss: {loss.item():.2f}")
                else:
                    optimizer.zero_grad()
                    loss = model(x, self.edge_index, self.edge_weight, self.labels, self.train_mask, self.val_mask, self.test_mask)

                    loss.backward()
                    optimizer.step()

                # print('Train loss:', loss.item())
                # Valid
                model.eval()
                if self.args.label_trick:
                    x = add_labels(self.x, self.labels, train_idx, self.args.n_class, ogbn=ogbn) # 1433 -> 1440 (cora 7 개 니까.)

                if self.inference_loader:
                    total_edges = 0
                    x_tmp = deepcopy(x.cpu())
                    for i in range(self.args.num_layers):
                        xs = []
                        for batch_size, n_id, adj in self.inference_loader:
                            edge_index, _, size = adj.to(self.args.device)
                            total_edges += edge_index.size(1)
                            x = x_tmp[n_id].to(self.args.device)
                            x_target = x[: size[1]]
                            x = model.classifier.conv1[i]((x, x_target), edge_index)
                            # x = model.classifier((x, x_target), edge_index, layer_idx=i)
                            x = F.relu(x)
                            if i == self.args.num_layers - 1:
                                x = model.classifier.classifier(x)

                            xs.append(x.detach().cpu())
                        
                        x_tmp = torch.cat(xs, dim=0)
                    
                    output = x_tmp

                else:                    
                    output = model.classifier(x, self.edge_index, self.edge_weight)

                    if (self.args.label_trick) & (self.args.n_reuse > 0):
                        if 'OGBN' not in self.args.dataset:
                            idx_val = torch.arange(x.shape[0])[self.val_mask].to(x.device)
                            idx_test = torch.arange(x.shape[0])[self.test_mask].to(x.device)
                        else:
                            idx_val = self.val_mask
                            idx_test = self.test_mask
                        unlabel_idx = torch.cat([train_idx, idx_val, idx_test])
                        for _ in range(self.args.n_reuse):
                            output = output.detach()
                            x[unlabel_idx, -self.args.n_class:] = F.softmax(output[unlabel_idx], dim=-1)
                            output = model.classifier(x, self.edge_index)

                acc_val, macro_F_val = utils.performance(output[self.val_mask], self.labels[self.val_mask], pre='valid', evaluator=self.evaluator)

                acc_vals.append(acc_val)

                if best_metric <= acc_val:
                    best_metric = acc_val
                    max_idx = acc_vals.index(max(acc_vals))
                    best_output = output[:]

                # Test
                acc_test, macro_F_test = utils.performance(output[self.test_mask], self.labels[self.test_mask], pre='test', evaluator=self.evaluator)

                test_results.append([acc_test, macro_F_test])
                best_test_result = test_results[max_idx]

                if epoch % self.args.print_result == 0:
                    __model_name = self.args.gnn + '-' + self.args.filling_method
                    _model_name = __model_name + f'_label_trick_mask_rate_{self.args.mask_rate}' if self.args.label_trick else __model_name
                    model_name = _model_name + f'_n_reuse_{self.args.n_reuse}' if self.args.n_reuse > 0 else _model_name
                    st = "[seed {}][{}-{}][{}][Epoch {}]".format(seed, self.args.dataset, self.args.missing_rate, model_name, epoch)
                    st += "[Val] ACC: {:.2f}, Macro-F1: {:.2f}|| ".format(acc_val, macro_F_val)
                    st += "[Test] ACC: {:.2f}, Macro-F1: {:.2f}\n".format(acc_test, macro_F_test)
                    st += "  [*Best Test Result*][Epoch {}] ACC: {:.2f}, Macro-F1: {:.2f}".format(max_idx, best_test_result[0], best_test_result[1])
                    print(st)
                      
                if (epoch - max_idx > self.args.patience) or (epoch+1 == self.args.epochs):
                    if epoch - max_idx > self.args.patience:
                        print("Early stop")
                    output = best_output
                    best_test_result[0], best_test_result[1] = utils.performance(output[self.test_mask], self.labels[self.test_mask], pre='test', evaluator=self.evaluator)
                    print("[Best Test Result] ACC: {:.2f}, Macro-F1: {:.2f}".format(best_test_result[0], best_test_result[1]))
                    # torch.cuda.empty_cache()
                    break

            seed_result['acc'].append(float(best_test_result[0]))
            seed_result['macro_F'].append(float(best_test_result[1]))

        acc = seed_result['acc']
        f1 = seed_result['macro_F']

        print('[Averaged result] ACC: {:.2f}+{:.2f}, Macro-F: {:.2f}+{:.2f}'.format(np.mean(acc), np.std(acc), np.mean(f1), np.std(f1)))
        print('{:.2f}+{:.2f} {:.2f}+{:.2f}'.format(np.mean(acc), np.std(acc), np.mean(f1), np.std(f1)))

        logger.info('')
        logger.info(datetime.datetime.now())
        logger.info(file)
        logger.info(f'----------- missing rate: {self.args.missing_rate} -----------')
        logger.info('{:.2f}+{:.2f} {:.2f}+{:.2f}'.format(np.mean(acc), np.std(acc), np.mean(f1), np.std(f1)))
        logger.info('{:.2f}+{:.2f}'.format(np.mean(acc), np.std(acc)))
        logger.info('{:.2f}+{:.2f}'.format(np.mean(f1), np.std(f1)))
        logger.info(self.args)
        logger.info(f'=================================')
        
        # print(self.args)


class modeler(nn.Module):
    def __init__(self, args):
        super(modeler, self).__init__()
        self.args = args

        ## Model Selection ##
        if args.gnn == 'GCN':
            classifier = GCN(args.n_layer, args.n_feat, args.n_hid, args.n_class, normalize=True, is_add_self_loops=False)
        elif args.gnn == 'GAT':
            classifier = GAT(args.n_layer, args.n_feat, args.n_hid, args.n_class, args.n_head, is_add_self_loops=False)
        elif args.gnn == "SAGE":
            classifier = SAGE(args.n_layer, args.n_feat, args.n_hid, args.n_class)
        elif args.gnn == "SGC":
            classifier = SGC(args.n_feat, args.n_class, args.n_layer, is_add_self_loops=False)
        else:
            raise NotImplementedError("Not Implemented Architecture!")        
        self.classifier = classifier

    def forward(self, x, edge_index, edge_weight, labels, idx_train, idx_val=None, idx_test=None):
        output = self.classifier(x, edge_index, edge_weight)
        if (self.args.label_trick) & (self.args.n_reuse > 0):
            if 'OGBN' not in self.args.dataset:
                idx_val = torch.arange(x.shape[0])[idx_val].to(x.device)
                idx_test = torch.arange(x.shape[0])[idx_test].to(x.device)

            unlabel_idx = torch.cat([idx_train, idx_val, idx_test])
            for _ in range(self.args.n_reuse):
                output = output.detach()
                x[unlabel_idx, -self.args.n_class:] = F.softmax(output[unlabel_idx], dim=-1)
                output = self.classifier(x, edge_index, edge_weight)

        if 'OGBN' in self.args.dataset:
            labels = labels.squeeze(1)
        loss_nodeclassification = F.cross_entropy(output[idx_train], labels[idx_train])

        return loss_nodeclassification