import copy
from components.episode_buffer import EpisodeBatch
from modules.mixers.nmix import Mixer
from modules.mixers.vdn import VDNMixer
from modules.mixers.qatten import QattenMixer
from envs.matrix_game import print_matrix_status
from utils.rl_utils import build_td_lambda_targets, build_q_lambda_targets
import torch as th
from torch.optim import RMSprop, Adam
import numpy as np
from utils.th_utils import get_parameters_num
from components.reward_model import reward_model,team_reward_model

class NQLearner:
    def __init__(self, mac, scheme, logger, args):
        self.args = args
        self.mac = mac
        self.logger = logger
        
        self.last_target_update_episode = 0
        self.device = th.device('cuda' if args.use_cuda  else 'cpu')
        self.params = list(mac.parameters())

        if args.mixer == "qatten":
            self.mixer = QattenMixer(args)
        elif args.mixer == "vdn":
            self.mixer = VDNMixer()
        elif args.mixer == "qmix":
            self.mixer = Mixer(args)
        else:
            raise "mixer error"
        self.target_mixer = copy.deepcopy(self.mixer)
        self.params += list(self.mixer.parameters())
        self.params_ind = list(mac.parameters())

        print('Mixer Size: ')
        print(get_parameters_num(self.mixer.parameters()))

        if self.args.optimizer == 'adam':
            self.optimiser = Adam(params=self.params,  lr=args.lr, weight_decay=getattr(args, "weight_decay", 0))
            self.optimiser_ind = Adam(params=self.params_ind,  lr=args.lr, weight_decay=getattr(args, "weight_decay", 0))
        else:
            self.optimiser = RMSprop(params=self.params, lr=args.lr, alpha=args.optim_alpha, eps=args.optim_eps)
            self.optimiser_ind = RMSprop(params=self.params_ind, lr=args.lr, alpha=args.optim_alpha, eps=args.optim_eps)

        # a little wasteful to deepcopy (e.g. duplicates action selector), but should work for any MAC
        self.target_mac = copy.deepcopy(mac)
        self.log_stats_t = -self.args.learner_log_interval - 1
        self.train_t = 0

        # priority replay
        self.use_per = getattr(self.args, 'use_per', False)
        self.return_priority = getattr(self.args, "return_priority", False)
        if self.use_per:
            self.priority_max = float('-inf')
            self.priority_min = float('inf')
            
    def _build_inputs(self, batch, t,n_agents):
        # Assumes homogenous agents with flat observations.
        # Other MACs might want to e.g. delegate building inputs to each agent
        bs = batch.batch_size
        inputs = []
        inputs.append(batch["obs"][:, t])  # b1av
        if self.args.obs_last_action:
            if t == 0:
                inputs.append(th.zeros_like(batch["actions_onehot"][:, t]))
            else:
                inputs.append(batch["actions_onehot"][:, t-1])
        if self.args.obs_agent_id:
            inputs.append(th.eye(n_agents, device=batch.device).unsqueeze(0).expand(bs, -1, -1))

        inputs = th.cat([x.reshape(bs, n_agents, -1) for x in inputs], dim=-1)
        return inputs
    
    def update_reward_in_replay_memory(self,reward_model_,replay_data:EpisodeBatch) :   
        max_ep_t = replay_data.max_t_filled()
        actions = replay_data["actions"].to(self.device) # [88,61,3,1]
        state = replay_data["state"].to(self.device) # [88,61,48]
        state_next = replay_data["state"].to(self.device) 
        obs = replay_data["obs"].to(self.device) # [88,61,3,30]
        start_point = obs.shape[3]
        reward = replay_data["reward"].to(self.device)
        ind_rewards = replay_data["ind_reward"].to(self.device) # [88,61,3,1]
        batch_size = obs.shape[0]
        n_agents = obs.shape[2]
        for step in range(max_ep_t-1) :       
            reward[:,step] = 0
            s = state[:,step,:].to(self.device)
            s_next = state[:,step+1,:].to(self.device)
            obs = self._build_inputs(replay_data,step,n_agents)
            end_point = obs.shape[2]-n_agents
            for n_agent in range(n_agents) :
                ind_rewards[:,step,n_agent,0] = 0
                o = obs[:,n_agent,:].to(self.device)
                a = obs[:,n_agent,start_point:end_point].to(self.device)
                with th.no_grad() :
                    ind_reward = 0
                    for reward_num in range(self.args.n_reward_functions) :                       
                        ind_reward = ind_reward + reward_model_[reward_num](s,s_next,o,a)
                    ind_reward = ind_reward / self.args.n_reward_functions
                    reward[:,step] += ind_reward
                    ind_rewards[:,step,n_agent] = ind_reward
        return reward, ind_rewards

        
    def train(self, reward_model_,batch: EpisodeBatch, t_env: int, episode_num: int, per_weight=None):
        # Get the relevant quantities
        if self.args.use_ori_reward or self.args.use_extrinsic_reward :
            rewards = batch["reward"][:, :-1]
            ind_rewards = batch["ind_reward"][:,:-1]
        else :
            rewards,ind_rewards = self.update_reward_in_replay_memory(reward_model_,batch)
            rewards = rewards[:,:-1]
            ind_rewards = ind_rewards[:,:-1]
            
        actions = batch["actions"][:, :-1]
        actions_ind = actions.clone()
        terminated = batch["terminated"][:, :-1].float()
        ind_terminated = batch["ind_terminated"][:,:-1].float()

        mask = batch["filled"][:, :-1].float()
        mask_ind = th.zeros(ind_terminated.shape).to(self.device)
        for agent in range(self.args.n_agents) :
            mask_ind[:,:,agent,:] = mask.clone()
        
        mask[:, 1:] = mask[:, 1:] * (1 - terminated[:, :-1])
        mask_ind[:,1:] = mask_ind[:,1:] * (1-ind_terminated[:,:-1])

        avail_actions = batch["avail_actions"]
        #print(batch["filled"][:,:-1].shape) torch.Size([128, 60, 1])
        #print(terminated[:,:-1].shape) torch.Size([128, 59, 1])
        #print(ind_terminated.shape) torch.Size([128, 60, 3, 1])
        #print(mask_ind.shape)
        
        ############################################################
        ##### Individual Q Learning ################################
        # Calculate estimated Q-Values
        if self.args.ind_q_learning :
            self.mac.agent.train()
            mac_out = []
            self.mac.init_hidden(batch.batch_size)
            for t in range(batch.max_seq_length):
                agent_outs = self.mac.forward(batch, t=t)
                mac_out.append(agent_outs)
            mac_out = th.stack(mac_out, dim=1)  # Concat over time

            # Pick the Q-Values for the actions taken by each agent
            chosen_action_qvals = th.gather(mac_out[:, :-1], dim=3, index=actions).squeeze(3)  # Remove the last dim

            # Calculate the Q-Values necessary for the target
            with th.no_grad():
                self.target_mac.agent.train()
                target_mac_out = []
                self.target_mac.init_hidden(batch.batch_size)
                for t in range(batch.max_seq_length):
                    target_agent_outs = self.target_mac.forward(batch, t=t)
                    target_mac_out.append(target_agent_outs)

                # We don't need the first timesteps Q-Value estimate for calculating targets
                target_mac_out = th.stack(target_mac_out, dim=1)  # Concat across time

                # Max over target Q-Values/ Double q learning
                mac_out_detach = mac_out.clone().detach()

                mac_out_detach[avail_actions == 0] = -9999999
                cur_max_actions = mac_out_detach.max(dim=3, keepdim=True)[1]

                target_max_qvals = th.gather(target_mac_out, 3, cur_max_actions).squeeze(3)[:,:-1]

            ### Individual Q Learning 
            with th.autograd.set_detect_anomaly(True) :
                #chosen_action_qvals_ind = chosen_action_qvals.clone()    
                target_ind = ind_rewards.squeeze() + self.args.gamma*target_max_qvals
                td_error_ind = chosen_action_qvals - target_ind.detach()
                td_error_ind2 = 0.5 * td_error_ind.pow(2)
                mask_td_error_ind = td_error_ind2 * mask_ind.squeeze()
                loss_ind = L_td_ind = mask_td_error_ind.sum() / mask_ind.sum()

                self.optimiser_ind.zero_grad()
                loss_ind.backward()
                grad_norm = th.nn.utils.clip_grad_norm_(self.params_ind, self.args.grad_norm_clip)
                self.optimiser_ind.step()

        '''
        print('traget_max_qvals_ind.shape : {}'.format(target_max_qvals_ind.shape))
        print('chosen_action_qvals_ind.shape : {}'.format(chosen_action_qvals_ind.shape))
        print('ind_rewards.shape : {}'.format(ind_rewards.shape))
        print('ind_rewards.squeeze().shape : {}'.format(ind_rewards.squeeze().shape))
        print('ind_rewards.shape : {}'.format(ind_rewards.shape))
        print('target_max_qvals_ind.shape : {}'.format(target_max_qvals_ind.shape))
        print('target_ind.shape : {}'.format(target_ind.shape))
        print('td_error_ind.shape : {}'.format(td_error_ind.shape))
        print('td_error_ind2.shape : {}'.format(td_error_ind2.shape))
        print('mask_ind.shape : {}'.format(mask_ind.shape))
        print('target_max_qvals_ind.shape : {}'.format(target_max_qvals_ind.shape))
        print('target_ind.shape : {}'.format(target_ind.shape))
        print('td_error_ind.shape : {}'.format(td_error_ind.shape))
        print('td_error_ind2.shape : {}'.format(td_error_ind2.shape))
        print('mask_td_error_ind : {}'.format(mask_td_error_ind.sum()))
        print('mask_ind.sum : {}'.format(mask_ind.sum()))
        print('mask_td_error_ind.shape : {}'.format(mask_td_error_ind.shape))
        '''
        
        ############################################################
        ##### TOTAL Q Learning ################################

        # Calculate estimated Q-Values
        self.mac.agent.train()
        mac_out = []
        self.mac.init_hidden(batch.batch_size)
        for t in range(batch.max_seq_length):
            agent_outs = self.mac.forward(batch, t=t)
            mac_out.append(agent_outs)
        mac_out = th.stack(mac_out, dim=1)  # Concat over time
        
        # Pick the Q-Values for the actions taken by each agent
        chosen_action_qvals = th.gather(mac_out[:, :-1], dim=3, index=actions).squeeze(3)  # Remove the last dim

        # Calculate the Q-Values necessary for the target
        with th.no_grad():
            self.target_mac.agent.train()
            target_mac_out = []
            self.target_mac.init_hidden(batch.batch_size)
            for t in range(batch.max_seq_length):
                target_agent_outs = self.target_mac.forward(batch, t=t)
                target_mac_out.append(target_agent_outs)

            # We don't need the first timesteps Q-Value estimate for calculating targets
            target_mac_out = th.stack(target_mac_out, dim=1)  # Concat across time

            # Max over target Q-Values/ Double q learning
            mac_out_detach = mac_out.clone().detach()
            mac_out_detach[avail_actions == 0] = -9999999
            cur_max_actions = mac_out_detach.max(dim=3, keepdim=True)[1]            
            target_max_qvals = th.gather(target_mac_out, 3, cur_max_actions).squeeze(3)
            
            # Calculate n-step Q-Learning targets
            target_max_qvals = self.target_mixer(target_max_qvals, batch["state"])        

            if getattr(self.args, 'q_lambda', False):
                qvals = th.gather(target_mac_out, 3, batch["actions"]).squeeze(3)
                qvals = self.target_mixer(qvals, batch["state"])
                #print('qvals : {}'.format(qvals.shape))
                targets = build_q_lambda_targets(rewards, terminated, mask, target_max_qvals, qvals,
                                    self.args.gamma, self.args.td_lambda)
                #print('targets : {}'.format(targets.shape))
            else:
                targets = build_td_lambda_targets(rewards, terminated, mask, target_max_qvals, 
                                                    self.args.n_agents, self.args.gamma, self.args.td_lambda)
                #print('targets : {}'.format(targets.shape))
        
        # Mixer        
        with th.autograd.set_detect_anomaly(True) :
            chosen_action_qvals = self.mixer(chosen_action_qvals, batch["state"][:, :-1])

            td_error = (chosen_action_qvals - targets.detach())
            td_error2 = 0.5 * td_error.pow(2)

            mask = mask.expand_as(td_error2)        
            masked_td_error = td_error2 * mask

            #mask.shape : torch.Size([128, 60, 1])
            #mac_out.shape : torch.Size([128, 61, 3, 9])
            #chosen_action_qvals : torch.Size([128, 60, 3])
            #rewards.shape : torch.Size([128, 60, 1])
            #terminated : torch.Size([128, 60, 1])
            #targets : torch.Size([128, 60, 1])
            #chosen_action_qvals2 : torch.Size([128, 60, 1]) : After #MIXER 
            #td_error : torch.Size([128, 60, 1])
            #target_max_qvals : torch.Size([128, 61, 3])
            #target_max_qvals2 : torch.Size([128, 61, 1]) : Afeter #Target Mixer

            # important sampling for PER
            if self.use_per:
                per_weight = th.from_numpy(per_weight).unsqueeze(-1).to(device=self.device)
                masked_td_error = masked_td_error.sum(1) * per_weight

            loss = L_td = masked_td_error.sum() / mask.sum()

            # Optimise
            self.optimiser.zero_grad()
            loss.backward()
            grad_norm = th.nn.utils.clip_grad_norm_(self.params, self.args.grad_norm_clip)
            self.optimiser.step()
            
        if (episode_num - self.last_target_update_episode) / self.args.target_update_interval >= 1.0:
            self._update_targets()
            self.last_target_update_episode = episode_num

        if t_env - self.log_stats_t >= self.args.learner_log_interval:
            self.logger.log_stat("loss_td", L_td.item(), t_env)
            self.logger.log_stat("grad_norm", grad_norm, t_env)
            mask_elems = mask.sum().item()
            self.logger.log_stat("td_error_abs", (masked_td_error.abs().sum().item()/mask_elems), t_env)
            self.logger.log_stat("q_taken_mean", (chosen_action_qvals * mask).sum().item()/(mask_elems * self.args.n_agents), t_env)
            self.logger.log_stat("target_mean", (targets * mask).sum().item()/(mask_elems * self.args.n_agents), t_env)
            self.log_stats_t = t_env
            
            # print estimated matrix
            if self.args.env == "one_step_matrix_game":
                print_matrix_status(batch, self.mixer, mac_out)

        # return info
        info = {}
        # calculate priority
        if self.use_per:
            if self.return_priority:
                info["td_errors_abs"] = rewards.sum(1).detach().to('cpu')
                # normalize to [0, 1]
                self.priority_max = max(th.max(info["td_errors_abs"]).item(), self.priority_max)
                self.priority_min = min(th.min(info["td_errors_abs"]).item(), self.priority_min)
                info["td_errors_abs"] = (info["td_errors_abs"] - self.priority_min) \
                                / (self.priority_max - self.priority_min + 1e-5)
            else:
                info["td_errors_abs"] = ((td_error.abs() * mask).sum(1) \
                                / th.sqrt(mask.sum(1))).detach().to('cpu')
        return info

    def _update_targets(self):
        self.target_mac.load_state(self.mac)
        if self.mixer is not None:
            self.target_mixer.load_state_dict(self.mixer.state_dict())
        self.logger.console_logger.info("Updated target network")

    def cuda(self):
        self.mac.cuda()
        self.target_mac.cuda()
        if self.mixer is not None:
            self.mixer.cuda()
            self.target_mixer.cuda()
            
    def save_models(self, path):
        self.mac.save_models(path)
        if self.mixer is not None:
            th.save(self.mixer.state_dict(), "{}/mixer.th".format(path))
        th.save(self.optimiser.state_dict(), "{}/opt.th".format(path))
        th.save(self.optimiser_ind.state_dict(), "{}/opt_ind.th".format(path))

    def load_models(self, path):
        self.mac.load_models(path)
        # Not quite right but I don't want to save target networks
        self.target_mac.load_models(path)
        if self.mixer is not None:
            self.mixer.load_state_dict(th.load("{}/mixer.th".format(path), map_location=lambda storage, loc: storage))
        self.optimiser.load_state_dict(th.load("{}/opt.th".format(path), map_location=lambda storage, loc: storage))
        self.optimiser_ind.load_state_dict(th.load("{}/opt_ind.th".format(path), map_location=lambda storage, loc: storage))
