import pygame
import random
import time
import os
import json
from pygame_screen_record import ScreenRecorder  # Use ScreenRecorder for video recording
import argparse

# Parse command-line arguments
def parse_arguments():
    parser = argparse.ArgumentParser(description="Pygame Scene Generator with Screen Recording")
    parser.add_argument('--output_dir', type=str, default="output/angle1", help='Directory to save output files')
    parser.add_argument('--tt', type=int, default=15, help='Total time for the game and recording in seconds')
    return parser.parse_args()

args = parse_arguments()

# Set window size and title
WIDTH, HEIGHT = 300, 300
LINE_WIDTH = 15
BOARD_ROWS = 3
BOARD_COLS = 3
SQUARE_SIZE = WIDTH // BOARD_COLS
CIRCLE_RADIUS = SQUARE_SIZE // 3
CIRCLE_WIDTH = 15
CROSS_WIDTH = 25
SPACE = SQUARE_SIZE // 4

# Define colors
BG_COLOR = (28, 170, 156)
LINE_COLOR = (23, 145, 135)
CIRCLE_COLOR = (239, 231, 200)
CROSS_COLOR = (66, 66, 66)

# Create output directory and file name
output_dir = args.output_dir
if not os.path.exists(output_dir):
    os.makedirs(output_dir)
timestamp = str(int(time.time()))
file_name = os.path.join(output_dir, f"{timestamp}.mp4")

# Initialize Pygame
pygame.init()
screen = pygame.display.set_mode((WIDTH, HEIGHT))
pygame.display.set_caption("Tic Tac Toe")

# Create screen recorder
recorder = ScreenRecorder(fps=60)

# Create game board
board = [[0 for _ in range(BOARD_COLS)] for _ in range(BOARD_ROWS)]

# Record each move
move_history = []

# Draw grid lines
def draw_lines():
    screen.fill(BG_COLOR)
    # Horizontal lines
    pygame.draw.line(screen, LINE_COLOR, (0, SQUARE_SIZE), (WIDTH, SQUARE_SIZE), LINE_WIDTH)
    pygame.draw.line(screen, LINE_COLOR, (0, 2 * SQUARE_SIZE), (WIDTH, 2 * SQUARE_SIZE), LINE_WIDTH)
    # Vertical lines
    pygame.draw.line(screen, LINE_COLOR, (SQUARE_SIZE, 0), (SQUARE_SIZE, HEIGHT), LINE_WIDTH)
    pygame.draw.line(screen, LINE_COLOR, (2 * SQUARE_SIZE, 0), (2 * SQUARE_SIZE, HEIGHT), LINE_WIDTH)

# Draw circles and crosses
def draw_figures():
    for row in range(BOARD_ROWS):
        for col in range(BOARD_COLS):
            if board[row][col] == 1:
                pygame.draw.circle(screen, CIRCLE_COLOR, 
                                   (int(col * SQUARE_SIZE + SQUARE_SIZE // 2), int(row * SQUARE_SIZE + SQUARE_SIZE // 2)), 
                                   CIRCLE_RADIUS, CIRCLE_WIDTH)
            elif board[row][col] == 2:
                pygame.draw.line(screen, CROSS_COLOR, 
                                 (col * SQUARE_SIZE + SPACE, row * SQUARE_SIZE + SQUARE_SIZE - SPACE), 
                                 (col * SQUARE_SIZE + SQUARE_SIZE - SPACE, row * SQUARE_SIZE + SPACE), CROSS_WIDTH)
                pygame.draw.line(screen, CROSS_COLOR, 
                                 (col * SQUARE_SIZE + SPACE, row * SQUARE_SIZE + SPACE), 
                                 (col * SQUARE_SIZE + SQUARE_SIZE - SPACE, row * SQUARE_SIZE + SQUARE_SIZE - SPACE), CROSS_WIDTH)

# Check if a player has won
def check_winner(player):
    # Rows
    for row in range(BOARD_ROWS):
        if board[row][0] == board[row][1] == board[row][2] == player:
            return True
    # Columns
    for col in range(BOARD_COLS):
        if board[0][col] == board[1][col] == board[2][col] == player:
            return True
    # Diagonals
    if board[0][0] == board[1][1] == board[2][2] == player:
        return True
    if board[2][0] == board[1][1] == board[0][2] == player:
        return True
    return False

# Check for draw
def is_draw():
    for row in range(BOARD_ROWS):
        for col in range(BOARD_COLS):
            if board[row][col] == 0:
                return False
    return True

# Randomly place a move for the player
def make_random_move(player):
    empty_squares = [(r, c) for r in range(BOARD_ROWS) for c in range(BOARD_COLS) if board[r][c] == 0]
    if empty_squares:
        row, col = random.choice(empty_squares)
        board[row][col] = player
        move_history.append({'player': player, 'position': (row, col)})

# Save questions and answers to JSON
def save_data_to_json(qa_pairs, file_name):
    data = {"questions_answers": qa_pairs}
    with open(file_name, 'w') as f:
        json.dump(data, f, indent=4)

# Main game loop
def main(total_t=15):
    draw_lines()
    player = 1  # 1 = circle, 2 = cross
    game_over = False
    winner = 0  # 0 = draw, 1 = circle wins, 2 = cross wins

    start_time = pygame.time.get_ticks()

    recorder.start_rec()  # Start screen recording

    try:
        while not game_over:
            current_time = pygame.time.get_ticks()
            elapsed_time = (current_time - start_time) / 1000  # seconds

            if elapsed_time > total_t:
                print("Game time over!")
                break

            for event in pygame.event.get():
                if event.type == pygame.QUIT:
                    pygame.quit()
                    return

            # Random move
            make_random_move(player)
            draw_figures()
            pygame.display.update()

            # Check for winner or draw
            if check_winner(player):
                print(f"Player {player} wins!")
                winner = player
                game_over = True
            elif is_draw():
                print("It's a draw!")
                game_over = True

            # Switch player
            player = 2 if player == 1 else 1

            pygame.time.wait(5000)  # Wait 5 seconds before next move

    finally:
        recorder.stop_rec()
        recorder.save_recording(file_name)
        pygame.quit()

        # Generate QA and save
        video_file = f"./{timestamp}.mp4"
        qa_list = generate_questions_answers(move_history, winner, video_file)
        qa_file = os.path.join(output_dir, f"{timestamp}.json")
        save_data_to_json(qa_list, qa_file)
        print("Questions and answers have been saved.")

# Placeholder: your generate_questions_answers function should be added above or imported

if __name__ == "__main__":
    main(total_t=args.tt)
