import pygame
import random
import math
import time
from pygame_screen_record import ScreenRecorder
from collections import Counter
import json
import os
import argparse

# Argument parsing
def parse_args():
    parser = argparse.ArgumentParser(description="Pygame simulation with question generation.")
    parser.add_argument('--t', type=int, default=20, help="Duration of the simulation in seconds")
    parser.add_argument('--x', type=int, default=10, help="Number of objects to generate")
    parser.add_argument('--n_q', type=int, default=5, help="Number of questions per type to generate")
    parser.add_argument('--output_dir', type=str, default="output", help="Directory to save the output files")
    return parser.parse_args()

args = parse_args()

# Initialize Pygame
pygame.init()

# Set screen size
screen = pygame.display.set_mode((800, 600))
clock = pygame.time.Clock()

# Define colors
RED = (255, 0, 0)
GREEN = (0, 255, 0)
BLUE = (0, 0, 255)

# Define shapes
SHAPES = ["circle", "triangle", "rect"]

# Define action types
ACTIONS = ["horizontal_move", "jump", "shake", "scale", "bounce", "vertical_move", "rotate_clockwise", "rotate_counterclockwise"]

# Get arguments for runtime, object count, questions, and output directory
t = args.t  # Duration of the simulation in seconds
x = args.x  # Number of objects
n_q = args.n_q  # Number of questions per type
output_dir = args.output_dir  # Output directory

# Generate a unique ID for saving files
id = str(int(time.time())) + '_D'
if not os.path.exists(output_dir):
    os.makedirs(output_dir)
video_file_name = os.path.join(output_dir, "angle_" + id + ".mp4")  # Video output path
qa_file_name = os.path.join(output_dir, "angle_" + id + ".json")  # QA output path

# Object update function
def update_object(obj):
    if obj["action"] == "horizontal_move":
        obj["pos"][0] += obj["speed"][0]
        if obj["pos"][0] > 800:
            obj["pos"][0] = 0
    elif obj["action"] == "jump":
        obj["pos"][1] += obj["y_speed"]
        obj["y_speed"] += 1
        if obj["pos"][1] > 500:
            obj["y_speed"] = -15
    elif obj["action"] == "shake":
        obj["pos"][0] += 2 * math.sin(pygame.time.get_ticks() / 100)
        obj["pos"][1] += 2 * math.sin(pygame.time.get_ticks() / 100)
    elif obj["action"] == "scale":
        scale = 1 + 0.5 * math.sin(pygame.time.get_ticks() / 500)
        if obj["shape"] == "circle":
            obj["radius"] = int(20 * scale)
        elif obj["shape"] == "rect":
            obj["width"] = int(50 * scale)
            obj["height"] = int(50 * scale)
        elif obj["shape"] == "triangle":
            obj["scale_factor"] = scale
    elif obj["action"] == "bounce":
        obj["pos"][0] += obj["speed"][0]
        obj["pos"][1] += obj["speed"][1]
        if obj["pos"][0] <= 0 or obj["pos"][0] >= 750:
            obj["speed"][0] = -obj["speed"][0]
        if obj["pos"][1] <= 0 or obj["pos"][1] >= 550:
            obj["speed"][1] = -obj["speed"][1]
    elif obj["action"] == "rotate_clockwise":
        if obj["shape"] == "rect":
            obj["angle"] -= abs(obj["rotation_speed"])
        else:
            obj["angle"] += abs(obj["rotation_speed"])
        if obj["angle"] >= 360 or obj["angle"] <= -360:
            obj["angle"] = 0
    elif obj["action"] == "rotate_counterclockwise":
        if obj["shape"] == "rect":
            obj["angle"] += abs(obj["rotation_speed"])
        else:
            obj["angle"] -= abs(obj["rotation_speed"])
        if obj["angle"] >= 360 or obj["angle"] <= -360:
            obj["angle"] = 0
    elif obj["action"] == "vertical_move":
        obj["pos"][1] += obj["speed"][1]
        if obj["pos"][1] <= 0 or obj["pos"][1] >= 550:
            obj["speed"][1] = -obj["speed"][1]

# Object drawing function
recorder = ScreenRecorder(120)
def draw_object(screen, obj):
    if obj["shape"] == "circle":
        radius = obj.get("radius", 20)
        pygame.draw.circle(screen, obj["color"], obj["pos"], radius)
    elif obj["shape"] == "rect":
        width = obj.get("width", 50)
        height = obj.get("height", 50)
        angle = obj.get("angle", 0)
        rect_surface = pygame.Surface((width, height), pygame.SRCALPHA)
        rect_surface.fill(obj["color"])
        rotated_surface = pygame.transform.rotate(rect_surface, angle)
        rect_center = rotated_surface.get_rect(center=(obj["pos"][0], obj["pos"][1]))
        screen.blit(rotated_surface, rect_center.topleft)
    elif obj["shape"] == "triangle":
        scale = obj.get("scale_factor", 1)
        angle = obj.get("angle", 0)
        size = 30 * scale
        points = [
            (obj["pos"][0] + size * math.cos(math.radians(angle)), obj["pos"][1] + size * math.sin(math.radians(angle))),
            (obj["pos"][0] + size * math.cos(math.radians(angle + 120)), obj["pos"][1] + size * math.sin(math.radians(angle + 120))),
            (obj["pos"][0] + size * math.cos(math.radians(angle + 240)), obj["pos"][1] + size * math.sin(math.radians(angle + 240))),
        ]
        pygame.draw.polygon(screen, obj["color"], points)

# Generate random objects
def create_random_objects(num_objects):
    objects = []
    SIZES = {
        "small": {"radius": 10, "width": 30, "height": 30},
        "medium": {"radius": 20, "width": 50, "height": 50},
        "large": {"radius": 30, "width": 70, "height": 70}
    }
    for _ in range(num_objects):
        shape = random.choice(SHAPES)
        color = random.choice([RED, GREEN, BLUE])
        size_label = random.choice(list(SIZES.keys()))
        size = SIZES[size_label]
        pos = [random.randint(50, 750), random.randint(50, 550)]
        
        if shape == "circle":
            action = random.choice([a for a in ACTIONS if "rotate" not in a])
        else:
            action = random.choice(ACTIONS)
        
        obj = {"color": color, "shape": shape, "pos": pos, "action": action, "size_label": size_label}
        
        if shape == "circle":
            obj["radius"] = size["radius"]
        elif shape == "rect":
            obj["width"] = size["width"]
            obj["height"] = size["height"]
        elif shape == "triangle":
            obj["scale_factor"] = 1
            
        if action in ["horizontal_move", "vertical_move", "bounce"]:
            obj["speed"] = [random.randint(1, 5), random.randint(1, 5)]
        if action == "jump":
            obj["y_speed"] = -15
        if action in ["rotate_clockwise", "rotate_counterclockwise"]:
            obj["angle"] = 0
            obj["rotation_speed"] = 2 if action == "rotate_clockwise" else -2

        objects.append(obj)
    
    return objects

# Initialize objects
objects = create_random_objects(x)

# Set simulation duration
start_time = pygame.time.get_ticks()

recorder.start_rec()

# Main loop
running = True
action_counter = Counter()
while running:
    if pygame.time.get_ticks() - start_time > t * 1000:
        running = False

    for event in pygame.event.get():
        if event.type == pygame.QUIT:
            running = False

    screen.fill((225, 225, 225))
    for obj in objects:
        update_object(obj)
        draw_object(screen, obj)
        action_counter[obj["action"]] += 1

    pygame.display.flip()
    clock.tick(60)

recorder.stop_rec()  
recorder.save_recording(video_file_name)

# Generate choices for a question
def generate_choices_for_question(question, answer, answer_type):
    COLORS = ["Red", "Green", "Blue"]
    SHAPES = ["circle", "triangle", "rect"]
    ACTIONS = ["horizontal_move", "jump", "shake", "scale", "bounce", "vertical_move", "rotate_clockwise", "rotate_counterclockwise"]
    if isinstance(answer, int):

        choices = [
            f"A. {answer - 1}",
            f"B. {answer}",
            f"C. {answer + 1}",
            f"D. {answer + 2}"
        ]
    elif isinstance(answer, str) and answer.lower() in ["yes", "no"]:

        choices = ["A. Yes", "B. No"]
    elif isinstance(answer, str) and "Do more objects perform the" in answer:

        action_pair = question.split(" or ")
        action_1 = action_pair[0].split(" ")[-1]
        action_2 = action_pair[1].split(" ")[0]
        choices = [f"A. Both are equal", f"B. {action_1}", f"C. {action_2}", "D. Neither"]
    elif isinstance(answer, str) and answer in ACTIONS:

        incorrect_actions = [action for action in ACTIONS if action != answer]
        random_actions = random.sample(incorrect_actions, 3)
        choices = random_actions + [answer]
        random.shuffle(choices)
        choices = [f"{chr(65 + i)}. {choice}" for i, choice in enumerate(choices)]

    elif "shape" or 'objects' in question.lower():
        incorrect_shapes = [shape for shape in SHAPES if shape != answer]
        random_shapes = random.sample(incorrect_shapes, 2)
        choices = random_shapes + [answer]
        random.shuffle(choices)
        choices = [f"{chr(65 + i)}. {choice}" for i, choice in enumerate(choices)]
    else:

        choices = [f"A. {answer}", "B. Another option", "C. Alternative", "D. Different option"]


    correct_answer = next((choice for choice in choices if str(answer) in choice), None)

    return choices, correct_answer

# Generate questions and answers
def generate_questions(objects, action_counter, n_q=5):
    qa_pairs = []
    generated_questions = set()  # Track generated questions to avoid repetition

    # Helper function to add questions without duplication
    def add_question(question, answer, answer_type):
        if question not in generated_questions:
            choices, correct_choice = generate_choices_for_question(question, answer, answer_type)
            qa_pairs.append({
                "question": question,
                "choices": choices,
                "answer": correct_choice,
                "ans": answer  # Original answer
            })
            generated_questions.add(question)

    # Question Type 1: Counting objects performing a specific action
    action_samples = random.sample(ACTIONS, min(len(ACTIONS), n_q))
    for action_type in action_samples:
        count = sum(1 for obj in objects if obj["action"] == action_type)
        question = f"How many objects are performing the '{action_type}' action in this scene?"
        add_question(question, count, 'number')

    # Question Type 2: Comparing counts between two actions
    for _ in range(n_q):
        action_type1, action_type2 = random.sample(ACTIONS, 2)
        count1 = sum(1 for obj in objects if obj["action"] == action_type1)
        count2 = sum(1 for obj in objects if obj["action"] == action_type2)
        if count1 == count2:
            answer = "Both have equal numbers."
        else:
            answer = action_type1 if count1 > count2 else action_type2
        question = f"Do more objects perform the '{action_type1}' or the '{action_type2}' action?"
        add_question(question, answer, 'action')

    # Question Type 3: Existence of objects with same shape performing different actions
    for _ in range(n_q):
        action_type1, action_type2 = random.sample(ACTIONS, 2)
        found = False
        for shape in SHAPES:
            actions_of_shape = {obj["action"] for obj in objects if obj["shape"] == shape}
            if action_type1 in actions_of_shape and action_type2 in actions_of_shape:
                found = True
                break
        answer = "Yes" if found else "No"
        question = f"Are there any objects of the same shape performing both '{action_type1}' and '{action_type2}' actions?"
        add_question(question, answer, 'yes_no')

    # Question Type 4: Identifying objects performing a specific action
    for _ in range(n_q):
        action_type = random.choice(ACTIONS)
        objs = [obj for obj in objects if obj["action"] == action_type]
        if objs:
            obj_descriptions = []
            for obj in objs:
                shape = obj["shape"]
                color_name = "Red" if obj["color"] == RED else "Green" if obj["color"] == GREEN else "Blue"
                size_label = obj["size_label"]
                obj_descriptions.append(f"{size_label} {color_name} {shape}")
            answer = ', '.join(obj_descriptions)
            question = f"Which objects are performing the '{action_type}' action?"
            add_question(question, answer, 'object_info')

    # Question Type 5: Most common action
    most_common_action = action_counter.most_common(1)[0][0]
    question = "Which action has the most objects performing it?"
    add_question(question, most_common_action, 'action')

    # Question Type 6: Action performed by a specific object
    sampled_objects = random.sample(objects, min(len(objects), n_q))
    for obj in sampled_objects:
        shape = obj["shape"]
        color_name = "Red" if obj["color"] == RED else "Green" if obj["color"] == GREEN else "Blue"
        size_label = obj["size_label"]
        action = obj["action"]
        question = f"What action is the {size_label} {color_name} {shape} performing?"
        add_question(question, action, 'action')

    # Question Type 7: Counting different actions
    action_types = set(obj["action"] for obj in objects)
    question = "How many different actions are being performed by the objects?"
    add_question(question, len(action_types), 'number')

    # Question Type 8: Shape performing the most of a specific action
    for action_type in random.sample(ACTIONS, min(len(ACTIONS), n_q)):
        shape_counts = Counter(obj["shape"] for obj in objects if obj["action"] == action_type)
        if shape_counts:
            most_common_shape = shape_counts.most_common(1)[0][0]
            question = f"What shape is performing the most '{action_type}' actions?"
            add_question(question, most_common_shape, 'shape')

    # Question Type 9: Existence of a shape performing an action
    for _ in range(n_q):
        shape = random.choice(SHAPES)
        action_type = random.choice(ACTIONS)
        exists = any(obj for obj in objects if obj["shape"] == shape and obj["action"] == action_type)
        answer = "Yes" if exists else "No"
        question = f"Is there any {shape} performing the '{action_type}' action?"
        add_question(question, answer, 'yes_no')

    return qa_pairs

def get_objects_info(objects):
    objects_info = []
    for obj in objects:
        color_name = "Red" if obj["color"] == RED else "Green" if obj["color"] == GREEN else "Blue"
        object_info = {
            "shape": obj["shape"],
            "color": color_name,
            "action": obj["action"],
            "size": obj["size_label"],
            "position": obj["pos"],
            "rotation_angle": obj.get("angle", 0)
        }
        objects_info.append(object_info)
    return objects_info

# Generate questions and save to file
qa_pairs = generate_questions(objects, action_counter, n_q)

objects_info = get_objects_info(objects)

output_data = {
    "questions_answers": qa_pairs,
    "objects_info": objects_info
}

with open(qa_file_name, 'w') as f:
    json.dump(output_data, f, indent=4)

pygame.quit()
