import pygame
from pygame_screen_record import ScreenRecorder
import random
import time 
import os
import argparse
import json
from collections import deque
import re

DIRECTIONS = ["Up", "Down", "Left", "Right"]

# Generate number-based multiple-choice answers
def generate_number_choices(correct_number, variation_range=2):
    choices = [correct_number - variation_range, correct_number - 1, correct_number, correct_number + 1]
    choices = [num for num in choices if num >= 0]
    random.shuffle(choices)
    choices = [f"{chr(65 + i)}. {num}" for i, num in enumerate(choices)]
    correct_answer = next((choice for choice in choices if str(correct_number) in choice), None)
    return choices, correct_answer

# Generate yes/no choices
def generate_yes_no_choices(answer):
    choices = ["A. Yes", "B. No"]
    correct_answer = f"A. Yes" if "yes" in answer.lower() else "B. No"
    return choices, correct_answer

# Generate direction-based choices
def generate_direction_choices(correct_direction):
    choices = [f"{chr(65 + i)}. {dir}" for i, dir in enumerate(DIRECTIONS)]
    correct_answer = next((choice for choice in choices if correct_direction.lower() in choice.lower()), None)
    return choices, correct_answer

# Generate multiple-choice options for movement sequences
def generate_movement_sequence_choices(correct_sequence):
    sequences = [
        "Down, Right, Right, Down, Right, Down, Right, Down",
        "Up, Right, Right, Up, Right, Up, Right, Up",
        "Down, Left, Left, Down, Left, Down, Left, Down",
        "Up, Left, Left, Up, Left, Up, Left, Up"
    ]
    incorrect_sequences = [seq for seq in sequences if seq != correct_sequence]
    choices = random.sample(incorrect_sequences, 3) + [correct_sequence]
    random.shuffle(choices)
    choices = [f"{chr(65 + i)}. {seq}" for i, seq in enumerate(choices)]
    correct_answer = next((choice for choice in choices if correct_sequence in choice), None)
    return choices, correct_answer

# Generate custom formatted choices (e.g. shortest path)
def generate_custom_format_choices(correct_answer, question):
    correct_number = int(re.search(r"\d+", correct_answer).group())
    incorrect_numbers = random.sample([correct_number - 2, correct_number - 1, correct_number + 1, correct_number + 2], 3)
    incorrect_choices = [f"The shortest path length is {num} steps." for num in incorrect_numbers]
    choices = [f"The shortest path length is {correct_number} steps."] + incorrect_choices
    random.shuffle(choices)
    choices = [f"{chr(65 + i)}. {choice}" for i, choice in enumerate(choices)]
    correct_answer = next((choice for choice in choices if str(correct_number) in choice), None)
    return choices, correct_answer

# Determine choice logic based on question type
def generate_choices_for_question(question, answer):
    if "how many" in question.lower():
        match = re.search(r'\d+', answer)
        if match:
            correct_number = int(match.group())
            choices, correct_answer = generate_number_choices(correct_number)
        else:
            choices, correct_answer = [], None
    elif "yes or no" in question.lower():
        choices, correct_answer = generate_yes_no_choices(answer)
    elif "which direction" in question.lower():
        match = re.search(r"The player moved (\w+)", answer)
        if match:
            correct_direction = match.group(1)
            choices, correct_answer = generate_direction_choices(correct_direction)
        else:
            correct_direction = answer.split()[-1]
            choices, correct_answer = generate_direction_choices(correct_direction)
    elif "sequence of movements" in question.lower():
        match = re.search(r"The movement sequence was: (.+)", answer)
        if match:
            correct_sequence = match.group(1)
            choices, correct_answer = generate_movement_sequence_choices(correct_sequence)
        else:
            choices, correct_answer = [], None
    elif "shortest path" in question.lower():
        choices, correct_answer = generate_custom_format_choices(answer, question)
    else:
        choices = [f"A. {answer}", "B. Option 1", "C. Option 2", "D. Option 3"]
        correct_answer = choices[0]
    return choices, correct_answer

# Breadth-first search for shortest path length
def find_shortest_path_length(grid, start_pos, end_pos):
    queue = deque()
    queue.append((start_pos[0], start_pos[1], 0))
    visited = set()
    visited.add((start_pos[0], start_pos[1]))

    while queue:
        x, y, dist = queue.popleft()

        if (x, y) == end_pos:
            return dist

        current_cell = grid[y][x]
        if not current_cell.walls['top'] and (x, y - 1) not in visited:
            queue.append((x, y - 1, dist + 1))
            visited.add((x, y - 1))
        if not current_cell.walls['right'] and (x + 1, y) not in visited:
            queue.append((x + 1, y, dist + 1))
            visited.add((x + 1, y))
        if not current_cell.walls['bottom'] and (x, y + 1) not in visited:
            queue.append((x, y + 1, dist + 1))
            visited.add((x, y + 1))
        if not current_cell.walls['left'] and (x - 1, y) not in visited:
            queue.append((x - 1, y, dist + 1))
            visited.add((x - 1, y))

    return None

# Count movement directions and sequence
def get_movement_stats(path):
    movement_sequence = []
    direction_counts = {'Up': 0, 'Down': 0, 'Left': 0, 'Right': 0}
    direction_mapping = {
        (0, -1): 'Up',
        (0, 1): 'Down',
        (-1, 0): 'Left',
        (1, 0): 'Right'
    }

    for i in range(1, len(path)):
        dx = path[i][0] - path[i - 1][0]
        dy = path[i][1] - path[i - 1][1]
        move = (dx, dy)
        direction = direction_mapping.get(move)
        if direction:
            movement_sequence.append(direction)
            direction_counts[direction] += 1

    return movement_sequence, direction_counts

# Command-line argument parsing
def parse_arguments():
    parser = argparse.ArgumentParser(description="Pygame Scene Generator with Screen Recording")
    parser.add_argument('--ts', type=int, default=20, help='Total run time of the video in seconds')
    parser.add_argument('--m', type=int, default=5, help='Number of rows in the grid')
    parser.add_argument('--n', type=int, default=5, help='Number of columns in the grid')
    parser.add_argument('--output_dir', type=str, default="output/angle1", help='Directory to save output files')
    parser.add_argument('--n_q', type=int, default=5, help='Number of questions to generate')
    return parser.parse_args()

# Question generation based on the player’s path
def generate_questions_answers(grid, path, start_pos, end_pos):
    questions_answers = []

    path_length = len(path) - 1
    shortest_path_length = find_shortest_path_length(grid, start_pos, end_pos)
    is_shortest = path_length == shortest_path_length
    movement_sequence, direction_counts = get_movement_stats(path)
    extra_steps = path_length - shortest_path_length
    max_direction = max(direction_counts, key=direction_counts.get)

    questions_answers.append({"question": "How many steps did the player take to solve the maze?", "answer": f"The player took {path_length} steps."})
    questions_answers.append({"question": "What is the length of the shortest path in the maze?", "answer": f"The shortest path length is {shortest_path_length} steps."})
    questions_answers.append({"question": "Did the player follow the optimal path?", "answer": "Yes, the player followed the shortest path." if is_shortest else "No, the player did not follow the shortest path."})
    questions_answers.append({"question": "How many times did the player move upwards?", "answer": f"The player moved upwards {direction_counts['Up']} times."})
    questions_answers.append({"question": "How many times did the player move downwards?", "answer": f"The player moved downwards {direction_counts['Down']} times."})
    questions_answers.append({"question": "How many times did the player move left?", "answer": f"The player moved left {direction_counts['Left']} times."})
    questions_answers.append({"question": "How many times did the player move right?", "answer": f"The player moved right {direction_counts['Right']} times."})
    questions_answers.append({"question": "Which direction did the player move the most?", "answer": f"The player moved {max_direction} the most, with {direction_counts[max_direction]} moves."})
    questions_answers.append({"question": "What was the complete sequence of movements?", "answer": "The movement sequence was: " + ", ".join(movement_sequence) + "."})
    if extra_steps == 0:
        answer10 = "The player took the shortest path with no extra steps."
    else:
        answer10 = f"The player took {extra_steps} extra steps compared to the shortest path."
    questions_answers.append({"question": "How many extra steps did the player take compared to the shortest path?", "answer": answer10})
    return questions_answers
