import itertools
import os
import subprocess
import time
import gc

import matplotlib.pyplot as plt
import numpy as np
import sklearn
import sklearn.metrics

from .lazy_greedy import FacilityLocation, lazy_greedy

SEED = 100
EPS = 1e-8
PLOT_NAMES = [
    "lr",
    "data_loss",
    "epoch_loss",
    "test_loss",
]  # 'cluster_compare', 'cosine_compare', 'euclidean_compare'


def similarity(X, Y=None, metric='euclidean'):
    """Computes the similarity between each pair of examples in X.

    Args
    - X: np.array, shape [N, d]
    - metric: str, one of ['cosine', 'euclidean']

    Returns
    - S: np.array, shape [N, N]
    """
    # print(f'Computing similarity for {metric}...', flush=True)
    start = time.time()

    # 1. condensed distance matrix
    # 2. square distance matrix
    #    - this must happen BEFORE converting to similarity
    #      because squareform() always puts 0 on the diagonal
    # 3. convert from distance to similarity
    # print(np.shape(X))
    # dists = scipy.spatial.distance.pdist(X, metric=metric)
    # dists = scipy.spatial.distance.squareform(dists)
    # print(f'------------> {np.shape(X)}')
    dists = sklearn.metrics.pairwise_distances(X, Y, metric=metric, n_jobs=1).T
    # dists = gdist(X, X, optimize_level=0, output='cpu')
    # dists = gdist(X, X, optimize_level=4, output='gpu')

    # ndx = int(len(X) / 2)
    # dists = np.empty((len(X), len(X)))
    # print(1)
    # s = time.time()
    # dists[:ndx, :ndx] = gdist(X[:ndx], X[:ndx], optimize_level=0, output='cpu')
    # t_1 = time.time() - s
    # print(f't1: {t_1}')
    # print(2)
    # s = time.time()
    # dists[:ndx, ndx:] = gdist(X[:ndx], X[ndx:], optimize_level=0, output='cpu')
    # t_2 = time.time() - s
    # print(f't2: {t_2}')
    # print(3)
    # s = time.time()
    # dists[ndx:, :ndx] = gdist(X[ndx:], X[:ndx], optimize_level=0, output='cpu')
    # t_3 = time.time() - s
    # print(f't3: {t_3}')
    # print(4)
    # s = time.time()
    # dists[ndx:, ndx:] = gdist(X[ndx:], X[ndx:], optimize_level=0, output='cpu')
    # t_4 = time.time() - s
    # print(f't4: {t_4}')

    elapsed = time.time() - start
    # print(f'elapsed: {elapsed} (s)')
    L0 = 0

    # print(dists)
    # print(np.shape(X))
    # print(np.shape(dists))
    # exit()
    # dists = dist(X, X, matmul='dot', method='accum', precision='auto')
    if metric == "cosine":
        S = 2 - dists
    elif metric == "euclidean" or metric == "l1":
        # S = np.exp(-dists)
        # print('before max')
        m = np.max(dists)
        # print(f'max: {m}')
        S = m - dists
        # max: 7.856385231018066
        # L0 = m * len(dists)
        # print(f'm-dist')
    else:
        raise ValueError(f"unknown metric: {metric}")

    # print(f'time (sec) for computing {metric} similarity: {elapsed}, max_d*n = {L0}', flush=True)
    return S, elapsed


def greedy_merge(X, y, B, part_num, metric, smtk=0, stoch_greedy=False):
    N = len(X)
    indices = list(range(N))
    # np.random.shuffle(indices)
    part_size = int(np.ceil(N / part_num))
    part_indices = [
        indices[slice(i * part_size, min((i + 1) * part_size, N))]
        for i in range(part_num)
    ]
    print(f"GreeDi with {part_num} parts, finding {B} elements...", flush=True)

    # pool = ThreadPool(part_num)
    # order_mg_all, cluster_sizes_all, _, _, ordering_time, similarity_time = zip(*pool.map(
    #     lambda p: get_orders_and_weights(
    #         int(B / 2), X[part_indices[p], :], metric, p + 1, stoch_greedy, y[part_indices[p]]), np.arange(part_num)))
    # pool.terminate()

    order_mg_all, cluster_sizes_all, _, _, ordering_time, similarity_time, F_val = zip(
        *map(
            lambda p: get_orders_and_weights(
                int(B / 2),
                X[part_indices[p], :],
                metric,
                p + 1,
                stoch_greedy,
                y[part_indices[p]],
            ),
            np.arange(part_num),
        )
    )

    # Returns the number of objects it has collected and deallocated
    collected = gc.collect()
    print(f"Garbage collector: collected {collected}")

    # order_mg_all = np.zeros((part_num, B))
    # cluster_sizes_all = np.zeros((part_num, B))
    # ordering_time = np.zeros(part_num)
    # similarity_time = np.zeros(part_num)
    # for p in range(part_num):
    #    order_mg_all[p,:], cluster_sizes_all[p,:], _, _, ordering_time[p], similarity_time[p] = get_orders_and_weights(
    #         B, X[part_indices[p], :], metric, p, stoch_greedy, y[part_indices[p]])
    order_mg_all = np.array(order_mg_all, dtype=np.int32)
    cluster_sizes_all = np.array(
        cluster_sizes_all, dtype=np.float32
    )  # / part_num (not needed)
    order_mg = order_mg_all.flatten(order="F")
    weights_mg = cluster_sizes_all.flatten(order="F")
    print(
        f"GreeDi stage 1: found {len(order_mg)} elements in: {np.max(ordering_time)} sec",
        flush=True,
    )

    # order_mg, weights_mg, order_sz, weights_sz, ordering_time, similarity_time
    (
        order,
        weights,
        order_sz,
        weights_sz,
        ordering_time_merge,
        similarity_time_merge,
    ) = get_orders_and_weights(
        B, X[order_mg, :], metric, smtk, stoch_greedy, y[order_mg], weights_mg
    )
    # weights /= (np.sum(weights))  # TODO <=============
    print(weights)
    total_ordering_time = np.max(ordering_time) + ordering_time_merge
    total_similarity_time = np.max(similarity_time) + similarity_time_merge
    print(
        f"GreeDi stage 2: found {len(order)} elements in: {total_ordering_time + total_similarity_time} sec",
        flush=True,
    )
    vals = (
        order,
        weights,
        order_sz,
        weights_sz,
        total_ordering_time,
        total_similarity_time,
    )
    return vals


def greedi(X, y, B, part_num, metric, smtk=0, stoch_greedy=False, seed=-1):
    N = len(X)
    indices = list(range(N))
    if seed != -1:
        np.random.seed(seed)
        np.random.shuffle(indices)  # Note: random shuffling
    part_size = int(np.ceil(N / part_num))
    part_indices = [
        indices[slice(i * part_size, min((i + 1) * part_size, N))]
        for i in range(part_num)
    ]
    print(f"GreeDi with {part_num} parts, finding {B} elements...", flush=True)

    # pool = ThreadPool(part_num)
    # order_mg_all, cluster_sizes_all, _, _, ordering_time, similarity_time = zip(*pool.map(
    #     lambda p: get_orders_and_weights(
    #         B, X[part_indices[p], :], metric, p + 1, stoch_greedy, y[part_indices[p]]), np.arange(part_num)))
    # pool.terminate()
    # Returns the number of objects it has collected and deallocated
    # collected = gc.collect()
    # print(f'Garbage collector: collected {collected}')
    order_mg_all, cluster_sizes_all, _, _, ordering_time, similarity_time = zip(
        *map(
            lambda p: get_orders_and_weights(
                B,
                X[part_indices[p], :],
                metric,
                p + 1,
                stoch_greedy,
                y[part_indices[p]],
            ),
            np.arange(part_num),
        )
    )
    gc.collect()

    order_mg_all = np.array(order_mg_all, dtype=np.int32)
    for c in np.arange(part_num):
        order_mg_all[c] = np.array(part_indices[c])[order_mg_all[c]]
    # order_mg_all = np.zeros((part_num, B))
    # cluster_sizes_all = np.zeros((part_num, B))
    # ordering_time = np.zeros(part_num)
    # similarity_time = np.zeros(part_num)
    # for p in range(part_num):
    #    order_mg_all[p,:], cluster_sizes_all[p,:], _, _, ordering_time[p], similarity_time[p] = get_orders_and_weights(
    #         B, X[part_indices[p], :], metric, p, stoch_greedy, y[part_indices[p]])
    cluster_sizes_all = np.array(
        cluster_sizes_all, dtype=np.float32
    )  # / part_num (not needed)
    order_mg = order_mg_all.flatten(order="F")
    weights_mg = cluster_sizes_all.flatten(order="F")
    print(
        f"GreeDi stage 1: found {len(order_mg)} elements in: {np.max(ordering_time)} sec",
        flush=True,
    )

    # order_mg, weights_mg, order_sz, weights_sz, ordering_time, similarity_time
    (
        order,
        weights,
        order_sz,
        weights_sz,
        ordering_time_merge,
        similarity_time_merge,
    ) = get_orders_and_weights(
        B, X[order_mg, :], metric, smtk, stoch_greedy, y[order_mg], weights_mg
    )
    # weights /= (np.sum(weights)) #TODO NOTE <=============
    # print(weights)
    order = order_mg[order]
    total_ordering_time = np.max(ordering_time) + ordering_time_merge
    total_similarity_time = np.max(similarity_time) + similarity_time_merge
    print(
        f"GreeDi stage 2: found {len(order)} elements in: {total_ordering_time + total_similarity_time} sec",
        flush=True,
    )
    vals = (
        order,
        weights,
        order_sz,
        weights_sz,
        total_ordering_time,
        total_similarity_time,
    )
    return vals


def get_facility_location_submodular_order(
    S, B, c, smtk=0, no=0, stoch_greedy=0, weights=None, subset_size=128
):
    """
    Args
    - S: np.array, shape [N, N], similarity matrix
    - B: int, number of points to select

    Returns
    - order: np.array, shape [B], order of points selected by facility location
    - sz: np.array, shape [B], type int64, size of cluster associated with each selected point
    """
    # print('Computing facility location submodular order...')
    N = S.shape[0]

    V = list(range(N))
    start = time.time()
    # encourage higher diversity
    F = FacilityLocation(S, V)
    order, _ = lazy_greedy(F, V, B)
    greedy_time = time.time() - start
    F_val = 0  # TODO

    order = np.asarray(order, dtype=np.int64)
    if len(order) < B:
        print(f'{len(order)} examples selected for class {c} is smaller than desired size {B}')
    #     unselected = np.setdiff1d(np.arange(N, dtype=np.int64), order)
    #     if len(order) > 1:
    #         group = np.argmax(S[:, order], axis=1)
    #         random_weights = [np.mean(group==g) for g in group[unselected]]
    #         random_weights = np.array(random_weights)
    #         random_subset = np.random.choice(unselected, size=B-len(order), replace=False, p=random_weights/np.sum(random_weights))
    #     else:
    #         random_subset = np.random.choice(unselected, size=B-len(order), replace=False)
    #     order = np.concatenate((order, random_subset), axis=None)
    sz = np.zeros(B, dtype=np.float64)
    for i in range(N):
        if np.max(S[i, order]) <= 0:
            continue
        if weights is None:
            sz[np.argmax(S[i, order])] += 1
        else:
            sz[np.argmax(S[i, order])] += weights[i]
    # print('time (sec) for computing facility location:', greedy_time, flush=True)
    collected = gc.collect()
    return order, sz, greedy_time, F_val


def save_cluster_sizes(sz, metric, outdir):
    """
    Args
    - sz: np.array, shape [B], type int64
    - metric: str
    - outdir: str, path to output directory, must already exist
    """
    # save list of cluster sizes
    file_path = os.path.join(outdir, metric + "_cluster_sizes.txt")
    np.savetxt(file_path, sz, fmt="%d")

    # plot histogram of cluster sizes
    file_path = os.path.join(outdir, metric + "_cluster_sizes.png")
    plt.figure()
    plt.hist(sz, bins="auto")
    plt.ylabel("count")
    plt.xlabel("size")
    plt.title("histogram of cluster sizes")
    plt.savefig(file_path)
    plt.close()


def faciliy_location_order(
    c, X, y, metric, num_per_class, smtk, no, stoch_greedy, weights=None, data_rep=None
):
    # print("here -1")
    class_indices = np.where(y == c)[0]
    # print(f"here {len(class_indices)}")
    if data_rep:
        data_rep = data_rep[class_indices]
    S, S_time = similarity(X[class_indices], data_rep, metric=metric)
    # print("here 2")
    order, cluster_sz, greedy_time, F_val = get_facility_location_submodular_order(
        S, num_per_class, c, smtk, no, stoch_greedy, weights
    )
    return class_indices[order], cluster_sz, greedy_time, S_time


def save_all_orders_and_weights(
    folder, X, metric="l2", stoch_greedy=False, y=None, equal_num=False, outdir="."
):  # todo
    N = X.shape[0]
    if y is None:
        y = np.zeros(N, dtype=np.int32)  # assign every point to the same class
    classes = np.unique(y)
    C = len(classes)  # number of classes
    # assert np.array_equal(classes, np.arange(C))
    # assert B % C == 0
    class_nums = [sum(y == c) for c in classes]
    print(class_nums)
    class_indices = [np.where(y == c)[0] for c in classes]

    tmp_path = "/lfs/local/0/baharanm/faster/tmp"
    no, smtk = 2, 2

    def greedy(B, c):
        print("Computing facility location submodular order...")
        print(
            f"Calculating ordering with SMTK... part size: {class_nums[c]}, B: {B}",
            flush=True,
        )
        command = f"/lfs/local/0/baharanm/{no}/smtk-master{smtk}/build/smraiz -sumsize {B} \
                                 -flnpy {tmp_path}/{no}/{smtk}-{c}.npy -pnpv -porder -ptime"
        if stoch_greedy:
            command += f" -stochastic-greedy -sg-epsilon {.9}"

        p = subprocess.check_output(command.split())
        s = p.decode("utf-8")
        str, end = ["([", ",])"]
        order = s[s.find(str) + len(str) : s.rfind(end)].split(",")
        order = np.asarray(order, dtype=np.int64)
        greedy_time = float(s[s.find("CPU") + 4 : s.find("s (User")])
        print(f"FL greedy time: {greedy_time}", flush=True)
        str = "f(Solution) = "
        F_val = float(s[s.find(str) + len(str) : s.find("Summary Solution") - 1])
        print(f"===========> f(Solution) = {F_val}")
        print("time (sec) for computing facility location:", greedy_time, flush=True)
        return order, greedy_time, F_val

    def get_subset_sizes(B, equal_num):
        if equal_num:
            # class_nums = [sum(y == c) for c in classes]
            num_per_class = int(np.ceil(B / C)) * np.ones(len(classes), dtype=np.int32)
            minority = class_nums < np.ceil(B / C)
            if sum(minority) > 0:
                extra = sum([max(0, np.ceil(B / C) - class_nums[c]) for c in classes])
                for c in classes[~minority]:
                    num_per_class[c] += int(np.ceil(extra / sum(minority)))
        else:
            num_per_class = np.int32(
                np.ceil(np.divide([sum(y == i) for i in classes], N) * B)
            )

        return num_per_class

    def merge_orders(order_mg_all, weights_mg_all, equal_num):
        order_mg, weights_mg = [], []
        if equal_num:
            props = np.rint([len(order_mg_all[i]) for i in range(len(order_mg_all))])
        else:
            # merging imbalanced classes
            class_ratios = np.divide([np.sum(y == i) for i in classes], N)
            props = np.rint(class_ratios / np.min(class_ratios))  # TODO
            print(f"Selecting with ratios {np.array(class_ratios)}")
            print(f"Class proportions {np.array(props)}")

        order_mg_all = np.array(order_mg_all)
        weights_mg_all = np.array(weights_mg_all)
        for i in range(
            int(np.rint(np.max([len(order_mg_all[c]) / props[c] for c in classes])))
        ):
            for c in classes:
                ndx = slice(
                    i * int(props[c]),
                    int(min(len(order_mg_all[c]), (i + 1) * props[c])),
                )
                order_mg = np.append(order_mg, order_mg_all[c][ndx])
                weights_mg = np.append(weights_mg, weights_mg_all[c][ndx])
        order_mg = np.array(order_mg, dtype=np.int32)
        weights_mg = np.array(weights_mg, dtype=np.float)
        return order_mg, weights_mg

    def calculate_weights(order, c):
        weight = np.zeros(len(order), dtype=np.float64)
        center = np.argmax(D[str(c)][:, order], axis=1)
        for i in range(len(order)):
            weight[i] = np.sum(center == i)
        return weight

    D, m = {}, 0
    similarity_times, max_similarity = [], []
    for c in classes:
        print(f"Computing distances for class {c}...")
        time.sleep(0.1)
        start = time.time()
        if metric in ["", "l2", "l1"]:
            dists = sklearn.metrics.pairwise_distances(
                X[class_indices[c]], metric=metric, n_jobs=1
            )
        else:
            p = float(metric)
            dim = class_nums[c]
            dists = np.zeros((dim, dim))
            for i in range(dim):
                dists[i, :] = np.power(
                    np.sum(
                        np.power(
                            np.abs(X[class_indices[c][i]] - X[class_indices[c]]), p
                        ),
                        axis=1,
                    ),
                    1.0 / p,
                )
                # for j in range(i+1, dim):
                #     dists[i,j] = np.power(np.sum(np.power(np.abs(X[class_indices[c][i]] - X[class_indices[c][j]]), p)), 1./p)
            # dists[np.triu_indices(dim, 1)] = d
            # dists = dists.T + dists
        similarity_times.append(time.time() - start)
        print(f"similarity times: {similarity_times}")
        print("Computing max")
        m = np.max(dists)
        print(f"max: {m}")
        S = m - dists
        np.save(f"{tmp_path}/{no}/{smtk}-{c}", S)
        D[str(c)] = S
        max_similarity.append(m)

    # Ordering all the data with greedy
    print(f"Greedy: selecting {class_nums} elements")
    # order_in_class, greedy_times, F_vals = zip(*map(lambda c: greedy(class_nums[c], c), classes))
    # order_all = [class_indices[c][order_in_class[c]] for c in classes]

    # for subset_size in [0.05, 0.1, 0.15, 0.2, 0.25, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9, 1.0]:
    for subset_size in [0.9, 1.0]:
        B = int(N * subset_size)
        num_per_class = get_subset_sizes(B, equal_num)

        # Note: for marginal gains
        order_in_class, greedy_times, F_vals = zip(
            *map(lambda c: greedy(num_per_class[c], c), classes)
        )
        order_all = [class_indices[c][order_in_class[c]] for c in classes]
        #####

        weights = [
            calculate_weights(order_in_class[c][: num_per_class[c]], c) for c in classes
        ]
        order_subset = [order_all[c][: num_per_class[c]] for c in classes]
        order_merge, weights_merge = merge_orders(order_subset, weights, equal_num)
        F_vals = np.divide(F_vals, class_nums)

        # folder = '/lfs/local/0/baharanm/faster/subsets/final/covtype'
        print(f"saving to {folder}_{subset_size}_{metric}_w.npz")
        np.savez(
            f"{folder}_{subset_size}_{metric}_w",
            order=order_merge,
            weight=weights_merge,
            order_time=greedy_times,
            similarity_time=similarity_times,
            F_vals=F_vals,
            max_dist=m,
        )
    # end for on subset sizes
    # return vals


def get_orders_and_weights(
    B,
    X,
    metric,
    smtk=0,
    no=0,
    stoch_greedy=0,
    y=None,
    weights=None,
    equal_num=False,
    outdir=".",
    data_rep=None
):  # todo
    """
    Ags
    - X: np.array, shape [N, d]
    - B: int, number of points to select
    - metric: str, one of ['cosine', 'euclidean'], for similarity
    - y: np.array, shape [N], integer class labels for C classes
      - if given, chooses B / C points per class, B must be divisible by C
    - outdir: str, path to output directory, must already exist

    Returns
    - order_mg/_sz: np.array, shape [B], type int64
      - *_mg: order points by their marginal gain in FL objective (largest gain first)
      - *_sz: order points by their cluster size (largest size first)
    - weights_mg/_sz: np.array, shape [B], type float32, sums to 1
    """
    N = X.shape[0]
    if y is None:
        y = np.zeros(N, dtype=np.int32)  # assign every point to the same class
    classes = np.unique(y)
    # print(classes)
    C = len(classes)  # number of classes
    # assert np.array_equal(classes, np.arange(C))
    # assert B % C == 0

    if equal_num:
        class_nums = [sum(y == c) for c in classes]
        num_per_class = int(np.ceil(B / C)) * np.ones(len(classes), dtype=np.int32)
        minority = class_nums < np.ceil(B / C)
        if sum(minority) > 0:
            extra = sum([max(0, np.ceil(B / C) - class_nums[c]) for c in classes])
            for c in classes[~minority]:
                num_per_class[c] += int(np.ceil(extra / sum(minority)))
    else:
        num_per_class = np.int32(
            np.ceil(np.divide([sum(y == i) for i in classes], N) * B)
        )
        # print("not equal_num")

    # print(f'Greedy: selecting {num_per_class} elements')

    # order_mg_all = np.zeros([C, num_per_class], dtype=np.int64)
    # cluster_sizes_all = np.zeros([C, num_per_class], dtype=np.float32)
    # greedy_time_all = np.zeros([C, num_per_class], dtype=np.int64)
    # similarity_time_all = np.zeros([C, num_per_class], dtype=np.int64)

    # pool = ThreadPool(C)
    # order_mg_all, cluster_sizes_all, greedy_times, similarity_times = zip(*pool.map(
    #     lambda c: faciliy_location_order(c, X, y, metric, num_per_class[c], smtk, stoch_greedy, weights), classes))
    # pool.terminate()
    # print("here 0")
    order_mg_all, cluster_sizes_all, greedy_times, similarity_times = zip(
        *map(
            lambda c: faciliy_location_order(
                c, X, y, metric, num_per_class[c], smtk, no, stoch_greedy, weights, data_rep
            ),
            classes,
        )
    )
    # print("here 1")

    order_mg, weights_mg = [], []
    if equal_num:
        props = np.rint([len(order_mg_all[i]) for i in range(len(order_mg_all))])
    else:
        # merging imbalanced classes
        class_ratios = np.divide([np.sum(y == i) for i in classes], N)
        props = np.rint(class_ratios / np.min(class_ratios))  # TODO
        # print(f"Selecting with ratios {np.array(class_ratios)}")
        # print(f"Class proportions {np.array(props)}")

    order_mg_all = np.array(order_mg_all)
    cluster_sizes_all = np.array(cluster_sizes_all)
    for i in range(
        int(np.rint(np.max([len(order_mg_all[c]) / props[c] for c in classes])))
    ):
        for c in classes:
            ndx = slice(
                i * int(props[c]), int(min(len(order_mg_all[c]), (i + 1) * props[c]))
            )
            order_mg = np.append(order_mg, order_mg_all[c][ndx])
            weights_mg = np.append(weights_mg, cluster_sizes_all[c][ndx])
    order_mg = np.array(order_mg, dtype=np.int32)

    # TODO!
    # class_ratios = np.divide([np.sum(y == i) for i in classes], N)
    # weights_mg[y[order_mg] == np.argmax(class_ratios)] /= (np.max(class_ratios) / np.min(class_ratios))

    weights_mg = np.array(
        weights_mg, dtype=np.float32
    )  # / sum(weights_mg) TODO: removed division!
    ordering_time = np.max(greedy_times)
    similarity_time = np.max(similarity_times)

    # for c in classes:
    #     class_indices = np.where(y == c)[0]
    #     S, similarity_time_all[c] = similarity(X[class_indices], metric=metric)
    #     order, cluster_sz, greedy_time_all[c], F_val = get_facility_location_submodular_order(S, num_per_class, c, smtk)
    #     order_mg_all[c] = class_indices[order]
    #     cluster_sizes_all[c] = cluster_sz
    #     save_cluster_sizes(cluster_sizes_all[c], metric=f'{metric}_class{c}', outdir=outdir)
    # cluster_sizes_all /= N

    # choose 1st from each class, then 2nd from each class, etc.
    # i.e. column-major order
    # order_mg_all = np.array(order_mg_all)
    # cluster_sizes_all = np.array(cluster_sizes_all, dtype=np.float32) / N
    # order_mg = order_mg_all.flatten(order='F')
    # weights_mg = cluster_sizes_all.flatten(order='F')

    # sort by descending cluster size within each class
    # cluster_order = np.argsort(-cluster_sizes_all, axis=1)
    # rows_selector = np.arange(C)[:, np.newaxis]
    order_sz = []  # order_mg_all[rows_selector, cluster_order].flatten(order='F')
    weights_sz = (
        []
    )  # cluster_sizes_all[rows_selector, cluster_order].flatten(order='F')
    vals = order_mg, weights_mg, order_sz, weights_sz, ordering_time, similarity_time
    return vals
