import numpy as np
import matplotlib.pyplot as plt
from scipy.optimize import minimize

# Parameters
rho, c, alpha = 0.8, 0.1, 0.5
alpha_cost = 1  # Assuming cost coefficient for Delta_rho

# Range for tau
tau_values = np.linspace(0, 1, 100)

# Store results
optimal_Delta_rho = []
optimal_Delta_c = []

# Define the objective function
def objective(x):
    Delta_rho, Delta_c = x
    return 3 * Delta_rho - ((rho + Delta_rho) * (1 - c - Delta_c)) / ((1 - alpha) * (rho + Delta_rho) + alpha)

# Constraints
def constraint1(x, tau):
    Delta_rho, Delta_c = x
    return ((1 - alpha) * (rho + Delta_rho) + alpha + c + Delta_c - 1) / (alpha - alpha * (rho + Delta_rho) + (c + Delta_c) * (rho + Delta_rho)) - tau

def constraint2(x):
    Delta_rho, Delta_c = x
    return (rho + Delta_rho) - (1 - (c + Delta_c) / (1 - alpha))

# Function to run optimization with multiple initial guesses
def optimize_for_tau(tau):
    # Bounds for Delta_rho and Delta_c
    bounds = [(0, 1 - rho), (0, 1 - c)]

    # Constraints
    cons = [{'type': 'ineq', 'fun': lambda x: constraint1(x, tau)},
            {'type': 'ineq', 'fun': constraint2}]

    # Try multiple initial guesses
    initial_guesses = [
        [0.05, 0.1],  # Original guess
        [0.1, 0.2],   # Another guess
        [0.2, 0.3],   # Another guess
        [0.01, 0.05]  # Small guess to start
    ]

    results = []

    for initial_guess in initial_guesses:
        result = minimize(objective, initial_guess, bounds=bounds, constraints=cons)
        if result.success:
            results.append(result.x)

    # Choose the best result (in terms of minimum objective value)
    if results:
        # Minimize the objective function value (the first result should be the best)
        best_result = min(results, key=lambda res: objective(res))
        return best_result[0], best_result[1]
    else:
        return None, None

# Optimization for different tau values
for tau in tau_values:
    optimal_Delta_rho_value, optimal_Delta_c_value = optimize_for_tau(tau)
    if optimal_Delta_rho_value is not None:
        optimal_Delta_rho.append(optimal_Delta_rho_value)
        optimal_Delta_c.append(optimal_Delta_c_value)
    else:
        optimal_Delta_rho.append(None)
        optimal_Delta_c.append(None)

# Plotting the results
plt.figure(figsize=(12, 6))
plt.plot(tau_values, optimal_Delta_rho, label='$\Delta_\\rho$ vs. $\\tau$', color='blue')
plt.plot(tau_values, optimal_Delta_c, label='$\Delta_c$ vs. $\\tau$', color='red')
plt.title('Optimal $(\Delta_\\rho, \Delta_c)$ vs. $\\tau$', fontsize=20)
plt.xlabel('$\\tau$', fontsize=20)
plt.ylabel('Optimal values', fontsize=20)
plt.grid(True)
plt.legend(fontsize=20)
plt.show()

