import numpy as np
import matplotlib.pyplot as plt
from scipy.optimize import fsolve
from scipy.stats import truncnorm

# Parameters
mu = 0.5         # Mean of the truncated normal
sigma = 0.1      # Standard deviation
alpha = 0.5      # Fixed alpha
c_values = [0.1, 0.2, 0.3, 0.4, 0.5]  # Values of c to vary

# Function to compute the CDF of the truncated normal distribution
def F_p(u, loc, scale):
    a, b = (0 - loc) / scale, (1 - loc) / scale  # Truncation bounds
    return truncnorm.cdf(u, a, b, loc=loc, scale=scale)

# Equation to solve for t_rho
def equation_for_t_rho(u, alpha, rho, c):
    F_p_val = F_p(u, mu, sigma)
    F_p_rho_val = F_p(u, rho * mu, sigma)
    return (1 - alpha) * F_p_val + alpha * F_p_rho_val - (1 - c)

# Define the range for rho
rho_range = np.linspace(0, 1, 1000)

# Compute baseline t_1 at rho = 1 for each c
plt.figure(figsize=(10, 6))

# Loop over each value of c
for c in c_values:
    # Compute t_1 baseline for rho = 1
    t_1 = fsolve(equation_for_t_rho, 0.5, args=(alpha, 1, c))[0]
    plt.axhline(y=t_1, color='gray', linestyle='--', alpha=0.2)

    # Compute t_rho values across rho_range
    t_rho_values = [fsolve(equation_for_t_rho, 0.5, args=(alpha, rho, c))[0] for rho in rho_range]
    plt.plot(rho_range, t_rho_values, label=f'$c={c}$')

# Plot settings
plt.title(r'Plot of $t_\rho$ for various $c$ with $\alpha = 0.5$', fontsize=25)
plt.xlabel(r'Bias $\rho$', fontsize=25)
plt.ylabel(r'Threshold $t_\rho$', fontsize=25)
plt.legend(fontsize=20)
plt.grid(True)
plt.xticks(fontsize=25)
plt.yticks(fontsize=25)
plt.tight_layout()
plt.show()
