import numpy as np
import matplotlib.pyplot as plt

# Parameters for both scenarios
alpha = 0.3
c = 0.1  # Adjusted to match your first scenario if needed
beta = 2  # Pareto distribution parameter, beta > 1

# Rho range
rho_range = np.linspace(0.001, 1, 1000)  # Adjust as needed

# Uniform Distribution R_1 and R_2 functions
def R1_A_uniform(rho, alpha=alpha, c=c):
    condition = rho < 1 - c / (1 - alpha)
    return np.where(condition, c / (1 - alpha), (1 + (1 - alpha) * rho + c * rho - rho - (1 - alpha)) / (1 + (1 - alpha) * rho - (1 - alpha)))

def R2_A_uniform(rho, alpha=alpha, c=c):
    condition = rho >= 1 - c / (1 - alpha)
    return np.where(condition, ((1 - alpha) * rho - (1 - alpha) + c) / (1 + (1 - alpha) * rho - (1 - alpha)), 0)

# Calculate R_2/R_1 for Uniform
R1_values_uniform = R1_A_uniform(rho_range)
R2_values_uniform = R2_A_uniform(rho_range)
R2_R1_ratio_uniform = R2_values_uniform / R1_values_uniform

# Find rho where R_2/R_1 is closest to 0.8
index_closest_to_08 = np.abs(R2_R1_ratio_uniform - 0.8).argmin()
rho_closest_to_08 = rho_range[index_closest_to_08]

# Plotting
plt.figure(figsize=(10, 6))
plt.plot(rho_range, R2_R1_ratio_uniform, label='', color='blue')

# Plot the modified axhline and axvline
plt.plot([0, rho_closest_to_08], [0.8, 0.8], color='gray', linestyle='--')  # Horizontal line
plt.plot([rho_closest_to_08, rho_closest_to_08], [0, 0.8], color='gray', linestyle='--')  # Vertical line

plt.scatter([rho_closest_to_08], [0.8], color='gray', zorder=5)  # Mark the intersection

# Calculate offsets for text positioning
xlim = plt.xlim()  # Get the current x-axis limits
ylim = plt.ylim()  # Get the current y-axis limits
x_offset = (xlim[1] - xlim[0]) * 0.01  # Calculate x offset as a percentage of the x-axis range
y_offset = (ylim[1] - ylim[0]) * 0.02  # Calculate y offset as a percentage of the y-axis range

# Place the text
plt.text(rho_closest_to_08 - x_offset, 0.8 + y_offset, f'({rho_closest_to_08:.3f}, 0.8)',
         verticalalignment='bottom', horizontalalignment='right', fontsize=25)

plt.title(r'Plot of $r_{\mathcal{R}}(A)$ for $c = 0.1$, $\alpha = 0.3$', fontsize=25)
plt.xlabel(r'$\rho$', fontsize=25)
plt.ylabel(r'Representation ratio $r_{\mathcal{R}}(A)$', fontsize=25)
# plt.legend()
plt.grid(True)

# Adjust tick label sizes
plt.xticks(fontsize=25)
plt.yticks(fontsize=25)

plt.show()