;;; Note: this file reads and writes to the current directory!

(define *debugging?* #f)

(define-structure experiment name mapping labels properties filter-properties)

(define (stringify strings)
 (reduce (lambda (s1 s2) (string-append s1 "-" s2))
	 (remove-if (lambda (s) (zero? (string-length s))) strings)
	 ""))

(define (hours-minutes-seconds s)
 (let* ((hours (inexact->exact (floor (/ s 60 60))))
	(minutes (inexact->exact (floor (/ (- s (* hours 60 60)) 60))))
	(seconds (- s (* hours 60 60) (* minutes 60))))
  (if (zero? hours)
      (format #f "~s:~a~s" minutes (if (< seconds 10) "0" "") seconds)
      (format #f "~s:~a~s:~a~s"
	      hours
	      (if (< minutes 10) "0" "")
	      minutes
	      (if (< seconds 10) "0" "")
	      seconds))))

(define (plural n) (if (= n 1) "" "s"))

(define (plural-es n) (if (= n 1) "" "es"))

(define (plural-y n) (if (= n 1) "y" "ies"))

(define (spaces->hyphens string)
 (list->string
  (map (lambda (c) (if (char=? c #\space) #\- c)) (string->list string))))

(define (protoclass->string protoclass)
 (cond ((string? protoclass) (spaces->hyphens protoclass))
       ((list? protoclass) (stringify (map protoclass->string protoclass)))
       (else (panic "not a string or list"))))

(define (protoclass->strings protoclass)
 (unless (every (lambda (element)
		 (or (member "on left" element) (member "on right" element)))
		protoclass)
  (fuck-up))
 (let ((left (map first
		  (remove-if-not (lambda (element) (member "on left" element))
				 protoclass)))
       (right (map first
		   (remove-if-not (lambda (element) (member "on right" element))
				  protoclass))))
  (list
   (if (member "does nothing" left)
       (cond ((and (or (member "Dan" left) (member "Scott" left))
		   (or (member "briefcase" left) (member "chair" left)))
	      (string-append
	       (cond ((member "Dan" left) "Dan")
		     ((member "Scott" left) "Scott")
		     (else (fuck-up)))
	       " and a "
	       (cond ((member "briefcase" left) "briefcase")
		     ((member "chair" left) "chair")
		     (else (fuck-up)))
	       " are on the left."))
	     ((or (member "Dan" left) (member "Scott" left))
	      (string-append
	       (cond ((member "Dan" left) "Dan")
		     ((member "Scott" left) "Scott")
		     (else (fuck-up)))
	       " is on the left."))
	     ((or (member "briefcase" left) (member "chair" left))
	      (string-append
	       "A "
	       (cond ((member "briefcase" left) "briefcase")
		     ((member "chair" left) "chair")
		     (else (fuck-up)))
	       " is on the left."))
	     (else "Nothing happens on the left."))
       (string-append
	(cond ((member "Dan" left) "Dan")
	      ((member "Scott" left) "Scott")
	      (else (fuck-up)))
	" "
	(cond ((member "pick up" left) "picked up")
	      ((member "put down" left) "put down")
	      (else (fuck-up)))
	" a "
	(cond ((member "briefcase" left) "briefcase")
	      ((member "chair" left) "chair")
	      (else (fuck-up)))
	" on the left."))
   (if (member "does nothing" right)
       (cond ((and (or (member "Dan" right) (member "Scott" right))
		   (or (member "briefcase" right) (member "chair" right)))
	      (string-append
	       (cond ((member "Dan" right) "Dan")
		     ((member "Scott" right) "Scott")
		     (else (fuck-up)))
	       " and a "
	       (cond ((member "briefcase" right) "briefcase")
		     ((member "chair" right) "chair")
		     (else (fuck-up)))
	       " are on the right."))
	     ((or (member "Dan" right) (member "Scott" right))
	      (string-append
	       (cond ((member "Dan" right) "Dan")
		     ((member "Scott" right) "Scott")
		     (else (fuck-up)))
	       " is on the right."))
	     ((or (member "briefcase" right) (member "chair" right))
	      (string-append
	       "A "
	       (cond ((member "briefcase" right) "briefcase")
		     ((member "chair" right) "chair")
		     (else (fuck-up)))
	       " is on the right."))
	     (else "Nothing happens on the right."))
       (string-append
	(cond ((member "Dan" right) "Dan")
	      ((member "Scott" right) "Scott")
	      (else (fuck-up)))
	" "
	(cond ((member "pick up" right) "picked up")
	      ((member "put down" right) "put down")
	      (else (fuck-up)))
	" a "
	(cond ((member "briefcase" right) "briefcase")
	      ((member "chair" right) "chair")
	      (else (fuck-up)))
	" on the right.")))))

(define (protoclass->filename protoclass)
 (string-append
  (stringify
   (map (lambda (element) (spaces->hyphens (first element)))
	(remove-if-not (lambda (element) (member "on left" element))
		       protoclass)))
  "-"
  (stringify
   (map (lambda (element) (spaces->hyphens (first element)))
	(remove-if-not (lambda (element) (member "on right" element))
		       protoclass)))))

(define (keep? props base-property-elements)
 (subsetp? (lambda (prop base-property-element)
	    (cond ((string? prop)
		   (cond ((string? base-property-element)
			  (equal? prop base-property-element))
			 ((list? base-property-element)
			  (or (equal? prop base-property-element)
			      (member prop base-property-element)))
			 (else (panic "not a string or list"))))
		  ((list? prop) (equal? prop base-property-element))
		  (else (panic "not a string or list"))))
	   props
	   base-property-elements))

;;; This can return fewer labels due to filter-properties.
(define (mapping-and-labels-for-experiment
	 base-properties properties filter-properties)
 (let* ((mapping-labels
         (all-values
	  ;; Pick a cross-product of properties.
	  (let* ((props (nondeterministic-map a-member-of properties))
		 (mapping
		  (map
		   cdr
		   (remove-if-not
		    (lambda (base-property)
		     (and
		      ;; Keep stimuli which have the desired properties.
		      (if (every list? (set-difference props *modalities*))
			  (keep? props (car base-property))
			  (or (keep? (map (lambda (prop)
					   (if (member prop *modalities*)
					       prop
					       (leftify prop)))
					  props)
				     (car base-property))
			      (keep? (map (lambda (prop)
					   (if (member prop *modalities*)
					       prop
					       (rightify prop)))
					  props)
				     (car base-property))))
		      ;; Filter undesirable properties.
		      (null?
		       (intersection filter-properties (car base-property)))))
		    base-properties))))
	   (when (null? mapping) (fail))
	   (list mapping (protoclass->string props)))))
        (mapping (map first mapping-labels)))
  ;; This checks that the labels for each class are disjoint.
  ;; This happens whenever there is neither on-left or on-right except for
  ;; the last experiment that is just modality, i.e. whenever there is actor,
  ;; verb, or object. This also coincides with the ISI error.
  (unless (every (lambda (a) (= (length a) 1))
		 (equivalence-classes (join mapping)))
   (panic "Bad mapping, duplicate data"))
  (list (map-indexed (lambda (a class)
		      (when (null? a) (panic "Property without any exemplars"))
		      (cons (+ class 1) a))
		     mapping)
	(map second mapping-labels))))

(define (experiment-from-base base-properties name properties filter-properties)
 (let ((mapping-labels (mapping-and-labels-for-experiment
			base-properties properties filter-properties)))
  (make-experiment name
		   (first mapping-labels)
		   (second mapping-labels)
		   properties
		   filter-properties)))

(define (logging-system command)
 (when *debugging?* (display command) (newline))
 (system command))

(define (logging-system-output command)
 (when *debugging?* (display command) (newline))
 (system-output command))

(define (label->string l) (if (list? l) (string-join "-" l) l))

(define (generate-runs-for-fold number-of-classes
				prefix-fixations
				postfix-fixations
				min-post-stimulus-fixations
				jitter-fixations
				stimulus-length
				fixation-length
				number-of-presentations
				runs/fold
				interleaved-modality?
				base-properties
				modalities)
 ;; Randomly generates a list of runs for a fold. Returns a list of lists of
 ;; integers, one for each TR. It is a 0 if it is a fixation and does not
 ;; correspond to any class. It is nonzero if that TR corresponds to that class.
 (unless (every exact?
                (list prefix-fixations
		      postfix-fixations
		      min-post-stimulus-fixations
		      jitter-fixations
		      stimulus-length
		      fixation-length
		      number-of-presentations
		      runs/fold))
  (panic "generate-runs-for-fold does not work on fractions of a TR"))
 ;; number-of-presentations is the number of presentations per class per fold.
 ;; So there are (* number-of-classes number-of-presentations) stimulus
 ;; presentations per fold. Each takes
 ;; (+ stimulus-length (* min-post-stimulus-fixations fixation-length)) TRs
 ;; because each stimulus is padded with min-post-stimulus-fixations fixations.
 (let ((stimuli/fold (* number-of-classes number-of-presentations)))
  (unless (zero? (modulo stimuli/fold runs/fold))
   (panic "The fold cannot be divided into equal-length runs"))
  (unless (zero? (modulo jitter-fixations runs/fold))
   (panic "The jitter fixations cannot be divided equaly among runs")))
 (when (and (not interleaved-modality?)
	    (not (zero? (modulo runs/fold (length modalities)))))
  (panic "The modalities cannot be divided equally among runs"))
 (let* ((stimuli-for-fold
	 ;; This returns something like
	 ;; ((1 0 0 0 0) (2 0 0 0 0) ... (K 0 0 0 0)
	 ;;  ...
	 ;;  (1 0 0 0 0) (2 0 0 0 0) ... (K 0 0 0 0))
	 ;; where K is the number-of-classes, 2 is the stimulus-length, 3
	 ;; is the min-post-stimulus-fixations*fixation-length , and each line
	 ;; is repeated number-of-presentations times.
	 (join
	  (map-n (lambda _
		  ;; This returns something like
		  ;; ((1 0 0 0 0) (2 0 0 0 0) ... (K 0 0 0 0))
		  ;; where K is the number-of-classes, 2 is the
		  ;; stimulus-length, and 3 is the
		  ;; min-post-stimulus-fixations*fixation-length.
		  (map-n (lambda (class)
			  ;; For each stimulus, a single TR with that stimulus
			  ;; is created followed by (- stimulus-length 1)
			  ;; fixations, during which the stimulus actually
			  ;; occurs. This is followed by
			  ;; min-post-stimulus-fixations actual fixations.
			  (cons (+ class 1)
				(map-n (lambda _ 0)
				       (+ (- stimulus-length 1)
					  (* min-post-stimulus-fixations
					     fixation-length)))))
			 number-of-classes))
		 number-of-presentations)))
	;; This randomly permutes the above, and divides the fold up into runs
	;; of equal length.
	(stimuli-for-runs
	 (split-into-lists-of-n
	  (if interleaved-modality?
	      (deal stimuli-for-fold)
	      (join
	       (map (lambda (modality)
		     (deal
		      (remove-if-not
		       (lambda (stimulus)
			(member modality
				(car (list-ref base-properties
					       (- (first stimulus) 1)))))
		       stimuli-for-fold)))
		    modalities)))
	  (quotient (length stimuli-for-fold) runs/fold))))
  (map (lambda (stimuli-for-run)
	(join
	 ;; This adds the prefix and postfix fixations.
	 (append
	  (map-n (lambda _ (map-n (lambda _ 0) fixation-length))
		 prefix-fixations)
	  ;; This randomly permutes each run, interspersing the jitter
	  ;; fixations before, during, and after the stimuli.
	  (deal
	   ;; This adds jitter, returning something like
	   ;; ((1 0 0 0 0)
	   ;;  (3 0 0 0 0)
	   ;;  (K 0 0 0 0)
	   ;;  ...
	   ;;  (2 0 0 0 0)
	   ;;  (0 0)
	   ;;  ...
	   ;;  (0 0))
	   (append
	    stimuli-for-run
	    ;; This returns something like ((0 0) ... (0 0)) where the length
	    ;; of each jitter fixation is fixation-length and the length of the
	    ;; whole sequence is the number of jitter fixations per run.
	    (map-n (lambda _ (map-n (lambda _ 0) fixation-length))
		   (quotient jitter-fixations runs/fold))))
	  (map-n (lambda _ (map-n (lambda _ 0) fixation-length))
		 postfix-fixations))))
       stimuli-for-runs)))

(define (generate-runs-for-subject subject
				   number-of-folds
				   seed
				   number-of-classes
				   prefix-fixations
				   postfix-fixations
				   min-post-stimulus-fixations
				   jitter-fixations
				   stimulus-length
				   fixation-length
				   number-of-presentations
				   runs/fold
				   trs/run
				   interleaved-modality?
				   base-properties
				   modalities)
 (mkdir (format #f "RSFstim/subject-~a"
		(number->padded-string-of-length (+ subject 1) 2)))
 (let ((number-of-runs (* number-of-folds runs/fold))
       ;; TR-CLASSES is a list of integers, one for each TR. The length of
       ;; this list is the number of runs times the number of TRs per run.
       (tr-classes
	(join
	 (if interleaved-modality?
	     (map-n (lambda _
		     ;; Returns a list of integers, one for each TR. This is
		     ;; for all the runs in a fold.
		     (join
		      (generate-runs-for-fold number-of-classes
					      prefix-fixations
					      postfix-fixations
					      min-post-stimulus-fixations
					      jitter-fixations
					      stimulus-length
					      fixation-length
					      number-of-presentations
					      runs/fold
					      interleaved-modality?
					      base-properties
					      modalities)))
		    number-of-folds)
	     (let ((runs-for-folds
		    (map-n (lambda _
			    ;; The runs for each fold are organized so that
			    ;; the first
			    ;; (quotiens runs/fold (length modalities)) runs
			    ;; are for the first modality, etc.
			    (generate-runs-for-fold number-of-classes
						    prefix-fixations
						    postfix-fixations
						    min-post-stimulus-fixations
						    jitter-fixations
						    stimulus-length
						    fixation-length
						    number-of-presentations
						    runs/fold
						    interleaved-modality?
						    base-properties
						    modalities))
			   number-of-folds)))
	      (map-n
	       (lambda (modality)
		(join
		 (map (lambda (runs-for-fold)
		       (join
			(map-n (lambda (run)
				(list-ref
				 runs-for-fold
				 (+ (* modality
				       (quotient runs/fold (length modalities)))
				    run)))
			       (quotient runs/fold (length modalities)))))
		      runs-for-folds)))
	       (length modalities)))))))
  ;; This writes an RSFstim*.1D file for each class. Each file has one line
  ;; per TR for the entire experiment (all runs). Each line is a 1 or a 0,
  ;; 1 indicating that that TR is for that class.
  (for-each-n
   (lambda (class)
    (write-file (map (lambda (tr-class)
		      ;; RSFgen puts spaces at the end of each line.
		      (string*-append (if (= tr-class (+ class 1)) 1 0) " "))
		     tr-classes)
		(format #f "RSFstim/subject-~a/RSFstim~a.1D"
			(number->padded-string-of-length (+ subject 1) 2)
			(+ class 1))))
   number-of-classes)
  (unless (= (/ (length tr-classes) number-of-runs) trs/run) (fuck-up))))

(define (generate-runs number-of-subjects
		       number-of-folds
		       seed
		       number-of-classes
		       prefix-fixations
		       postfix-fixations
		       min-post-stimulus-fixations
		       jitter-fixations
		       stimulus-length
		       fixation-length
		       number-of-presentations
		       runs/fold
		       trs/run
		       interleaved-modality?
		       base-properties
		       modalities)
 (srand seed)
 (rm-if-necessary "RSFstim")
 (mkdir "RSFstim")
 (for-each-n (lambda (subject)
	      (generate-runs-for-subject subject
					 number-of-folds
					 seed
					 number-of-classes
					 prefix-fixations
					 postfix-fixations
					 min-post-stimulus-fixations
					 jitter-fixations
					 stimulus-length
					 fixation-length
					 number-of-presentations
					 runs/fold
					 trs/run
					 interleaved-modality?
					 base-properties
					 modalities))
	     number-of-subjects))

(define (generate-presenter-files-for-subject subject
					      timings
					      stimuli
					      trs/run
					      runs/fold
					      number-of-folds
					      base-properties
					      stimulus-length
					      fixation-length
					      interleaved-modality?
					      modalities)
 (mkdir (format #f "presenter/subject-~a"
		(number->padded-string-of-length (+ subject 1) 2)))
 (mkdir (format #f "presenter-log/subject-~a"
		(number->padded-string-of-length (+ subject 1) 2)))
 (let* ((stimuli-by-protoclass-and-fold
	 (map (lambda (protoclass)
	       (let* ((stimuli (map first
				    (remove-if-not
				     (lambda (stimulus)
				      (equal? (second stimulus) protoclass))
				     stimuli)))
		      (stimuli-per-fold (/ (length stimuli) number-of-folds)))
		(unless (exact? stimuli-per-fold) (panic "unbalanced design"))
		(cons
		 protoclass
		 ;; Randomly permute the stimuli and divide into folds.
		 (split-into-lists-of-n (deal stimuli) stimuli-per-fold))))
	      (remove-duplicates (map second stimuli))))
	(stimulus-indices
	 (map (lambda (by-protoclass)
	       (cons (first by-protoclass)
		     (map (lambda _ 0) (rest by-protoclass))))
	      stimuli-by-protoclass-and-fold)))
  (for-each-indexed
   (lambda (run-data run)
    (write-object-to-file
     (join
      (map (lambda (a)
	    (let ((class (position 1 a)))
	     (if class
		 (let ((properties (car (list-ref base-properties class))))
		  (cond
		   ((member "video" properties)
		    (let* ((fold
			    (if interleaved-modality?
				(quotient run runs/fold)
				(+ (* (position "video" modalities)
				      (quotient runs/fold (length modalities)))
				   (quotient
				    run
				    (quotient runs/fold (length modalities))))))
			   (index (list-ref
				   (rest (assoc (remove "video" properties)
						stimulus-indices))
				   fold)))
		     (list-set! (rest (assoc (remove "video" properties)
					     stimulus-indices))
				fold
				(+ index 1))
		     `((play-multiple
			,(list-ref
			  (list-ref
			   (rest (assoc (remove "video" properties)
					stimuli-by-protoclass-and-fold))
			   fold)
			  index)
			,stimulus-length))))
		   ((member "text" properties)
		    (if *text2?*
			(let ((size (random-member '(19 21 23 25)))
			      ;; This generates a uniform sample in [0.25,0.75].
			      (y (+ (* 0.5 (random-real)) 0.25))
			      (strings
			       (deal
				(protoclass->strings
				 (remove "text" properties)))))

			 `((text2-multiple
			    ,(first strings)
			    ,(second strings)
			    ,(string-append
			      (random-member '("DejaVuSans-BoldOblique"
					       "DejaVuSans-Bold"
					       "DejaVuSansCondensed-BoldOblique"
					       "DejaVuSansCondensed-Bold"
					       "DejaVuSansCondensed-Oblique"
					       "DejaVuSansCondensed"
					       "DejaVuSans-ExtraLight"
					       "DejaVuSansMono-BoldOblique"
					       "DejaVuSansMono-Bold"
					       "DejaVuSansMono-Oblique"
					       "DejaVuSansMono"
					       "DejaVuSans-Oblique"
					       "DejaVuSans"
					       "DejaVuSerif-BoldItalic"
					       "DejaVuSerif-Bold"
					       "DejaVuSerifCondensed-BoldItalic"
					       "DejaVuSerifCondensed-Bold"
					       "DejaVuSerifCondensed-Italic"
					       "DejaVuSerifCondensed"
					       "DejaVuSerif-Italic"
					       "DejaVuSerif"))
			      "/"
			      (number->string size))
			    0.01
			    ,y
			    ,(+ y (/ (* 2 size) 600))
			    ,stimulus-length)))
			`((text-multiple
			   ,(protoclass->string (remove "text" properties))
			   ,(string-append
			     (random-member '("DejaVuSans-BoldOblique"
					      "DejaVuSans-Bold"
					      "DejaVuSansCondensed-BoldOblique"
					      "DejaVuSansCondensed-Bold"
					      "DejaVuSansCondensed-Oblique"
					      "DejaVuSansCondensed"
					      "DejaVuSans-ExtraLight"
					      "DejaVuSansMono-BoldOblique"
					      "DejaVuSansMono-Bold"
					      "DejaVuSansMono-Oblique"
					      "DejaVuSansMono"
					      "DejaVuSans-Oblique"
					      "DejaVuSans"
					      "DejaVuSerif-BoldItalic"
					      "DejaVuSerif-Bold"
					      "DejaVuSerifCondensed-BoldItalic"
					      "DejaVuSerifCondensed-Bold"
					      "DejaVuSerifCondensed-Italic"
					      "DejaVuSerifCondensed"
					      "DejaVuSerif-Italic"
					      "DejaVuSerif"))
			     "/"
			     ;; This used to be (20 30 40 50).
			     (number->string (random-member '(19 21 23 25))))
			   ;; This is a hack to prevent the text from going off
			   ;; screen. I don't know of a way to place the center
			   ;; of the text. I only know of a way to place the
			   ;; lower left corner.
			   ;; This generates a uniform sample in [0,0.4].
			   ;; This used to be ,(* 0.4 (random-real)) and the
			   ;; above used to be true.
			   0.01
			   ;; This generates a uniform sample in [0.25,0.75].
			   ,(+ (* 0.5 (random-real)) 0.25)
			   ,stimulus-length))))
		   ((member "speech" properties)
		    (panic "Can't render the predication protoclass in speech")
		    (if *speech-as-wav?*
			`((wave-multiple
			   ,(string-append
			     (random-member '("USER"
					      "sbroniko"
					      "zburchil"
					      "dpbarret"))
			     "/"
			     (protoclass->filename
			      (remove "speech" properties))
			     ".wav")
			   ,stimulus-length))
			`((speech-multiple
			   ,(second
			     (assoc (remove "speech" properties)
				    *speech-mapping*))
			   ,(random-member '("us2_mbrola"
					     "don_diphone"
					     "us1_mbrola"
					     "kal_diphone"
					     "en1_mbrola"
					     "us3_mbrola"
					     "rab_diphone"
					     "ked_diphone"))
			   ,stimulus-length))))
		   (else (fuck-up))))
		 `((fixation-multiple ,fixation-length)))))
	   ;; This is because each play-multiple, text-multiple, wave-multiple,
	   ;; speech-multiple, or fixation-multiple lasts multiple TRs.
	   (let loop ((i run-data) (l '()))
	    (if (null? i)
		(reverse l)
		(loop (drop (if (position 1 (first i))
				stimulus-length
				fixation-length)
			    i)
		      (cons (first i) l))))))
     (string*-append "presenter/subject-"
		     (number->padded-string-of-length (+ subject 1) 2)
		     "/run-"
		     (number->padded-string-of-length (+ run 1) 2)
		     ".sc")))
   ;; timings is a list of lists, one inner list per class. The inner lists have
   ;; length the total number of TRs for the whole experiment, all runs. It is
   ;; 0/1, 1 indicating that that TR has a stimulus of that class. The unzip
   ;; transposes this into a list of lists, where the outer list has length the
   ;; total number of TRs for the whole experiment, all runs. The inner lists
   ;; have length the number of classes. These are 0/1 where a 1 is in the
   ;; position of the class for the TR. Persumably there can be at most one 1.
   (split-into-lists-of-n (unzip timings) trs/run))))

(define (generate-presenter-files number-of-subjects
				  number-of-classes
				  trs/run
				  runs/fold
				  number-of-folds
				  base-properties
				  stimulus-length
				  fixation-length
				  interleaved-modality?
				  modalities)
 (rm-if-necessary "presenter")
 (mkdir "presenter")
 (rm-if-necessary "presenter-log")
 (mkdir "presenter-log")
 (for-each-n (lambda (subject)
	      (generate-presenter-files-for-subject
	       subject
	       (map-n (lambda (class)
		       (map string->number
			    (read-file
			     (string*-append
			      "RSFstim/subject-"
			      (number->padded-string-of-length (+ subject 1) 2)
			      "/RSFstim"
			      (+ class 1)
			      ".1D"))))
		      number-of-classes)
	       (read-object-from-file "stimuli.sc")
	       trs/run
	       runs/fold
	       number-of-folds
	       base-properties
	       stimulus-length
	       fixation-length
	       interleaved-modality?
	       modalities))
	     number-of-subjects))

(define (generate-experiment-classes-for-subject subject experiments)
 (mkdir (string-append "generated-experiments/subject-"
		       (number->padded-string-of-length (+ subject 1) 2)))
 (for-each
  (lambda (experiment)
   (mkdir (string-append "generated-experiments/subject-"
			 (number->padded-string-of-length (+ subject 1) 2)
			 "/"
			 (experiment-name experiment)))
   (write-object-to-file
    (map label->string (experiment-labels experiment))
    (string-append "generated-experiments/subject-"
		   (number->padded-string-of-length (+ subject 1) 2)
		   "/"
		   (experiment-name experiment)
		   "/classes.sc")))
  experiments))

(define (generate-experiment-classes number-of-subjects experiments)
 (rm-if-necessary "generated-experiments")
 (mkdir "generated-experiments")
 (for-each-n (lambda (subject)
	      (generate-experiment-classes-for-subject subject experiments))
	     number-of-subjects))

(define (generate-experiments-from-timings-for-subject subject
						       timings
						       experiments
						       trs/run
						       tr
						       stimulus-duration
						       model
						       polort
						       number-of-runs)
 (map
  (lambda (experiment)
   (let ((directory (string-append
                     "generated-experiments/subject-"
		     (number->padded-string-of-length (+ subject 1) 2)
		     "/"
		     (experiment-name experiment))))
    (define (stim-file i)
     (string*-append (experiment-name experiment) "_" (+ i 1) ".1D"))
    (define (timing-file i)
     (string*-append (experiment-name experiment)
		     "."
		     (if (>= (+ i 1) 100)
			 (number->padded-string-of-length (+ i 1) 3)
			 (number->padded-string-of-length (+ i 1) 2))
                     ".1D"))
    (for-each-indexed
     (lambda (s i)
      (write-file
       ;; RSFgen puts in a space at the end of line
       (map (lambda (l) (format #f "~a " (if (member 1 l) 1 0)))
            (unzip (map (lambda (a) (list-ref timings (- a 1))) (rest s))))
       (string*-append directory "/" (stim-file i))))
     (experiment-mapping experiment))
    (logging-system
     (format
      #f
      "cd ~a; make_stim_times.py -prefix ~a -nt ~a -tr ~a -nruns ~a -files ~a"
      directory
      (experiment-name experiment)
      trs/run
      tr
      number-of-runs
      (string-join
       " " (map-n stim-file (length (experiment-mapping experiment))))))
    (logging-system
     (format
      #f
      "cd ~a; timing_tool.py -multi_timing ~a -run_len ~a -multi_stim_dur ~a -multi_show_isi_stats"
      directory
      (string-join
       " " (map-n timing-file (length (experiment-mapping experiment))))
      (* trs/run tr)
      stimulus-duration))
    (unless (string=? (experiment-name experiment) "raw")
     (let ((out
	    (logging-system-output
	     (format
	      #f
	      ;; Not clear what the trailing space is for.
	      "cd ~a; 3dDeconvolve -concat '1D: ~a' -xjpeg ~a-xmat.jpg -nodata ~a ~a -nfirst 4 -num_glt ~a -polort ~a -num_stimts ~a ~a ~a "
	      directory
	      (string*-join
	       " " (map-n (lambda (run) (* run trs/run)) number-of-runs))
	      (experiment-name experiment)
	      (* number-of-runs trs/run)
	      tr
	      (length (map-all-unordered-pairs
		       (lambda (c1 c2) #f) (experiment-labels experiment)))
	      polort
	      (length (experiment-mapping experiment))
	      (string-join
	       " "
	       (map-n
		(lambda (a)
		 (format #f "-stim_times ~a \"~a\" \"~a\" -stim_label ~a \"~a\""
			 (+ a 1)
			 (timing-file a)
			 model
			 (+ a 1)
			 (label->string
			  (list-ref (experiment-labels experiment) a))))
		(length (experiment-mapping experiment))))
	      (string-join
	       " "
	       (map-all-unordered-pairs
		(lambda (c1 c2)
		 (format #f "-gltsym \"SYM: ~a -~a\""
			 (label->string c1) (label->string c2)))
		(experiment-labels experiment)))))))
      (when *debugging?* (display (unlines out)) (newline))
      (map (lambda (a) (string->number (second (pregexp-split "=" a))))
	   (remove-if-not (lambda (a) (prefix? "  LC[0] norm. std. dev. =" a))
			  out))))))
  experiments))

(define (generate-experiments-from-timings number-of-subjects
					   number-of-classes
					   experiments
					   trs/run
					   tr
					   stimulus-duration
					   model
					   polort
					   number-of-folds
					   runs/fold)
 (for-each-n (lambda (subject)
	      ;; This returns a result which is ignored.
	      (generate-experiments-from-timings-for-subject
	       subject
	       (map-n (lambda (class)
		       (map string->number
			    (read-file
			     (string*-append
			      "RSFstim/subject-"
			      (number->padded-string-of-length (+ subject 1) 2)
			      "/RSFstim"
			      (+ class 1)
			      ".1D"))))
		      number-of-classes)
	       experiments
	       trs/run
	       tr
	       stimulus-duration
	       model
	       polort
	       (* number-of-folds runs/fold)))
	     number-of-subjects))

(define (report-statistics tr
			   stimulus-length
			   min-post-stimulus-fixations
			   fixation-length
			   prefix-fixations
			   postfix-fixations
			   jitter-fixations
			   runs/fold
			   modalities
			   number-of-classes
			   number-of-presentations
			   number-of-folds
			   trs/run)
 (format #t "~a s/TR~%" (number->string-of-length tr 4))
 (format #t "~a TR~a/stimulus~%"
	 (number->string-of-length stimulus-length 4)
	 (plural stimulus-length))
 (format #t "~a TR~a minimum post stimulus fixation~%"
	 (number->string-of-length
	  (* min-post-stimulus-fixations fixation-length) 4)
	 (plural (* min-post-stimulus-fixations fixation-length)))
 (format #t "~a TR~a minimum prefix fixation/run~%"
	 (number->string-of-length (* prefix-fixations fixation-length) 4)
	 (plural (* prefix-fixations fixation-length)))
 (format #t "~a TR~a minimum postfix fixation/run~%"
	 (number->string-of-length (* postfix-fixations fixation-length) 4)
	 (plural (* postfix-fixations fixation-length)))
 (format #t "~a TR~a jitter fixation/run~%"
	 (number->string-of-length
	  (* (quotient jitter-fixations runs/fold) fixation-length) 4)
	 (plural (* (quotient jitter-fixations runs/fold) fixation-length)))
 (format #t "~a modalit~a~%"
	 (number->string-of-length (length modalities) 4)
	 (plural-y (length modalities)))
 (format #t "~a protoclass~a~%"
	 (number->string-of-length
	  (/ number-of-classes (length modalities)) 4)
	 (plural-es (/ number-of-classes (length modalities))))
 (format #t "~a class~a~%"
	 (number->string-of-length number-of-classes 4)
	 (plural number-of-classes))
 (format #t "~a stimulus presentation~a/protoclass/fold~%"
	 (number->string-of-length
	  (* number-of-presentations (length modalities)) 4)
	 (plural (* number-of-presentations (length modalities))))
 (format #t "~a stimulus presentation~a/class/fold~%"
	 (number->string-of-length number-of-presentations 4)
	 (plural number-of-presentations))
 (format #t "~a stimulus presentation~a/class/subject~%"
	 (number->string-of-length
	  (* number-of-presentations number-of-folds) 4)
	 (plural (* number-of-presentations number-of-folds)))
 (format #t "~a stimulus presentation~a/modality/subject~%"
	 (number->string-of-length
	  (/ (* number-of-presentations number-of-folds number-of-classes)
	     (length modalities))
	  4)
	 (plural
	  (/ (* number-of-presentations number-of-folds number-of-classes)
	     (length modalities))))
 (format #t "~a TR~a/run~%"
	 (number->string-of-length trs/run 4) (plural trs/run))
 (format #t "~a/run~%" (hours-minutes-seconds (* trs/run tr)))
 (format #t "~a run~a/subject~%"
	 (number->string-of-length (* number-of-folds runs/fold) 4)
	 (plural (* number-of-folds runs/fold)))
 (format #t "~a fold~a/subject~%"
	 (number->string-of-length number-of-folds 4)
	 (plural number-of-folds))
 (format #t "~a/subject~%"
	 (hours-minutes-seconds (* number-of-folds runs/fold trs/run tr))))

(define (old number-of-subjects
	     number-of-folds
	     seed
	     number-of-classes
	     prefix-fixations
	     postfix-fixations
	     min-post-stimulus-fixations
	     jitter-fixations
	     stimulus-length
	     fixation-length
	     number-of-presentations
	     runs/fold
	     trs/run
	     base-properties
	     modalities
	     experiments
	     tr
	     stimulus-duration
	     model
	     polort
	     interleaved-modality?)
 ;; 285 maximum number of TRs/run with 35 slices
 (when (> trs/run 285) (panic "More than 285 TRs/run"))
 (generate-runs number-of-subjects
		number-of-folds
		seed
		number-of-classes
		prefix-fixations
		postfix-fixations
		min-post-stimulus-fixations
		jitter-fixations
		stimulus-length
		fixation-length
		number-of-presentations
		runs/fold
		trs/run
		interleaved-modality?
		base-properties
		modalities)
 (generate-presenter-files number-of-subjects
			   number-of-classes
			   trs/run
			   runs/fold
			   number-of-folds
			   base-properties
			   stimulus-length
			   fixation-length
			   interleaved-modality?
			   modalities)
 (generate-experiment-classes number-of-subjects experiments)
 (generate-experiments-from-timings number-of-subjects
				    number-of-classes
				    experiments
				    trs/run
				    tr
				    stimulus-duration
				    model
				    polort
				    number-of-folds
				    runs/fold)
 (report-statistics tr
		    stimulus-length
		    min-post-stimulus-fixations
		    fixation-length
		    prefix-fixations
		    postfix-fixations
		    jitter-fixations
		    runs/fold
		    modalities
		    number-of-classes
		    number-of-presentations
		    number-of-folds
		    trs/run))

(define (new number-of-subjects
	     number-of-folds
	     seed
	     number-of-classes
	     prefix-fixations
	     postfix-fixations
	     min-post-stimulus-fixations
	     jitter-fixations
	     stimulus-length
	     fixation-length
	     number-of-presentations
	     runs/fold
	     trs/run
	     base-properties
	     modalities
	     experiments
	     tr
	     stimulus-duration
	     model
	     polort
	     interleaved-modality?)
 ;; 285 maximum number of TRs/run with 35 slices
 (when (> trs/run 285) (panic "More than 285 TRs/run"))
 (srand seed)
 (rm-if-necessary "RSFstim")
 (mkdir "RSFstim")
 (rm-if-necessary "presenter")
 (mkdir "presenter")
 (rm-if-necessary "presenter-log")
 (mkdir "presenter-log")
 (rm-if-necessary "generated-experiments")
 (mkdir "generated-experiments")
 (for-each-n (lambda (subject)
	      (generate-runs-for-subject subject
					 number-of-folds
					 seed
					 number-of-classes
					 prefix-fixations
					 postfix-fixations
					 min-post-stimulus-fixations
					 jitter-fixations
					 stimulus-length
					 fixation-length
					 number-of-presentations
					 runs/fold
					 trs/run
					 interleaved-modality?
					 base-properties
					 modalities)
	      (generate-presenter-files-for-subject
	       subject
	       (map-n (lambda (class)
		       (map string->number
			    (read-file
			     (string*-append
			      "RSFstim/subject-"
			      (number->padded-string-of-length (+ subject 1) 2)
			      "/RSFstim"
			      (+ class 1)
			      ".1D"))))
		      number-of-classes)
	       (read-object-from-file "stimuli.sc")
	       trs/run
	       runs/fold
	       number-of-folds
	       base-properties
	       stimulus-length
	       fixation-length
	       interleaved-modality?
	       modalities)
	      (generate-experiment-classes-for-subject subject experiments)
	      ;; This returns a result which is ignored.
	      (generate-experiments-from-timings-for-subject
	       subject
	       (map-n (lambda (class)
		       (map string->number
			    (read-file
			     (string*-append
			      "RSFstim/subject-"
			      (number->padded-string-of-length (+ subject 1) 2)
			      "/RSFstim"
			      (+ class 1)
			      ".1D"))))
		      number-of-classes)
	       experiments
	       trs/run
	       tr
	       stimulus-duration
	       model
	       polort
	       (* number-of-folds runs/fold)))
	     number-of-subjects)
 (report-statistics tr
		    stimulus-length
		    min-post-stimulus-fixations
		    fixation-length
		    prefix-fixations
		    postfix-fixations
		    jitter-fixations
		    runs/fold
		    modalities
		    number-of-classes
		    number-of-presentations
		    number-of-folds
		    trs/run))

(define *modalities* '("video" "text"))

(define *speech-mapping* '())

(define *actors* '("nobody" "Dan" "Scott"))

(define *verbs* '("does nothing" "pick up" "put down"))

(define *objects* '("nothing" "briefcase" "chair"))

(define (leftify thing) (list thing "on left"))

(define (rightify thing) (list thing "on right"))

(define (no-duplicate-actors cross-product)
 (when (and (not (string=? (first (first cross-product)) "nobody"))
	    (not (string=? (first (fourth cross-product)) "nobody"))
	    (string=? (first (first cross-product))
		      (first (fourth cross-product))))
  (fail))
 cross-product)

(define (verb-requires-actor-and-object cross-product)
 (when (or (and (not (string=? (first (second cross-product)) "does nothing"))
		(or (string=? (first (first cross-product)) "nobody")
		    (string=? (first (third cross-product)) "nothing")))
	   (and (not (string=? (first (fifth cross-product)) "does nothing"))
		(or (string=? (first (fourth cross-product)) "nobody")
		    (string=? (first (sixth cross-product)) "nothing"))))
  (fail))
 cross-product)

(define (something-must-happen cross-product)
 (when (and (string=? (first (second cross-product)) "does nothing")
	    (string=? (first (fifth cross-product)) "does nothing"))
  (fail))
 cross-product)

(define *base-properties*
 (map-indexed
  (lambda (a class) (cons a (+ class 1)))
  (all-values
   (something-must-happen
    (verb-requires-actor-and-object
     (no-duplicate-actors
      (nondeterministic-map
       a-member-of
       (list (map leftify *actors*)
	     (map leftify *verbs*)
	     (map leftify *objects*)
	     (map rightify *actors*)
	     (map rightify *verbs*)
	     (map rightify *objects*)
	     *modalities*))))))))

(define *experiments*
 ;; This has 7*3*4+1=85 experiments:
 ;;   {actor,verb,object,actor-verb,actor-object,verb-object,actor-verb-object}*
 ;;   {-on-left,-on-right,-on-either}*
 ;;   {-from-video,-from-text,-from-both,-modality},
 ;;   modality.
 ;; The pair and triple are joint. Independent must be handled elsewhere.
 ;; Ditto for location=modality.
 ;; The location=-on-either is "joint". Could also have an "independent" but it
 ;; must be handled elsewhere. I put quotes because it is disjunctive, not
 ;; conjunctive.
 ;; Ditto for modality=-from-both.
 ;; needs work: This does not include transfer from on left to on right and
 ;;             vice versa.
 ;; needs work: This does not include transfer from video to text and vice
 ;;             versa.
 ;; On h27jun2015 I eliminated on-either because it results in the
 ;; "Bad mapping, duplicate data" error and ISI error. Because there can be
 ;; two different actors, one on the left and one on the right. Ditto for verb
 ;; and object. Now there are 7*2*4+1=57 experiments.
 (append
  (all-values
   (let ((constituents (a-subset-of '(actor verb object)))
	 ;; This used to contain on-either.
	 (location (a-member-of '(on-left on-right)))
	 (modality (a-member-of '(from-video from-text from-both modality))))
    (when (null? constituents) (fail))
    (experiment-from-base
     *base-properties*
     (stringify
      (append (map (lambda (constituent)
		    (case constituent
		     ((actor) "actor")
		     ((verb) "verb")
		     ((object) "object")
		     (else (fuck-up))))
		   constituents)
	      (list (case location
		     ((on-left) "on-left")
		     ((on-right) "on-right")
		     ((on-either) "on-either")
		     (else (fuck-up))))
	      (list (case modality
		     ((from-video) "from-video")
		     ((from-text) "from-text")
		     ((from-both) "from-both")
		     ((modality) "modality")
		     (else (fuck-up))))))
     (append (map (lambda (constituent)
		   (map (case location
			 ((on-left) leftify)
			 ((on-right) rightify)
			 ((on-either) identity)
			 (else (fuck-up)))
			(case constituent
			 ((actor) *actors*)
			 ((verb) *verbs*)
			 ((object) *objects*)
			 (else (fuck-up)))))
		  constituents)
	     (if (eq? modality 'modality) (list *modalities*) '()))
     (case modality
      ((from-video) '("text"))
      ((from-text) '("video"))
      ((from-both) '())
      ((modality) '())
      (else (fuck-up))))))
  (list (experiment-from-base
	 *base-properties* "modality" (list *modalities*) '())
	(experiment-from-base
	 *base-properties*
	 "raw"
	 (list (map leftify *actors*)
	       (map leftify *verbs*)
	       (map leftify *objects*)
	       (map rightify *actors*)
	       (map rightify *verbs*)
	       (map rightify *objects*)
	       *modalities*)
	 '()))))

(define *speech-as-wav?* #t)
(define *spread* 1)
(define *number-of-subjects* 13)
(define *number-of-folds* 4)
(define *seed* 19)
(define *number-of-classes* (length *base-properties*))
(define *prefix-fixations* 4)
(define *postfix-fixations* 10)
(define *min-post-stimulus-fixations* 1)
(define *jitter-fixations* 192)
(define *stimulus-length* (* 2 *spread*)) ; number of TRs
(define *fixation-length* (* 1 *spread*)) ; number of TRs
(define *number-of-presentations* 1)
(define *runs/fold* 4)
(define *trs/run*
 (+ (* (quotient (* *number-of-classes* *number-of-presentations*) *runs/fold*)
       *stimulus-length*)
    (* (+ (* (quotient (* *number-of-classes* *number-of-presentations*)
		       *runs/fold*)
	     *min-post-stimulus-fixations*)
	  (quotient *jitter-fixations* *runs/fold*)
	  *prefix-fixations*
	  *postfix-fixations*)
       *fixation-length*)))
(define *tr* 2)				; seconds/TR
(define *stimulus-duration* (* *stimulus-length* *tr*))	; seconds
;;; needs work: I don't know what this is.
(define *model* "GAM")			; BLOCK(2,1)
;;; needs work: I don't know what this is.
(define *polort* 5)
(define *interleaved-modality?* #t)
(define *text2?* #t)

;;;   2 s/TR
;;;   2 TRs/stimulus
;;;   1 TR minimum post stimulus fixation
;;;   4 TRs minimum prefix fixation/run
;;;  10 TRs minimum postfix fixation/run
;;;  48 TRs jitter fixation/run
;;;   2 modalities
;;; 128 protoclasses
;;; 256 classs
;;;   2 stimulus presentations/protoclass/fold
;;;   1 stimulus presentation/class/fold
;;;   4 stimulus presentations/class/subject
;;; 512 stimulus presentations/modality/subject
;;; 254 TRs/run
;;; 8:28/run
;;;  16 runs/subject
;;;   4 folds/subject
;;; 2:15:28/subject

(new *number-of-subjects*
     *number-of-folds*
     *seed*
     *number-of-classes*
     *prefix-fixations*
     *postfix-fixations*
     *min-post-stimulus-fixations*
     *jitter-fixations*
     *stimulus-length*
     *fixation-length*
     *number-of-presentations*
     *runs/fold*
     *trs/run*
     *base-properties*
     *modalities*
     *experiments*
     *tr*
     *stimulus-duration*
     *model*
     *polort*
     *interleaved-modality?*)
