(MODULE PRESENTER (WITH QOBISCHEME XLIB) (MAIN MAIN))

;;; Two windows, a regular define-application and a window for displaying to
;;; the subject
;;; The latter has no buttons, clickability, or anything similar
;;; A preview in the main window shows you what the subjects are seeing

(include "QobiScheme.sch")
(include "presenter.sch")
(include "presenterlib-c-macros.sch")

(c-include "Imlib2.h")
(c-include "stdlib.h")
(c-include "presenterlib-c.h")

(set! *program* "presenter")
(set! *panic?* #f)

;;; Structures

(define-structure rc-renderer-log
 start-timestamp volumes counter iteration type)

(define-structure rc-renderer-result
 ;; finished-sequence or was-stopped
 stop-reason
 timepoints-processed
 log)

(c-define-struct-field "advance" "s" double)

(c-define-struct-field "sleep" "s" double)

(c-define-struct-field "festival" "text" pointer)

(c-define-struct-field "fill_rectangle" "x" double)
(c-define-struct-field "fill_rectangle" "y" double)
(c-define-struct-field "fill_rectangle" "width" double)
(c-define-struct-field "fill_rectangle" "height" double)
(c-define-struct-field "fill_rectangle" "r" int)
(c-define-struct-field "fill_rectangle" "g" int)
(c-define-struct-field "fill_rectangle" "b" int)
(c-define-struct-field "fill_rectangle" "a" int)

(c-define-struct-field "image" "image" pointer)
(c-define-struct-field "image" "x" double)
(c-define-struct-field "image" "y" double)
(c-define-struct-field "image" "width" double)
(c-define-struct-field "image" "height" double)

(c-define-struct-field "text" "text" pointer)
(c-define-struct-field "text" "font" pointer)
(c-define-struct-field "text" "r" int)
(c-define-struct-field "text" "g" int)
(c-define-struct-field "text" "b" int)
(c-define-struct-field "text" "a" int)
(c-define-struct-field "text" "direction" int)
(c-define-struct-field "text" "angle" double)
(c-define-struct-field "text" "x" double)
(c-define-struct-field "text" "y" double)

(c-define-struct-field "load_video" "ffmpeg_video" pointer)
(c-define-struct-field "load_video" "id" int)

(c-define-struct-field "show_video_frame" "id" int)
(c-define-struct-field "show_video_frame" "x" double)
(c-define-struct-field "show_video_frame" "y" double)
(c-define-struct-field "show_video_frame" "width" double)
(c-define-struct-field "show_video_frame" "height" double)
(c-define-struct-field "show_video_frame" "a" int)

(c-define-struct-field "advance_video_frame" "id" int)

(c-define-struct-field "loop" "iterations" unsigned)

(c-define-struct-field "renderer_commands" "type" unsigned)
(c-define-struct-field "renderer_commands" "data" pointer)
(c-define-struct-field "renderer_commands" "next" pointer)

(c-define-struct-field "renderer_sequence" "commands" pointer)
(c-define-struct-field "renderer_sequence" "next" pointer)

(c-define-struct-field "renderer_target" "window" pointer)
(c-define-struct-field "renderer_target" "x" int)
(c-define-struct-field "renderer_target" "y" int)
(c-define-struct-field "renderer_target" "width" int)
(c-define-struct-field "renderer_target" "height" int)

(c-define-struct-field "renderer_task_args" "sequence_length" int)
(c-define-struct-field "renderer_task_args" "sequence" pointer)
(c-define-struct-field "renderer_task_args" "number_of_targets" int)
(c-define-struct-field "renderer_task_args" "targets" pointer)
(c-define-struct-field "renderer_task_args" "wakeup_target" pointer)
(c-define-struct-field "renderer_task_args" "trs" unsigned)
(c-define-struct-field "renderer_task_args" "tr" double)
(c-define-struct-field "renderer_task_args" "countdown" int)
(c-define-struct-field "renderer_task_args" "message_pane" int)
(c-define-struct-field "renderer_task_args" "message_pane_y" int)

(c-define-struct-field "renderer_log" "start_timestamp" double)
(c-define-struct-field "renderer_log" "volumes" unsigned)
(c-define-struct-field "renderer_log" "counter" unsigned)
(c-define-struct-field "renderer_log" "iteration" unsigned)
(c-define-struct-field "renderer_log" "type" unsigned)

(c-define-struct-field "renderer_result" "stop_reason" unsigned)
(c-define-struct-field "renderer_result" "timepoints_processed" int)
(c-define-struct-field "renderer_result" "log" pointer)

;;; Variables

(define *window-width* 800)
(define *window-height* 600)

(define *stimulus-x* 0)
(define *stimulus-y* 0)
(define *stimulus-width* 800)
(define *stimulus-height* 600)
(define *stimulus-window* #f)

(define *disable-preview* #f)

(define *image-cache* '())

(define *renderer-running?* #f)

(define *run-sequences* '())

(define *run* 1)

(define *subject* 1)

(define *renderer-task-args* #f)

(define *log-directory* #f)

(define *trs* #f)

(define *tr* #f)

;;; General

(define presenter-malloc (c-function pointer ("presenter_malloc" int)))

(define free (c-function void ("free" pointer)))

(define (foldr f i l)
 (let loop ((l l)) (if (null? l) i (f (first l) (loop (rest l))))))

(define (string->c-string string)
 ;; needs work: We never free these strings.
 (let ((c-string (presenter-malloc (+ (string-length string) 1))))
  (for-each-indexed
   (lambda (char i) (c-byte-set! c-string i (char->integer char)))
   (string->list string))
  (c-byte-set! c-string (string-length string) 0)
  c-string))

;;; Image Cache

(define (load-image-with-cache! filename)
 (let ((entry (assoc filename *image-cache*)))
  (if entry
      (cdr entry)
      (let ((image ((c-function pointer ("imlib_load_image_immediately" string))
		    filename)))
       (when (zero? image) (panic "Failed to load ~a" filename))
       (set! *image-cache* (cons (cons filename image) *image-cache*))
       image))))

(define (free-image-cache)
 (for-each (lambda (entry)
            ((c-function void ("imlib_context_set_image" pointer)) (cdr entry))
            ((c-function void ("imlib_free_image"))))
	   *image-cache*)
 (set! *image-cache* '()))

;;; Commands

(define (type-of type)
 (cond
  ((= type (c-value unsigned "ADVANCE")) 'advance)
  ((= type (c-value unsigned "SLEEP")) 'sleep)
  ((= type (c-value unsigned "FESTIVAL")) 'festival)
  ((= type (c-value unsigned "FILL_RECTANGLE")) 'fill-rectangle)
  ((= type (c-value unsigned "IMAGE")) 'image)
  ((= type (c-value unsigned "TEXT")) 'text)
  ((= type (c-value unsigned "RENDER")) 'render)
  ((= type (c-value unsigned "WAKE_GUI")) 'wake-gui)
  ((= type (c-value unsigned "LOAD_VIDEO")) 'load-video)
  ((= type (c-value unsigned "SHOW_VIDEO_FRAME")) 'show-video-frame)
  ((= type (c-value unsigned "ADVANCE_VIDEO_FRAME")) 'advance-video-frame)
  ((= type (c-value unsigned "LOOP")) 'loop)
  ((= type (c-value unsigned "START_VOLUME")) 'start-volume)
  ((= type (c-value unsigned "WAIT_FOR_VOLUME")) 'wait-for-volume)
  ((= type (c-value unsigned "STOP_ON_VOLUME_WITHOUT_CLEARING"))
   'stop-on-volume-without-clearing)
  ((= type (c-value unsigned "STOP_ON_VOLUME_AND_CLEAR"))
   'stop-on-volume-and-clear)
  (else (fuck-up))))

(define (rc-advance s)
 (let ((command (presenter-malloc (c-sizeof "struct renderer_commands"))))
  (renderer-commands-type-set! command (c-value unsigned "ADVANCE"))
  (renderer-commands-data-set!
   command (presenter-malloc (c-sizeof "struct advance")))
  (advance-s-set! (renderer-commands-data command) s)
  command))

(define (rc-sleep s)
 (let ((command (presenter-malloc (c-sizeof "struct renderer_commands"))))
  (renderer-commands-type-set! command (c-value unsigned "SLEEP"))
  (renderer-commands-data-set!
   command (presenter-malloc (c-sizeof "struct sleep")))
  (sleep-s-set! (renderer-commands-data command) s)
  command))

(define (rc-festival text)
 (let ((command (presenter-malloc (c-sizeof "struct renderer_commands"))))
  (renderer-commands-type-set!
   command (c-value unsigned "FESTIVAL"))
  (renderer-commands-data-set!
   command (presenter-malloc (c-sizeof "struct festival")))
  (festival-text-set! (renderer-commands-data command) (string->c-string text))
  command))

(define (rc-fill-rectangle x y width height rgba)
 (let ((command (presenter-malloc (c-sizeof "struct renderer_commands"))))
  (renderer-commands-type-set! command (c-value unsigned "FILL_RECTANGLE"))
  (renderer-commands-data-set!
   command (presenter-malloc (c-sizeof "struct fill_rectangle")))
  (fill-rectangle-x-set! (renderer-commands-data command) x)
  (fill-rectangle-y-set! (renderer-commands-data command) y)
  (fill-rectangle-width-set! (renderer-commands-data command) width)
  (fill-rectangle-height-set! (renderer-commands-data command) height)
  (fill-rectangle-r-set! (renderer-commands-data command) (vector-ref rgba 0))
  (fill-rectangle-g-set! (renderer-commands-data command) (vector-ref rgba 1))
  (fill-rectangle-b-set! (renderer-commands-data command) (vector-ref rgba 2))
  (fill-rectangle-a-set! (renderer-commands-data command) (vector-ref rgba 3))
  command))

(define (rc-image filename x y width height)
 ;; not currently used
 (let ((command (presenter-malloc (c-sizeof "struct renderer_commands"))))
  (renderer-commands-type-set! command (c-value unsigned "IMAGE"))
  (renderer-commands-data-set!
   command (presenter-malloc (c-sizeof "struct image")))
  (image-image-set! (renderer-commands-data command)
		    (load-image-with-cache! filename))
  (image-x-set! (renderer-commands-data command) x)
  (image-y-set! (renderer-commands-data command) y)
  (image-width-set! (renderer-commands-data command) width)
  (image-height-set! (renderer-commands-data command) height)
  command))

(define (rc-text text font rgba direction angle x y)
 (let ((command (presenter-malloc (c-sizeof "struct renderer_commands"))))
  (renderer-commands-type-set! command (c-value unsigned "TEXT"))
  (renderer-commands-data-set!
   command (presenter-malloc (c-sizeof "struct text")))
  (text-text-set! (renderer-commands-data command) (string->c-string text))
  (text-font-set! (renderer-commands-data command) (string->c-string font))
  (text-r-set! (renderer-commands-data command) (vector-ref rgba 0))
  (text-g-set! (renderer-commands-data command) (vector-ref rgba 1))
  (text-b-set! (renderer-commands-data command) (vector-ref rgba 2))
  (text-a-set! (renderer-commands-data command) (vector-ref rgba 3))
  (text-direction-set! (renderer-commands-data command) direction)
  (text-angle-set! (renderer-commands-data command) angle)
  (text-x-set! (renderer-commands-data command) x)
  (text-y-set! (renderer-commands-data command) y)
  command))

(define (rc-render)
 (let ((command (presenter-malloc (c-sizeof "struct renderer_commands"))))
  (renderer-commands-type-set! command (c-value unsigned "RENDER"))
  command))

(define (rc-wake-gui)
 ;; not currently used
 (let ((command (presenter-malloc (c-sizeof "struct renderer_commands"))))
  (renderer-commands-type-set! command (c-value unsigned "WAKE_GUI"))
  command))

(define (rc-load-video filename id)
 (let ((command (presenter-malloc (c-sizeof "struct renderer_commands"))))
  (renderer-commands-type-set! command (c-value unsigned "LOAD_VIDEO"))
  (renderer-commands-data-set!
   command (presenter-malloc (c-sizeof "struct load_video")))
  (load-video-ffmpeg-video-set!
   (renderer-commands-data command)
   ((c-function pointer ("ffmpeg_open_video" string)) filename))
  (load-video-id-set! (renderer-commands-data command) id)
  command))

(define (rc-show-video-frame id x y width height a)
 (let ((command (presenter-malloc (c-sizeof "struct renderer_commands"))))
  (renderer-commands-type-set! command (c-value unsigned "SHOW_VIDEO_FRAME"))
  (renderer-commands-data-set!
   command (presenter-malloc (c-sizeof "struct show_video_frame")))
  (show-video-frame-id-set! (renderer-commands-data command) id)
  (show-video-frame-x-set! (renderer-commands-data command) x)
  (show-video-frame-y-set! (renderer-commands-data command) y)
  (show-video-frame-width-set! (renderer-commands-data command) width)
  (show-video-frame-height-set! (renderer-commands-data command) height)
  (show-video-frame-a-set! (renderer-commands-data command) a)
  command))

(define (rc-advance-video-frame id)
 (let ((command (presenter-malloc (c-sizeof "struct renderer_commands"))))
  (renderer-commands-type-set! command (c-value unsigned "ADVANCE_VIDEO_FRAME"))
  (renderer-commands-data-set!
   command (presenter-malloc (c-sizeof "struct advance_video_frame")))
  (advance-video-frame-id-set! (renderer-commands-data command) id)
  command))

(define (rc-loop iterations)
 (let ((command (presenter-malloc (c-sizeof "struct renderer_commands"))))
  (renderer-commands-type-set! command (c-value unsigned "LOOP"))
  (renderer-commands-data-set!
   command (presenter-malloc (c-sizeof "struct loop")))
  (loop-iterations-set! (renderer-commands-data command) iterations)
  command))

(define (rc-start-volume)
 (let ((command (presenter-malloc (c-sizeof "struct renderer_commands"))))
  (renderer-commands-type-set! command (c-value unsigned "START_VOLUME"))
  command))

(define (rc-wait-for-volume)
 (let ((command (presenter-malloc (c-sizeof "struct renderer_commands"))))
  (renderer-commands-type-set! command (c-value unsigned "WAIT_FOR_VOLUME"))
  command))

(define (rc-stop-on-volume-without-clearing)
 (let ((command (presenter-malloc (c-sizeof "struct renderer_commands"))))
  (renderer-commands-type-set!
   command (c-value unsigned "STOP_ON_VOLUME_WITHOUT_CLEARING"))
  command))

(define (rc-stop-on-volume-and-clear)
 (let ((command (presenter-malloc (c-sizeof "struct renderer_commands"))))
  (renderer-commands-type-set!
   command (c-value unsigned "STOP_ON_VOLUME_AND_CLEAR"))
  command))

(define (sequence-length sequence)
 (map-reduce
  +
  0
  (lambda (commands)
   (+ (map-reduce
       max
       0
       (lambda (command) (loop-iterations (renderer-commands-data command)))
       (remove-if-not
	(lambda (command)
	 (= (renderer-commands-type command) (c-value unsigned "LOOP")))
	commands))
      1))
  sequence))

(define (commands->c commands)
 (foldr (lambda (command next)
	 (renderer-commands-next-set! command next)
	 command)
	0
	commands))

(define (sequence->c sequence)
 (foldr (lambda (commands next)
	 (let ((s (presenter-malloc (c-sizeof "struct renderer_sequence"))))
	  (renderer-sequence-commands-set! s (commands->c commands))
	  (renderer-sequence-next-set! s next)
	  s))
	0
	sequence))

(define (renderer-task-args->c sequence trs tr)
 (let ((a (presenter-malloc (c-sizeof "struct renderer_task_args"))))
  (renderer-task-args-sequence-length-set! a (sequence-length sequence))
  (renderer-task-args-sequence-set! a (sequence->c sequence))
  (cond
   (*disable-preview*
    (let ((ts (presenter-malloc (c-sizeof "struct renderer_target"))))
     (renderer-task-args-number-of-targets-set! a 1)
     (renderer-target-window-set! ts *stimulus-window*)
     (renderer-target-width-set! ts *stimulus-width*)
     (renderer-target-height-set! ts *stimulus-height*)
     (renderer-target-x-set! ts 0)
     (renderer-target-y-set! ts 0)
     (renderer-task-args-targets-set! a ts)))
   (else
    (let ((ts (presenter-malloc (* 2 (c-sizeof "struct renderer_target")))))
     (renderer-task-args-number-of-targets-set! a 2)
     (renderer-target-window-set! ts *stimulus-window*)
     (renderer-target-width-set! ts *stimulus-width*)
     (renderer-target-height-set! ts *stimulus-height*)
     (renderer-target-x-set! ts 0)
     (renderer-target-y-set! ts 0)
     (renderer-target-window-set! (+ ts (c-sizeof "struct renderer_target"))
				  *display-pane*)
     (renderer-target-width-set! (+ ts (c-sizeof "struct renderer_target"))
				 *display-pane-width*)
     (renderer-target-height-set! (+ ts (c-sizeof "struct renderer_target"))
				  *display-pane-height*)
     (renderer-target-x-set! (+ ts (c-sizeof "struct renderer_target")) 0)
     (renderer-target-y-set! (+ ts (c-sizeof "struct renderer_target")) 0)
     (renderer-task-args-number-of-targets-set! a 2)
     (renderer-task-args-targets-set! a ts))))
  (renderer-task-args-wakeup-target-set! a *display-pane*)
  (renderer-task-args-trs-set! a trs)
  (renderer-task-args-tr-set! a tr)
  (renderer-task-args-countdown-set! a (if *disable-preview* 0 1))
  (unless *disable-preview*
   (renderer-task-args-message-pane-set! a *message-pane*)
   (renderer-task-args-message-pane-y-set!
    a (if *disable-preview* 0 (- *who-line-height* (+ *roman-baseline* 2)))))
  a))

(define (free-renderer)
 ((c-function void ("free_renderer_task_args" pointer)) *renderer-task-args*)
 (free-image-cache))

;;; Logs

(define (log->scheme-and-free timepoints-processed log)
 (let ((l (map-n (lambda (offset)
                  (let ((renderer-log
			 (+ log (* offset (c-sizeof "struct renderer_log")))))
                   (make-rc-renderer-log
		    (renderer-log-start-timestamp renderer-log)
		    (renderer-log-volumes renderer-log)
		    (renderer-log-counter renderer-log)
		    (renderer-log-iteration renderer-log)
		    (type-of (renderer-log-type renderer-log)))))
		 timepoints-processed)))
  (free log)
  l))

(define (result->scheme-and-free result)
 (let* ((stop-reason (renderer-result-stop-reason result))
	(timepoints-processed (renderer-result-timepoints-processed result))
        (log (log->scheme-and-free
	      timepoints-processed (renderer-result-log result))))
  (free result)
  (make-rc-renderer-result
   (cond ((= stop-reason (c-value unsigned "RENDERER_FINISHED_SEQUENCE"))
          'finished-sequence)
         ((= stop-reason (c-value unsigned "RENDERER_WAS_STOPPED"))
          'was-stopped)
         (else (panic "unknown stop reason")))
   timepoints-processed
   log)))

(define (system-output command)
 (with-temporary-file "/tmp/system.out"
		      (lambda (file)
		       (system (format #f "~a > ~s" command file))
		       (read-file file))))

(define (stop-and-log)
 (free-renderer)
 (set! *renderer-running?* #f)
 (let ((log (result->scheme-and-free
	     ((c-function pointer ("stop_renderer_thread"))))))
  (when *log-directory*
   (write-object-to-file
    log
    (format #f "~a/subject-~a/~a-~a.log"
	    *log-directory*
	    (number->padded-string-of-length *subject* 2)
	    (number->padded-string-of-length *run* 2)
	    (first (system-output "date +%s")))))))

;;; Timepoints

(define (crosshair)
 ;; White crosshair on black background.
 `(,(rc-fill-rectangle 0 0 1 1 '#(0 0 0 255))
   ,(rc-fill-rectangle 0.47 0.495 0.06 0.01 '#(255 255 255 255))
   ,(rc-fill-rectangle 0.495 0.46 0.01 0.08 '#(255 255 255 255))))

(define (simple-timepoint . contents) contents)

(define (standard-timepoint time . contents)
 `(,(rc-advance time)
   ,@contents
   ,(rc-render)
   ,(rc-sleep time)))

(define (tr-timepoint . contents)
 `(,@contents
   ,(rc-render)
   ,(rc-wait-for-volume)))

(define (fixation-timepoint) (apply tr-timepoint (crosshair)))

(define (optional-fixation-until-trigger-timepoint)
 ;; If the trigger already happened, don't do anything and clear the flag. If
 ;; not, show a crosshair, wait for the trigger, and then clear the flag.
 `(,(rc-stop-on-volume-and-clear)
   ,@(crosshair)
   ,(rc-render)
   ,(rc-wait-for-volume)))

(define (text-timepoint text font x y)
 ;; Black text on white background.
 ;; This assumes is that the immediate previous timepoint ended with a trigger
 ;; by rc-wait-for-volume which clears the flag. This is the case if the
 ;; previous timepoint was a fixation or a text-timepoint.
 `(,(rc-fill-rectangle 0 0 1 1 '#(255 255 255 255))
   ,(rc-text text font '#(0 0 0 255) (c-value int "IMLIB_TEXT_TO_RIGHT") 0 x y)
   ,(rc-render)
   ,(rc-wait-for-volume)))

(define (text2-timepoint text1 text2 font x y1 y2)
 ;; Like text-timepoint except draws two lines of text at different y positions
 ;; but the same font and x position.
 `(,(rc-fill-rectangle 0 0 1 1 '#(255 255 255 255))
   ,(rc-text
     text1 font '#(0 0 0 255) (c-value int "IMLIB_TEXT_TO_RIGHT") 0 x y1)
   ,(rc-text
     text2 font '#(0 0 0 255) (c-value int "IMLIB_TEXT_TO_RIGHT") 0 x y2)
   ,(rc-render)
   ,(rc-wait-for-volume)))

;;; Top Level

(define-application gui *window-width* *window-height* #f 1 8
 (lambda ()
  (define-button 0 0 "Help" #f help-command)
  (define-button 1 0 "Start" #f
   (lambda ()
    (when *renderer-running?* (message "already running") (abort))
    (message (format #f "setting up subject ~s run ~s" *subject* *run*))
    (set! *renderer-running?* #t)
    (set! *renderer-task-args*
	  (renderer-task-args->c
	   ((list-ref (list-ref *run-sequences* (- *subject* 1)) (- *run* 1)))
	   *trs*
	   *tr*))
    (message (format #f "starting subject ~s run ~s" *subject* *run*))
    ((c-function void ("start_renderer_thread" pointer)) *renderer-task-args*)))
  (define-button 2 0 "Stop" #f
   (lambda ()
    (message "stopped renderer")
    (unless *renderer-running?* (message "renderer isn't running") (abort))
    (stop-and-log)))
  (define-integer-range-buttons 3 0 4 0 *run* 1
   (length (list-ref *run-sequences* (- *subject* 1)))
   (lambda () (format #f "-run ~s" *run*))
   (lambda () (format #f "+run ~s" *run*))
   (lambda () #f))
  (define-integer-range-buttons 5 0 6 0 *subject* 1 (length *run-sequences*)
   ;; +subject and -subject only allow one digit to fit in button width
   (lambda () (format #f "-subj ~s" *subject*))
   (lambda () (format #f "+subj ~s" *subject*))
   (lambda ()
    (set! *run*
	  (min *run* (length (list-ref *run-sequences* (- *subject* 1)))))))
  (define-button 7 0 "Quit" #f quit)
  (define-key (control #\h) "Help" help-command)
  (define-key (list (control #\x) (control #\c)) "Quit" quit)
  (define-key escape "Finish"
   (lambda ()
    (when (and (c-value bool "renderer_stopped") *renderer-running?*)
     (message "renderer finished")
     (stop-and-log))))
  (set! *stimulus-window*
        (xcreatesimplewindow
         *display* *root-window*
         *stimulus-x* *stimulus-y*
         *stimulus-width* *stimulus-height*
         1
         (xcolor-pixel (second *foreground*))
         (xcolor-pixel (second *background*))))
  (let ((hints (make-xsizehints)))
   (xsizehints-x! hints *stimulus-x*)
   (xsizehints-y! hints *stimulus-y*)
   (xsizehints-min_width! hints *stimulus-width*)
   (xsizehints-max_width! hints *stimulus-width*)
   (xsizehints-min_height! hints *stimulus-height*)
   (xsizehints-max_height! hints *stimulus-height*)
   (xsizehints-flags! hints (+ usposition pposition pminsize pmaxsize))
   (xsetwmnormalhints *display* *stimulus-window* hints))
  (xmapraised *display* *stimulus-window*))
 (lambda () #f)
 (lambda () (xdestroywindow *display* *stimulus-window*))
 (lambda () #f))

(define (xpdf-beamer root-window x y width height)
 (set! *display* (xopendisplay *display-name*))
 (when (null-pointer? *display*)
  (panic "Cannot connect to X server: ~a" (xdisplayname *display-name*)))
 (set! *screen* (xdefaultscreen *display*))
 (set! *stimulus-window*
       (xcreatesimplewindow
	*display* root-window x y width height 0
	(xwhitepixel *display* *screen*)
	(xblackpixel *display* *screen*)))
 (let ((hints (make-xsizehints)))
  (xsizehints-x! hints x)
  (xsizehints-y! hints y)
  (xsizehints-min_width! hints width)
  (xsizehints-max_width! hints width)
  (xsizehints-min_height! hints height)
  (xsizehints-max_height! hints height)
  (xsizehints-flags! hints (+ usposition pposition pminsize pmaxsize))
  (xsetwmnormalhints *display* *stimulus-window* hints))
 (xmapsubwindows *display* *stimulus-window*)
 (xmapraised *display* *stimulus-window*)
 (xflush *display*)
 (set! *renderer-running?* #t)
 ;; This is only for wakeup target.
 (set! *display-pane* *stimulus-window*)
 (set! *renderer-task-args*
       (renderer-task-args->c
	((list-ref (list-ref *run-sequences* (- *subject* 1)) (- *run* 1)))
	*trs*
	*tr*))
 ((c-function void ("start_renderer_thread" pointer)) *renderer-task-args*)
 (let loop () (when (c-value bool "renderer_stopped") (loop)))
 (let loop () (unless (c-value bool "renderer_stopped") (loop)))
 (stop-and-log)
 (xdestroywindow *display* *stimulus-window*))

(define-command (main
		 (at-most-one
		  ("xpdf-beamer"
		   xpdf-beamer?
		   (xpdf-beamer-wid "wid" string-argument 0)
		   (xpdf-beamer-x "x" integer-argument 0)
		   (xpdf-beamer-y "y" integer-argument 0)
		   (xpdf-beamer-width "width" integer-argument 0)
		   (xpdf-beamer-height "height" integer-argument 0)))
		 (at-most-one ("window-position"
			       window-position?
			       (window-x "x" integer-argument 0)
			       (window-y "y" integer-argument 0)))
		 (at-most-one ("window-size"
			       window-size?
			       (window-width "width" integer-argument 800)
			       (window-height "height" integer-argument 600)))
                 (at-most-one ("stimulus-position"
			       stimulus-position?
			       (stimulus-x "x" integer-argument 0)
			       (stimulus-y "y" integer-argument 0)))
                 (at-most-one ("stimulus-size"
			       stimulus-size?
			       (stimulus-width "width" integer-argument 800)
			       (stimulus-height "height" integer-argument 600)))
                 (at-most-one ("disable-preview" disable-preview?))
                 (at-most-one
		  ("play-multiple-alternate" play-multiple-alternate?))
		 (at-most-one ("wave"
			       wave?
			       (wave-directory "directory" string-argument "")))
                 (exactly-one
		  ("subjects-directory"
		   subjects-directory?
		   (subjects-directory "subjects-directory" string-argument "")
		   (stimuli-directory "stimuli-directory" string-argument "")
		   ;; needs work: This really belongs in the run file.
		   (fps "fps" integer-argument 0)
		   ;; needs work: This really belongs in the run file.
		   (frames "frames" integer-argument 0)))
		 ;; needs work: This really belongs in the run file.
                 (exactly-one ("tr" tr?
			       ;; Could probably determine this from the run
			       ;; file but only for more modern run files.
			       (trs "TRs" integer-argument 0)
                               (seconds/tr "seconds" real-argument 0)
                               (slices/tr "slices" integer-argument 0)))
		 (at-most-one ("log" log?
                               (log-directory "directory" string-argument "")))
		 ;; This is because triggers are sometimes lost on the real
		 ;; scanner.
		 (at-most-one ("but-last" but-last?)))
 (define (process-event e)
  ;; ME-Y2 has 48 PLAY3-BLANK-PLAY3 interspersed between multiple
  ;; FIXATION, a total of 158. Each PLAY3-BLANK-PLAY3 takes
  ;; exactly 2.5+0.5+2.5=5.5 seconds. Since PLAY3 ends with a
  ;; START_VOLUME, the first FIXATION terminates immediately.
  ;;
  ;; 9events has 72 PLAY2 interspersed between multiple FIXATION,
  ;; a total of 172. Each PLAY2 and each FIXATION waits for a
  ;; (single) volume. Thus there are 144 volumes/run. The command
  ;; line specifies two crucial parameters: fps and frames, that
  ;; govern how long PLAY2 lasts. The command line specifies one
  ;; crucial parameter slices/tr that governs the trigger and
  ;; thus how long to wait for a volumne for both PLAY2 and
  ;; FIXATION.
  (cond
   ((eq? e 'fixation)
    ;; This is obsolete and superseded by fixation-multiple.
    ;; It was used in ME-Y2, 9events, and hollwood2.
    ;; Draw a fixation crosshair, render, and wait for a volume.
    (list (fixation-timepoint)))
   ((and (list? e) (eq? (first e) 'fixation-multiple))
    ;; Draw a fixation crosshair, render, and wait for the
    ;; specified number of volumes.
    (map-n (lambda _ (fixation-timepoint)) (second e)))
   ((and (list? e) (eq? (first e) 'blank))
    ;; This was used in ME-Y2 but probably shouldn't be used anymore.
    ;; Clear the screen, render, and sleep for the specified
    ;; amount of time, checking time.
    (list (standard-timepoint
	   (second e)
	   (rc-fill-rectangle 0 0 1 1 '#(0 0 0 255)))))
   ((and (list? e) (= (length e) 3) (eq? (first e) 'play))
    ;; This was used in ME-Y2 but probably shouldn't be used anymore.
    ;; Two timepoints
    ;;   - Load a video, show the first frame, render, and
    ;;     sleep for a frame, checking time.
    ;;   - Advance the video, show the next frame, start a
    ;;     volume, render, and sleep for a frame, checking time.
    ;;     LOOP for the amount of time specified in seconds
    ;;     minus 2 frames (one for the first timepoint and one
    ;;     for the first iteration of the second timepoint).
    ;; What this in effect does is play the video for the amount
    ;; of time specified in seconds, checking time for each frame,
    ;; and starting a volume for each frame.
    ;; needs work: Why does it START_VOLUME? This is the only
    ;;             place it ever does.
    ;; needs work: What happens if the video runs out before the
    ;;             time specified in seconds?
    (list (standard-timepoint
	   (/ fps)
	   (rc-load-video
	    (string-append stimuli-directory "/" (second e)) 0)
	   (rc-show-video-frame 0 0 0 1 1 255))
	  (standard-timepoint
	   (/ fps)
	   (rc-advance-video-frame 0)
	   (rc-show-video-frame 0 0 0 1 1 255)
	   (rc-loop (- (* fps (third e)) 2))
	   (rc-start-volume))))
   ((and (list? e) (= (length e) 2) (eq? (first e) 'play))
    ;; This was used in 9events but probably shouldn't be used
    ;; anymore.
    ;; Three timepoints
    ;;  - Load a video, skip 4 frames, show a frame, advance a
    ;;    frame, render, and sleep for a frame, checking time.
    ;;  - STOP-ON-VOLUME-WITHOUT-CLEARING, advance a frame, show
    ;;    a frame, render, and sleep for a frame, checking time.
    ;;    LOOP for the specified number of frames minus 5.
    ;;  - STOP-ON-VOLUME-AND-CLEAR, draw a fixation crosshair,
    ;;    render, and wait for a volume.
    ;; What this does in effect is play the video for the
    ;; specified number of frames, minus 3, skipping the first
    ;; 4 frames and the 6th frame, showing a fixation crosshair
    ;; after the specified number of frames until a volume is
    ;; captured. The video is cut short, without showing a
    ;; fixation crosshair, after playing at least one frame, when
    ;; a volume is captured.
    ;; needs work: Why does it play 3 less frames?
    ;;             Why does it skip the first 4 frames?
    ;;             Why does it skip the 6th frame?
    ;; needs work: What happens if the video runs out before the
    ;;             specified number of frames?
    (list (standard-timepoint
	   (/ fps)
	   (rc-load-video
	    (string-append stimuli-directory "/" (second e)) 0)
	   (rc-advance-video-frame 0)
	   (rc-advance-video-frame 0)
	   (rc-advance-video-frame 0)
	   (rc-advance-video-frame 0)
	   (rc-show-video-frame 0 0 0 1 1 255)
	   (rc-advance-video-frame 0))
	  (standard-timepoint
	   (/ fps)
	   (rc-stop-on-volume-without-clearing)
	   (rc-advance-video-frame 0)
	   (rc-show-video-frame 0 0 0 1 1 255)
	   (rc-loop (- frames 5)))
	  (optional-fixation-until-trigger-timepoint)))
   ((and (list? e) (eq? (first e) 'play4))
    ;; This is obsolete and superseded by play-multiple.
    ;; It was used in hollwood2.
    ;; Show video for exactly 2 TRs, which is 4s with 2s TRs.
    ;; Assumes 10fps and assumes video has at least 40 frames.
    ;; Shows at most 40 frames. Pads at end with fixation if
    ;; necessary.
    (list
     ;; This shows the first frame. The assumption is that the
     ;; immediate previous timepoint was fixation so that it
     ;; ended with a trigger by rc-wait-for-volume which clears
     ;; the flag.
     (standard-timepoint
      (/ fps)
      (rc-load-video
       (string-append stimuli-directory "/" (second e)) 0)
      (rc-show-video-frame 0 0 0 1 1 255))
     ;; This is hardwired to show the next 29 frames (at 10fps
     ;; with 2s TRs) is to make sure that we are well into the
     ;; second TR before ceasing to ignore the trigger.
     (standard-timepoint
      (/ fps)
      (rc-advance-video-frame 0)
      (rc-show-video-frame 0 0 0 1 1 255)
      (rc-start-volume)
      (rc-loop (- 29 1)))
     ;; This is hardwired to show at most 10 more frames but to
     ;; stop with fewer that 10 frames if there is a trigger.
     ;; But it doesn't clear the flag. The previous time point
     ;; ended with rc-start-volume so this time point will start
     ;; with the flag clear. This time point will end with the
     ;; flag set if the trigger happened (and thus possibly fewer
     ;; than 40 frames have been shown or with the flag clear if
     ;; the trigger hasn't happened yet.
     (standard-timepoint
      (/ fps)
      (rc-stop-on-volume-without-clearing)
      (rc-advance-video-frame 0)
      (rc-show-video-frame 0 0 0 1 1 255)
      (rc-loop (- 10 1)))
     (optional-fixation-until-trigger-timepoint)))
   ((and (list? e) (eq? (first e) 'play-multiple))
    (unless (= (* seconds/tr fps (third e)) frames)
     (panic "calculated number of frames ~s doesn't match specified number of frames ~s"
	    (* seconds/tr fps (third e)) frames))
    (if play-multiple-alternate?
	;; Show video for exactly a specified number of TRs at the
	;; specified frame rate. Assumes video has sufficient frames.
	;; Shows at most the number of frames as calculated by the
	;; specified number of TRs, the specified framerate, and the
	;; specified TR. Pads at end with fixation if necessary.
	(list
	 ;; This shows the first frame. The assumption is that the
	 ;; immediate previous timepoint was fixation so that it
	 ;; ended with a trigger by rc-wait-for-volume which clears
	 ;; the flag.
	 (standard-timepoint
	  (/ fps)
	  (rc-load-video
	   (string-append stimuli-directory "/" (second e)) 0)
	  (rc-show-video-frame 0 0 0 1 1 255))
	 ;; This shows all of the remaining frames except half of a
	 ;; TR's worth to make sure that we are well into the last TR
	 ;; before ceasing to ignore the trigger.
	 (standard-timepoint
	  (/ fps)
	  (rc-advance-video-frame 0)
	  (rc-show-video-frame 0 0 0 1 1 255)
	  (rc-start-volume)
	  (rc-loop
	   ;; The - 2 is to remove the first frame shown by the
	   ;; previous time point and the first iteration of the
	   ;; current time point.
	   (- (inexact->exact
	       (floor (- frames (/ (* seconds/tr fps) 2))))
	      2)))
	 ;; This shows at most all of the remaining frames but stops
	 ;; with unplayed frames if there is a trigger. But it
	 ;; doesn't clear the flag. The previous time point ended
	 ;; with rc-start-volume so this time point will start with
	 ;; the flag clear. This time point will end with the flag
	 ;; set if the trigger happened (and thus possibly fewer than
	 ;; all frames will have been shown or with the flag clear if
	 ;; the trigger hasn't happened yet.
	 (standard-timepoint
	  (/ fps)
	  (rc-stop-on-volume-without-clearing)
	  (rc-advance-video-frame 0)
	  (rc-show-video-frame 0 0 0 1 1 255)
	  (rc-loop
	   ;; The - 1 is to remove the first iteration of the
	   ;; current time point.
	   (- (- frames
		 (inexact->exact
		  (floor (- frames (/ (* seconds/tr fps) 2)))))
	      1)))
	 (optional-fixation-until-trigger-timepoint))
	;; Show video for exactly a specified number of TRs at the
	;; specified frame rate. Assumes video has sufficient frames.
	;; Shows exactly the number of frames as calculated by the
	;; specified number of TRs, the specified framerate, and the
	;; specified TR. The frames are synced with the TR trigger.
	(cons
	 ;; This loads the video. The assumption is that the
	 ;; immediate previous timepoint was fixation so that it
	 ;; ended with a trigger by rc-wait-for-volume which clears
	 ;; the flag.
	 (simple-timepoint
	  (rc-load-video
	   (string-append stimuli-directory "/" (second e)) 0))
	 (map-reduce-n
	  append
	  '()
	  (lambda _
	   (append
	    ;; Show all but the last frame for this TR.
	    (map-n
	     (lambda _
	      (standard-timepoint
	       (/ fps)
	       (rc-show-video-frame 0 0 0 1 1 255)
	       (rc-advance-video-frame 0)))
	     (- (inexact->exact (round (* seconds/tr fps))) 1))
	    ;; Show the last frame for this TR and wait for the
	    ;; trigger.
	    (list (tr-timepoint
		   (rc-show-video-frame 0 0 0 1 1 255)
		   (rc-advance-video-frame 0)))))
	  (third e)))))
   ((and (list? e) (eq? (first e) 'text4))
    ;; This is obsolete and superseded by text-multiple.
    ;; It was used in hollwood2.
    ;; Show text for exactly 2 TRs, which is 4s with 2s TRs.
    (list
     (text-timepoint (second e) (third e) (fourth e) (fifth e))
     (text-timepoint (second e) (third e) (fourth e) (fifth e))))
   ((and (list? e) (eq? (first e) 'text-multiple))
    ;; Show text for exactly a specified number of TRs, synced with
    ;; the TR trigger.
    (map-n
     (lambda _
      (text-timepoint (second e) (third e) (fourth e) (fifth e)))
     (sixth e)))
   ((and (list? e) (eq? (first e) 'text2-multiple))
    ;; Show text for exactly a specified number of TRs, synced with
    ;; the TR trigger.
    (map-n
     (lambda _
      (text2-timepoint
       (second e) (third e) (fourth e) (fifth e) (sixth e) (seventh e)))
     (eighth e)))
   ((and (list? e) (eq? (first e) 'speech-multiple))
    (cons
     (simple-timepoint
      (rc-festival "(audio_mode 'async)")
      (rc-festival (string-append "(voice_" (third e) ")"))
      (rc-festival (string-append "(SayText \"" (second e) "\")")))
     ;; Draw a fixation crosshair, render, and wait for the
     ;; specified number of volumes.
     (map-n (lambda _ (fixation-timepoint)) (fourth e))))
   ((and (list? e) (eq? (first e) 'wave-multiple))
    (cons (simple-timepoint
	   (rc-festival "(audio_mode 'async)")
	   (rc-festival "(set! u (Utterance Text \"\"))")
	   (rc-festival
	    (string-append "(utt.import.wave u \""
			   wave-directory
			   "/"
			   (second e)
			   "\")"))
	   (rc-festival "(utt.play u)"))
	  ;; Draw a fixation crosshair, render, and wait for the
	  ;; specified number of volumes.
	  (map-n (lambda _ (fixation-timepoint)) (third e))))
   (else (panic "Unknown run command ~s" e))))
 (when (and xpdf-beamer? window-position?)
  (panic "Cannot simultaneously specify -xpdf-beamer and -window-position"))
 (when (and xpdf-beamer? window-size?)
  (panic "Cannot simultaneously specify -xpdf-beamer and -window-size"))
 (when (and xpdf-beamer? stimulus-position?)
  (panic "Cannot simultaneously specify -xpdf-beamer and -stimulus-position"))
 (when (and xpdf-beamer? stimulus-size?)
  (panic "Cannot simultaneously specify -xpdf-beamer and -stimulus-size"))
 (when (and xpdf-beamer? disable-preview?)
  (panic "Cannot simultaneously specify -xpdf-beamer and -disable-preview"))
 (when (and xpdf-beamer? log?)
  (panic "Cannot simultaneously specify -xpdf-beamer and -log"))
 (set! *trs* trs)
 (set! *tr* seconds/tr)
 (cond
  (xpdf-beamer?
   (set! *disable-preview* #t)
   (set!
    *run-sequences*
    (list
     (list
      (lambda ()
       (map-reduce
	append '() process-event (read-object-from-file subjects-directory))))))
   ((c-function void ("imlib_add_path_to_font_path" string))
    (cond ((file-exists? "/usr/share/fonts/truetype/ttf-dejavu")
	   "/usr/share/fonts/truetype/ttf-dejavu")
	  ((file-exists? "/usr/share/fonts/dejavu")
	   "/usr/share/fonts/dejavu")
	  ((file-exists? "/usr/share/fonts/TTF")
	   "/usr/share/fonts/TTF")
	  (else (panic "Can't find a font directory"))))
   ((c-function void ("start_trigger_thread" int)) slices/tr)
   (xpdf-beamer (string->number
		 (substring xpdf-beamer-wid 2 (string-length xpdf-beamer-wid))
		 16)
		xpdf-beamer-x
		xpdf-beamer-y
		xpdf-beamer-width
		xpdf-beamer-height))
  (else
   (when window-position?
    (set! *window-position?* #t)
    (set! *window-position-x* window-x)
    (set! *window-position-y* window-y))
   (set! *window-width* window-width)
   (set! *window-height* window-height)
   (set! *stimulus-x* stimulus-x)
   (set! *stimulus-y* stimulus-y)
   (set! *stimulus-width* stimulus-width)
   (set! *stimulus-height* stimulus-height)
   (set! *disable-preview* disable-preview?)
   (when log? (set! *log-directory* log-directory))
   (set!
    *run-sequences*
    (map
     (lambda (runs-directory)
      (map (lambda (run)
	    (lambda ()
	     (map-reduce
	      append
	      '()
	      process-event
	      ((if but-last? but-last identity)
	       (read-object-from-file
		(string-append
		 subjects-directory "/" runs-directory "/" run))))))
	   ;; This assumes that the files are returned in order to correspond
	   ;; to runs 1,...,R.
	   (directory-list
	    (string-append subjects-directory "/" runs-directory))))
     (directory-list subjects-directory)))
   ((c-function void ("imlib_add_path_to_font_path" string))
    (cond ((file-exists? "/usr/share/fonts/truetype/ttf-dejavu")
	   "/usr/share/fonts/truetype/ttf-dejavu")
	  ((file-exists? "/usr/share/fonts/dejavu")
	   "/usr/share/fonts/dejavu")
	  ((file-exists? "/usr/share/fonts/TTF")
	   "/usr/share/fonts/TTF")
	  (else (panic "Can't find a font directory"))))
   ((c-function void ("start_trigger_thread" int)) slices/tr)
   (gui '()))))
