import torch
from torchvision import transforms
import random
from PIL import ImageFilter
import torchvision.transforms.functional as F
import math
import torch.distributed as dist

class ThreeAugment:
    def __init__(self):
        self.grayscale = transforms.Grayscale(num_output_channels=3)
        self.solarize = transforms.Lambda(lambda x: transforms.functional.solarize(x, threshold=128))
        self.gaussian_blur = transforms.GaussianBlur(kernel_size=5)
        self.augmentations = [self.grayscale, self.solarize, self.gaussian_blur]

    def __call__(self, img):
        augmentation = random.choice(self.augmentations)
        img = augmentation(img)
        return img
    
class SimpleRandomCrop:
    def __init__(self, size):
        self.size = size

    def __call__(self, img):
        # Ensure the image is in RGB mode
        img = img.convert("RGB")
        w, h = img.size
        target_size = self.size

        # Resize the image so the smallest side matches the target size
        if w < h:
            new_w = target_size
            new_h = int(target_size * h / w)
        else:
            new_h = target_size
            new_w = int(target_size * w / h)
        img = F.resize(img, (new_h, new_w))

        # Apply reflect padding
        img = F.pad(img, (4, 4, 4, 4), padding_mode='reflect')

        # Randomly crop a square region of the target size
        w, h = img.size
        left = torch.randint(0, w - target_size + 1, (1,)).item()
        top = torch.randint(0, h - target_size + 1, (1,)).item()
        img = F.crop(img, top, left, target_size, target_size)
        return img
    

# From DeiT-III
class RASampler(torch.utils.data.Sampler):
    """Sampler that restricts data loading to a subset of the dataset for distributed,
    with repeated augmentation.
    It ensures that different each augmented version of a sample will be visible to a
    different process (GPU)
    Heavily based on torch.utils.data.DistributedSampler
    """

    def __init__(self, dataset, num_replicas=None, rank=None, shuffle=True, num_repeats: int = 3):
        if num_replicas is None:
            if not dist.is_available():
                raise RuntimeError("Requires distributed package to be available")
            num_replicas = dist.get_world_size()
        if rank is None:
            if not dist.is_available():
                raise RuntimeError("Requires distributed package to be available")
            rank = dist.get_rank()
        if num_repeats < 1:
            raise ValueError("num_repeats should be greater than 0")
        self.dataset = dataset
        self.num_replicas = num_replicas
        self.rank = rank
        self.num_repeats = num_repeats
        self.epoch = 0
        self.num_samples = int(math.ceil(len(self.dataset) * self.num_repeats / self.num_replicas))
        self.total_size = self.num_samples * self.num_replicas
        # self.num_selected_samples = int(math.ceil(len(self.dataset) / self.num_replicas))
        self.num_selected_samples = int(math.floor(len(self.dataset) // 256 * 256 / self.num_replicas))
        self.shuffle = shuffle

    def __iter__(self):
        if self.shuffle:
            # deterministically shuffle based on epoch
            g = torch.Generator()
            g.manual_seed(self.epoch)
            indices = torch.randperm(len(self.dataset), generator=g)
        else:
            indices = torch.arange(start=0, end=len(self.dataset))

        # add extra samples to make it evenly divisible
        indices = torch.repeat_interleave(indices, repeats=self.num_repeats, dim=0).tolist()
        padding_size: int = self.total_size - len(indices)
        if padding_size > 0:
            indices += indices[:padding_size]
        assert len(indices) == self.total_size

        # subsample
        indices = indices[self.rank:self.total_size:self.num_replicas]
        assert len(indices) == self.num_samples

        return iter(indices[:self.num_selected_samples])

    def __len__(self):
        return self.num_selected_samples

    def set_epoch(self, epoch):
        self.epoch = epoch