import argparse
import datetime
from inspect import trace
import json
import numpy as np
import os
import time
from pathlib import Path

import torch
import torch.backends.cudnn as cudnn
from torch.utils.tensorboard import SummaryWriter
import torchvision.transforms as transforms
import torchvision.datasets as datasets

import timm
# assert timm.__version__ == "0.3.2"  # version check
import timm.optim.optim_factory as optim_factory

import util.misc as misc
from util.misc import NativeScalerWithGradNormCount as NativeScaler

import model_mae

from engine_pretrain import train_one_epoch, evaluate, KD_one_epoch, train_CNN_one_epoch, evaluate
from util.pos_embed import interpolate_pos_embed
import baseline_models as bm
import distributed as dist

def get_args_parser():
    parser = argparse.ArgumentParser('MAE pre-training', add_help=False)
    parser.add_argument('--batch_size', default=64, type=int,
                        help='Batch size per GPU (effective batch size is batch_size * accum_iter * # gpus')
    parser.add_argument('--epochs', default=1, type=int)
    parser.add_argument('--accum_iter', default=1, type=int,
                        help='Accumulate gradient iterations (for increasing the effective batch size under memory constraints)')

    # Model parameters
    ##
    parser.add_argument('--model', default='mae_vit_base_p16_nodp', type=str, metavar='MODEL',
                        help='Name of model to train')
    ##
    parser.add_argument('--input_size', default=224, type=int,
                        help='images input size')

    parser.add_argument('--mask_ratio', default=0.75, type=float,
                        help='Masking ratio (percentage of removed patches).')

    parser.add_argument('--norm_pix_loss', action='store_true',
                        help='Use (per-patch) normalized pixels as targets for computing loss')
    parser.set_defaults(norm_pix_loss=False)

    # Optimizer parameters
    parser.add_argument('--weight_decay', type=float, default=0.05,
                        help='weight decay (default: 0.05)')

    parser.add_argument('--lr', type=float, default=None, metavar='LR',
                        help='learning rate (absolute lr)')
    parser.add_argument('--blr', type=float, default=1.5e-4, metavar='LR',
                        help='base learning rate: absolute_lr = base_lr * total_batch_size / 256')
    parser.add_argument('--min_lr', type=float, default=0., metavar='LR',
                        help='lower lr bound for cyclic schedulers that hit 0')

    parser.add_argument('--warmup_epochs', type=int, default=0, metavar='N',
                        help='epochs to warmup LR')

    # Dataset parameters
    parser.add_argument('--data_path', default='/datasets01/imagenet_full_size/061417/', type=str,
                        help='dataset path')

    ##
    parser.add_argument('--output_dir', default='./checkpoint/client',
                        help='path where to save, empty for no saving')
    ##                    
    parser.add_argument('--log_dir', default='./checkpoint/client',
                        help='path where to tensorboard log')
    parser.add_argument('--device', default='cuda',
                        help='device to use for training / testing')
    parser.add_argument('--seed', default=0, type=int)
    parser.add_argument('--resume', default='',
                        help='resume from checkpoint')

    parser.add_argument('--start_epoch', default=0, type=int, metavar='N',
                        help='start epoch')
    parser.add_argument('--num_workers', default=10, type=int)
    parser.add_argument('--pin_mem', action='store_true',
                        help='Pin CPU memory in DataLoader for more efficient (sometimes) transfer to GPU.')
    parser.add_argument('--no_pin_mem', action='store_false', dest='pin_mem')
    parser.set_defaults(pin_mem=True)

    # distributed training parameters
    parser.add_argument('--world_size', default=1, type=int,
                        help='number of distributed processes')
    parser.add_argument('--local_rank', default=-1, type=int)
    parser.add_argument('--dist_on_itp', action='store_true')
    parser.add_argument('--dist_url', default='env://',
                        help='url used to set up distributed training')


    # arguments from main.py
    parser.add_argument('-rd', '--rounds', default=200, type=int,
                        help='The number of rounds for federated learning')
    parser.add_argument('-tt', '--training_times', default=3, type=int,
                        help='The bounded number of times of training on each worker')
    parser.add_argument('-sb', '--stale_bound', default=5, type=int,
                        help='The bounded number of train round gap between worker stored local model and visitor model')
    parser.add_argument('-nw', '--num_total_workers', default=100, type=int,
                        help='The total number of workers in the network')
    parser.add_argument('-nm', '--num_models', default=1, type=int,
                        help='The number of autoencoders being trained') 
    parser.add_argument('-le', '--num_of_local_epochs', default=5, type=int,
                        help='The number of epochs in local training')
    parser.add_argument('-c', '--connectivity', default=0.15, type=float,
                        help='The connectivity of network')  
    parser.add_argument('-hcp', '--highest_cp', default=5, type=int,
                        help='The highest computing power for workers') 
    parser.add_argument('-bd', '--base_depth', default=1, type=int,  
                        help='The depth of base transformer')
    
    parser.add_argument('-sp', '--save_path', default='./checkpoint/', type=str, help='checkpoint save path')
    parser.add_argument('-gp', '--graph_path', default='./graph/network_G.adjlist', type=str, help='network graph path')
    parser.add_argument('-dsp', '--data_split_path', default='./graph/data_split.pkl', type=str, help='network data split path') 
    parser.add_argument('-cpp', '--computing_power_path', default='./graph/computing_power.pkl', type=str, help='network computing powers of workers path') 
    parser.add_argument('-swp', '--starting_workers_path', default='./graph/starting_workers.pkl', type=str, help='network starting workers path') 

    parser.add_argument('-d', '--dataset', default=4, type=int, 
                        help='The id of dataset to use: 0 - CIFAR10; 1 - CIFAR100; 2 - Food101; 3 - ImageNet; 4 - Mini-ImageNet; 5 - Road-Sign, 6 - Mini-INAT;')
    parser.add_argument('-ra', '--ratio', default=1, type=float,
                        help='The ratio of labelled images')
    parser.add_argument('-samp', '--sampling', default="iid", 
                        help='The way of samping, iid or dir')
    parser.add_argument('--alpha', default=1e-1, type=float, 
                        help='The required parameter for dir sampling, which decides the statistical heterogenity')
    parser.add_argument('-ri', '--record_interval', default=100, type=int, 
                        help='The interval of saving checkpoint')
    parser.add_argument('-m', '--mode', default=2, type=int, 
                        help='The mode of next worker finding algorithm: 0 - the beginning algorithm; 1 - random; 2 - new one')
    parser.add_argument('-agg', '--agg', default=3, type=int, 
                        help='The mode of aggregation: 0 - average weights; 1 - data volume weights; 2 - round weights; 3 - our weights')
    parser.add_argument('-bl', '--baseline', default=0, type=int, 
                        help='The baselines: 0 - our algorithm; 1 - FedMAE, 2 - GossipMAE, 3 - DecenCNN')

    # Model Size Study
    parser.add_argument('-sc', '--scenario', default=0, type=int, 
                        help='The scenarios: 0 - Centralized; 1 - Federated, 2 - Single Client')
    parser.add_argument('-dp', '--depths_range', default=10, type=int, 
                        help='The number of model size options')
    parser.add_argument('-scp', '--start_model_ckpt_path', default='./checkpoint_cen/', type=str, help='The save path of start model checkpoints (Used for ensuring same starting weights)')
    parser.add_argument('-logp', '--logs_save_path', default='./model_size_logs/', type=str, help='The save path of training logs')

    parser.add_argument('-p', '--phase', default="pretrain",  
                        help='specify the codes to: pretrain, finetune')
    parser.add_argument('-ftd', '--ft_depth', default=5, type=int,  
                        help='The depth of model in finetuning')


    return parser

def pretrain_MAE(main_args, worker_ID, dataset_train, depths, r_eps, 
                teacher_depths=None, teacher_load_path=None, 
                teacher_save_path=None, record_round=None, input_bs=None,
                dataset_val=None,
                load_path='./checkpoint/global/fed_checkpoint.pth', save_path="./checkpoint/global/fed_checkpoint.pth"):

    args = get_args_parser()
    args = args.parse_args()
    args.batch_size = torch.cuda.device_count() * args.batch_size

    if worker_ID:
        args.output_dir = '%sworker/%s' % (main_args.save_path, worker_ID)
        args.log_dir = '%sworker/%s' % (main_args.save_path, worker_ID)
    else:
        args.output_dir = '%sglobal/' % main_args.save_path
        args.log_dir = '%sglobal/' % main_args.save_path

    if not os.path.isdir(args.output_dir):
        os.makedirs(args.output_dir, exist_ok=True)

    misc.init_distributed_mode(args)

    print('job dir: {}'.format(os.path.dirname(os.path.realpath(__file__))))
    # rint("{}".format(args).replace(', ', ',\n'))
    print("Mask ratio: %s" % args.mask_ratio)

    device = torch.device(args.device)

    torch.cuda.empty_cache()

    cudnn.benchmark = True

    if True:  # args.distributed:
        num_tasks = misc.get_world_size()
        global_rank = misc.get_rank()
        if dataset_train:
            sampler_train = torch.utils.data.DistributedSampler(
                dataset_train, num_replicas=num_tasks, rank=global_rank, shuffle=True
            )
            print("Sampler_train = %s" % str(sampler_train))
        if dataset_val:
            sampler_val = torch.utils.data.DistributedSampler(
                dataset_val, num_replicas=num_tasks, rank=global_rank, shuffle=True
            )
            print("Sampler_val = %s" % str(sampler_val))
    else:
        if dataset_train:
            sampler_train = torch.utils.data.RandomSampler(dataset_train)

    if global_rank == 0 and args.log_dir is not None:
        os.makedirs(args.log_dir, exist_ok=True)
        log_writer = SummaryWriter(log_dir=args.log_dir)
    else:
        log_writer = None

    if input_bs == None:
        eff_batch_size = args.batch_size * args.accum_iter * misc.get_world_size()
    else:
        eff_batch_size = input_bs

    if dataset_train:
        data_loader_train = torch.utils.data.DataLoader(
            dataset_train, sampler=sampler_train,
            batch_size=eff_batch_size,
            num_workers=args.num_workers,
            pin_memory=args.pin_mem,
            drop_last=True,
        )
    

    if dataset_val:
        data_loader_val = torch.utils.data.DataLoader(
            dataset_val, sampler=sampler_val,
            batch_size=eff_batch_size,
            num_workers=args.num_workers,
            pin_memory=args.pin_mem,
            drop_last=True
        )

    # define the model
    if len(depths) == 2:
        model = model_mae.__dict__[args.model](depth=depths[0], decoder_depth=depths[1], norm_pix_loss=args.norm_pix_loss)
    else:
        model = model_mae.__dict__[args.model](depth=depths[0], decoder_depth=depths[1], embed_dim=depths[2], norm_pix_loss=args.norm_pix_loss)
    #model = bm.simsiam('res18-origin')
    if load_path and os.path.exists(load_path):
        ckpt_fed = torch.load(load_path, map_location='cpu')
        model.load_state_dict(ckpt_fed['model'], strict=False)
        print("Load checkpoint from: %s" % load_path) 

    if teacher_depths:
        model_T = model_mae.__dict__[args.model](depth=teacher_depths[0], decoder_depth=teacher_depths[1], norm_pix_loss=args.norm_pix_loss, student_model=model)
        if teacher_load_path and os.path.exists(teacher_load_path):
            ckpt_fed = torch.load(teacher_load_path, map_location='cpu')
            model_T.load_state_dict(ckpt_fed['model'], strict=False)
            print("Load teacher checkpoint from: %s" % teacher_load_path) 
        model_T.to(device)
        model_T_without_ddp = model_T
   

    model.to(device)
    model_without_ddp = model
    # print("Model = %s" % str(model_without_ddp))

    if teacher_depths:
        n_parameters = sum(p.numel() for p in model_T.parameters() if p.requires_grad)
    else:
        n_parameters = sum(p.numel() for p in model.parameters() if p.requires_grad)
    print('number of params (M): %.2f' % (n_parameters / 1.e6))

    
    if args.lr is None:  # only base_lr is specified
        args.lr = args.blr * eff_batch_size / 256
        # args.lr = args.blr

    print("base lr: %.2e" % (args.lr * 256 / eff_batch_size))
    print("actual lr: %.2e" % args.lr)

    print("accumulate grad iterations: %d" % args.accum_iter)
    print("effective batch size: %d" % eff_batch_size)

    if args.distributed:
        model = torch.nn.parallel.DistributedDataParallel(model, device_ids=[args.gpu], find_unused_parameters=True)
        model_without_ddp = model.module
        if teacher_depths:
            model_T = torch.nn.parallel.DistributedDataParallel(model_T, device_ids=[args.gpu], find_unused_parameters=True)
            model_T_without_ddp = model_T.module

    if torch.cuda.device_count() > 1:
        model = torch.nn.parallel.DataParallel(model)
        model_without_ddp = model.module
        if teacher_depths:
            model_T = torch.nn.parallel.DataParallel(model_T)
            model_T_without_ddp = model_T.module

    if teacher_depths:
        param_groups_T = optim_factory.add_weight_decay(model_T_without_ddp, args.weight_decay)
        optimizer = torch.optim.AdamW(param_groups_T, lr=args.lr, betas=(0.9, 0.95))
    else:
        # following timm: set wd as 0 for bias and norm layers
        param_groups = optim_factory.add_weight_decay(model_without_ddp, args.weight_decay)
        
        optimizer = torch.optim.AdamW(param_groups, lr=args.lr, betas=(0.9, 0.95))
        # print(optimizer)
    
    loss_scaler = NativeScaler() 

    if dataset_train:
        print(f"Start training for {r_eps} epochs")
        start_time = time.time()
        best_loss = float('inf')
        for epoch in range(r_eps):
            if args.distributed:
                data_loader_train.sampler.set_epoch(epoch)
            if not teacher_depths:   
                train_stats = train_one_epoch(
                    model, data_loader_train,
                    optimizer, device, epoch, loss_scaler,
                    log_writer=log_writer,
                    args=args
                )
            else:
                train_stats = KD_one_epoch(
                    model, model_T, data_loader_train,
                    optimizer, device, epoch, loss_scaler, 
                    log_writer=log_writer,
                    args=args
                )

            if args.output_dir:
                if train_stats['loss'] < best_loss:
                    best_loss = train_stats['loss']
                    misc.save_model_pretrain(
                        args=args, epoch=epoch, model=model_without_ddp, optimizer=optimizer, loss_scaler=loss_scaler, best=True)
                if epoch % 20 == 0 or epoch + 1 == args.epochs:
                    misc.save_model_pretrain(
                        args=args, epoch=epoch, model=model_without_ddp, optimizer=optimizer, loss_scaler=loss_scaler)

            log_stats = {**{f'train_{k}': v for k, v in train_stats.items()},
                            'epoch': epoch,}

            if args.output_dir and misc.is_main_process():
                if log_writer is not None:
                    log_writer.flush()
                with open(os.path.join(args.output_dir, "log.txt"), mode="a", encoding="utf-8") as f:
                    f.write(json.dumps(log_stats) + "\n")  

        if dataset_val:
            total_loss = evaluate(data_loader_val, model, device, args)
            print("Val_loss: %s" % total_loss)

        total_time = time.time() - start_time
        total_time_str = str(datetime.timedelta(seconds=int(total_time)))
        print('Training time {}'.format(total_time_str))

    new_checkpoint = {
        'model': model_without_ddp.state_dict(),
        'optimizer': optimizer.state_dict(),
        'loss_scaler': loss_scaler.state_dict(),
        #'lr': train_stats['lr']
    }

    if dataset_train:
        new_checkpoint['loss'] = train_stats['loss']

    if save_path:
        misc.save_on_master(new_checkpoint, save_path)
        if record_round:
            backup_path = '%sglobal/checkpoint_%srd.pth' % (main_args.save_path, record_round)
            misc.save_on_master(new_checkpoint, backup_path)
            print("Backup federated checkpoint at %s rounds" % record_round)
    
    if teacher_depths:
        new_checkpoint_T = {
            'model': model_T_without_ddp.state_dict(),
            'optimizer': optimizer.state_dict(),
            'loss_scaler': loss_scaler.state_dict(),
        }
        misc.save_on_master(new_checkpoint_T, teacher_save_path)

    if dataset_train:
        if dataset_val:
            return new_checkpoint, total_loss, n_parameters / 1.e6
        else:
            return new_checkpoint, train_stats['loss'], n_parameters / 1.e6
    else:
        return new_checkpoint

# def models_KD_agg(main_args, worker_ID, model_IDs, dataset_train, depths, r_eps, 
#                 ratios, record_round=None,
#                 load_path='./checkpoint/global/fed_checkpoint.pth', save_path="./checkpoint/global/fed_checkpoint.pth"):

#     args = get_args_parser()
#     args = args.parse_args()
#     args.batch_size = torch.cuda.device_count() * args.batch_size

#     if worker_ID:
#         args.output_dir = '%sworker/%s' % (main_args.save_path, worker_ID)
#         args.log_dir = '%sworker/%s' % (main_args.save_path, worker_ID)
#     else:
#         args.output_dir = '%sglobal/' % main_args.save_path
#         args.log_dir = '%sglobal/' % main_args.save_path

#     if not os.path.isdir(args.output_dir):
#         os.makedirs(args.output_dir, exist_ok=True)

#     misc.init_distributed_mode(args)

#     print('job dir: {}'.format(os.path.dirname(os.path.realpath(__file__))))
#     # rint("{}".format(args).replace(', ', ',\n'))
#     print("Mask ratio: %s" % args.mask_ratio)

#     device = torch.device(args.device)

#     cudnn.benchmark = True

#     if True:  # args.distributed:
#         num_tasks = misc.get_world_size()
#         global_rank = misc.get_rank()
#         if dataset_train:
#             sampler_train = torch.utils.data.DistributedSampler(
#                 dataset_train, num_replicas=num_tasks, rank=global_rank, shuffle=True
#             )
#             print("Sampler_train = %s" % str(sampler_train))
#     else:
#         if dataset_train:
#             sampler_train = torch.utils.data.RandomSampler(dataset_train)

#     if global_rank == 0 and args.log_dir is not None:
#         os.makedirs(args.log_dir, exist_ok=True)
#         log_writer = SummaryWriter(log_dir=args.log_dir)
#     else:
#         log_writer = None

#     if dataset_train:
#         data_loader_train = torch.utils.data.DataLoader(
#             dataset_train, sampler=sampler_train,
#             batch_size=args.batch_size,
#             num_workers=args.num_workers,
#             pin_memory=args.pin_mem,
#             drop_last=True,
#         )
    
#     # define the model
#     model_S = model_mae.__dict__[args.model](depth=depths[0], decoder_depth=depths[1], norm_pix_loss=args.norm_pix_loss)
#     if os.path.exists(load_path):
#         load_ckpt = torch.load(load_path, map_location='cpu')
#         model_S.load_state_dict(load_ckpt['model'], strict=False)
#         print("Load checkpoint from: %s" % load_path) 

#     model_S.to(device)
#     model_without_ddp = model_S
#     # print("Model = %s" % str(model_without_ddp))

#     model_T_dict = {}
#     for mid in model_IDs:
#         model_T_dict[mid] = model_mae.__dict__[args.model](depth=depths[0], decoder_depth=depths[1], norm_pix_loss=args.norm_pix_loss)
#         if mid == -1:
#             load_path = '%sworker/%s/local.pth' % (main_args.save_path, worker_ID)
#         else:
#             load_path = '%sglobal/model_%s.pth' % (main_args.save_path, mid)
#         load_ckpt = torch.load(load_path, map_location='cpu')
#         model_T_dict[mid].load_state_dict(load_ckpt['model'], strict=False)
#         print("Load checkpoint from: %s" % load_path) 
#         model_T_dict[mid].to(device)  

#     n_parameters = sum(p.numel() for p in model_S.parameters() if p.requires_grad)
#     print('number of params (M): %.2f' % (n_parameters / 1.e6))

#     eff_batch_size = args.batch_size * args.accum_iter * misc.get_world_size()
    
#     if args.lr is None:  # only base_lr is specified
#         args.lr = args.blr * eff_batch_size / 256
#         # args.lr = args.blr

#     print("base lr: %.2e" % (args.lr * 256 / eff_batch_size))
#     print("actual lr: %.2e" % args.lr)

#     print("accumulate grad iterations: %d" % args.accum_iter)
#     print("effective batch size: %d" % eff_batch_size)

#     if args.distributed:
#         model_S = torch.nn.parallel.DistributedDataParallel(model_S, device_ids=[args.gpu], find_unused_parameters=True)
#         model_without_ddp = model_S.module

#     if torch.cuda.device_count() > 1:
#         model_S = torch.nn.parallel.DataParallel(model_S)
#         model_without_ddp = model_S.module

#     # following timm: set wd as 0 for bias and norm layers
#     param_groups = optim_factory.add_weight_decay(model_without_ddp, args.weight_decay)
    
#     optimizer = torch.optim.AdamW(param_groups, lr=args.lr, betas=(0.9, 0.95))
#     # optimizer = torch.optim.SGD(model.parameters(), lr=args.lr, momentum=0.9, weight_decay=1e-4)
#     # print(optimizer)
#     loss_scaler = NativeScaler()

#     if dataset_train:
#         print(f"Start training for {r_eps} epochs")
#         start_time = time.time()
#         best_loss = float('inf')

#         for epoch in range(r_eps):
#             if args.distributed:
#                 data_loader_train.sampler.set_epoch(epoch)   
#             train_stats = KD_one_epoch(
#                 model_S, model_T_dict, data_loader_train,
#                 optimizer, device, epoch, loss_scaler, ratios, 
#                 log_writer=log_writer,
#                 args=args
#             )

#             if args.output_dir:
#                 if train_stats['loss'] < best_loss:
#                     best_loss = train_stats['loss']
#                     misc.save_model_pretrain(
#                         args=args, epoch=epoch, model=model_without_ddp, optimizer=optimizer, loss_scaler=loss_scaler, best=True)
#                 if epoch % 20 == 0 or epoch + 1 == args.epochs:
#                     misc.save_model_pretrain(
#                         args=args, epoch=epoch, model=model_without_ddp, optimizer=optimizer, loss_scaler=loss_scaler)

#             log_stats = {**{f'train_{k}': v for k, v in train_stats.items()},
#                             'epoch': epoch,}

#             if args.output_dir and misc.is_main_process():
#                 if log_writer is not None:
#                     log_writer.flush()
#                 with open(os.path.join(args.output_dir, "log.txt"), mode="a", encoding="utf-8") as f:
#                     f.write(json.dumps(log_stats) + "\n")  

#         total_time = time.time() - start_time
#         total_time_str = str(datetime.timedelta(seconds=int(total_time)))
#         print('Training time {}'.format(total_time_str))


#     new_checkpoint = {
#         'model': model_without_ddp.state_dict(),
#         'optimizer': optimizer.state_dict(),
#         'loss_scaler': loss_scaler.state_dict(),
#         #'lr': train_stats['lr']
#     }

#     if dataset_train:
#         new_checkpoint['loss'] = train_stats['loss']

#     if save_path:
#         misc.save_on_master(new_checkpoint, save_path)
#         if record_round:
#             backup_path = '%s/global/KD_checkpoint_%srd.pth' % (main_args.save_path, record_round)
#             misc.save_on_master(new_checkpoint, backup_path)
#             print("Backup KD checkpoint at %s rounds" % record_round)

#     return new_checkpoint
    
def pretrain_CNN(main_args, worker_ID, dataset_train, depth, r_eps, 
                num_classes, record_round=None, 
                load_path='./checkpoint/global/fed_checkpoint.pth', save_path="./checkpoint/global/fed_checkpoint.pth"):

    args = get_args_parser()
    args = args.parse_args()
    args.batch_size = torch.cuda.device_count() * args.batch_size

    if worker_ID:
        args.output_dir = '%sworker/%s' % (main_args.save_path, worker_ID)
        args.log_dir = '%sworker/%s' % (main_args.save_path, worker_ID)
    else:
        args.output_dir = '%sglobal/' % main_args.save_path
        args.log_dir = '%sglobal/' % main_args.save_path

    if not os.path.isdir(args.output_dir):
        os.makedirs(args.output_dir, exist_ok=True)

    misc.init_distributed_mode(args)

    print('job dir: {}'.format(os.path.dirname(os.path.realpath(__file__))))
    # rint("{}".format(args).replace(', ', ',\n'))
    print("Mask ratio: %s" % args.mask_ratio)

    device = torch.device(args.device)

    torch.cuda.empty_cache()

    cudnn.benchmark = True

    if True:  # args.distributed:
        num_tasks = misc.get_world_size()
        global_rank = misc.get_rank()
        if dataset_train:
            sampler_train = torch.utils.data.DistributedSampler(
                dataset_train, num_replicas=num_tasks, rank=global_rank, shuffle=True
            )
            print("Sampler_train = %s" % str(sampler_train))
    else:
        if dataset_train:
            sampler_train = torch.utils.data.RandomSampler(dataset_train)

    if global_rank == 0 and args.log_dir is not None:
        os.makedirs(args.log_dir, exist_ok=True)
        log_writer = SummaryWriter(log_dir=args.log_dir)
    else:
        log_writer = None

    if dataset_train:
        data_loader_train = torch.utils.data.DataLoader(
            dataset_train, sampler=sampler_train,
            batch_size=args.batch_size,
            num_workers=args.num_workers,
            pin_memory=args.pin_mem,
            drop_last=True,
        )
    
    # define the model
    # model = bm.simsiam('res%s-origin' % depth)
    model = bm.create_backbone('res%s-origin' % depth, num_classes=num_classes)

    if os.path.exists(load_path):
        ckpt_fed = torch.load(load_path, map_location='cpu')
        model.load_state_dict(ckpt_fed['model'], strict=False)
        print("Load checkpoint from: %s" % load_path)    

    model.to(device)
    model_without_ddp = model
    # print("Model = %s" % str(model_without_ddp))

    n_parameters = sum(p.numel() for p in model.parameters() if p.requires_grad)
    print('number of params (M): %.2f' % (n_parameters / 1.e6))

    eff_batch_size = args.batch_size * args.accum_iter * misc.get_world_size()
    
    if args.lr is None:  # only base_lr is specified
        args.lr = args.blr * eff_batch_size / 256
        # args.lr = args.blr

    print("base lr: %.2e" % (args.lr * 256 / eff_batch_size))
    print("actual lr: %.2e" % args.lr)

    print("accumulate grad iterations: %d" % args.accum_iter)
    print("effective batch size: %d" % eff_batch_size)

    if args.distributed:
        model = torch.nn.parallel.DistributedDataParallel(model, device_ids=[args.gpu], find_unused_parameters=True)
        model_without_ddp = model.module

    if torch.cuda.device_count() > 1:
        model = torch.nn.parallel.DataParallel(model)
        model_without_ddp = model.module


    optimizer = torch.optim.SGD(model_without_ddp.parameters(), lr=args.lr, momentum=0.9, weight_decay=1e-4)

    # print(optimizer)
    loss_scaler = NativeScaler()

    if dataset_train:
        print(f"Start training for {r_eps} epochs")
        start_time = time.time()
        best_loss = float('inf')
        
        for epoch in range(r_eps):
            if args.distributed:
                data_loader_train.sampler.set_epoch(epoch)  

            train_stats = train_CNN_one_epoch(
                model, data_loader_train,
                optimizer, device, epoch, loss_scaler,
                log_writer=log_writer,
                args=args
            )

            if args.output_dir:
                if train_stats['loss'] < best_loss:
                    best_loss = train_stats['loss']
                    misc.save_model_pretrain(
                        args=args, epoch=epoch, model=model_without_ddp, optimizer=optimizer, loss_scaler=loss_scaler, best=True)
                if epoch % 20 == 0 or epoch + 1 == args.epochs:
                    misc.save_model_pretrain(
                        args=args, epoch=epoch, model=model_without_ddp, optimizer=optimizer, loss_scaler=loss_scaler)

            log_stats = {**{f'train_{k}': v for k, v in train_stats.items()},
                            'epoch': epoch,}

            if args.output_dir and misc.is_main_process():
                if log_writer is not None:
                    log_writer.flush()
                with open(os.path.join(args.output_dir, "log.txt"), mode="a", encoding="utf-8") as f:
                    f.write(json.dumps(log_stats) + "\n")  

        total_time = time.time() - start_time
        total_time_str = str(datetime.timedelta(seconds=int(total_time)))
        print('Training time {}'.format(total_time_str))

    new_checkpoint = {
        'model': model_without_ddp.state_dict(),
        'optimizer': optimizer.state_dict(),
        'loss_scaler': loss_scaler.state_dict(),
        #'lr': train_stats['lr']
    }

    if dataset_train:
        new_checkpoint['loss'] = train_stats['loss']

    if save_path:
        misc.save_on_master(new_checkpoint, save_path)
        if record_round:
            backup_path = '%sglobal/fed_checkpoint_%srd.pth' % (main_args.save_path, record_round)
            misc.save_on_master(new_checkpoint, backup_path)
            print("Backup federated checkpoint at %s rounds" % record_round)

    if dataset_train:
        return new_checkpoint, train_stats['loss'], n_parameters / 1.e6
    else:
        return new_checkpoint
