"""
Taken from https://github.com/automl/HPOlib1.5/tree/development


                    GNU GENERAL PUBLIC LICENSE
                       Version 3, 29 June 2007

 Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>
 Everyone is permitted to copy and distribute verbatim copies
 of this license document, but changing it is not allowed.

                            Preamble

  The GNU General Public License is a free, copyleft license for
software and other kinds of works.

  The licenses for most software and other practical works are designed
to take away your freedom to share and change the works.  By contrast,
the GNU General Public License is intended to guarantee your freedom to
share and change all versions of a program--to make sure it remains free
software for all its users.  We, the Free Software Foundation, use the
GNU General Public License for most of our software; it applies also to
any other work released this way by its authors.  You can apply it to
your programs, too.

  When we speak of free software, we are referring to freedom, not
price.  Our General Public Licenses are designed to make sure that you
have the freedom to distribute copies of free software (and charge for
them if you wish), that you receive source code or can get it if you
want it, that you can change the software or use pieces of it in new
free programs, and that you know you can do these things.

  To protect your rights, we need to prevent others from denying you
these rights or asking you to surrender the rights.  Therefore, you have
certain responsibilities if you distribute copies of the software, or if
you modify it: responsibilities to respect the freedom of others.

  For example, if you distribute copies of such a program, whether
gratis or for a fee, you must pass on to the recipients the same
freedoms that you received.  You must make sure that they, too, receive
or can get the source code.  And you must show them these terms so they
know their rights.

  Developers that use the GNU GPL protect your rights with two steps:
(1) assert copyright on the software, and (2) offer you this License
giving you legal permission to copy, distribute and/or modify it.

  For the developers' and authors' protection, the GPL clearly explains
that there is no warranty for this free software.  For both users' and
authors' sake, the GPL requires that modified versions be marked as
changed, so that their problems will not be attributed erroneously to
authors of previous versions.

  Some devices are designed to deny users access to install or run
modified versions of the software inside them, although the manufacturer
can do so.  This is fundamentally incompatible with the aim of
protecting users' freedom to change the software.  The systematic
pattern of such abuse occurs in the area of products for individuals to
use, which is precisely where it is most unacceptable.  Therefore, we
have designed this version of the GPL to prohibit the practice for those
products.  If such problems arise substantially in other domains, we
stand ready to extend this provision to those domains in future versions
of the GPL, as needed to protect the freedom of users.

  Finally, every program is threatened constantly by software patents.
States should not allow patents to restrict development and use of
software on general-purpose computers, but in those that do, we wish to
avoid the special danger that patents applied to a free program could
make it effectively proprietary.  To prevent this, the GPL assures that
patents cannot be used to render the program non-free.

  The precise terms and conditions for copying, distribution and
modification follow.

                       TERMS AND CONDITIONS

  0. Definitions.

  "This License" refers to version 3 of the GNU General Public License.

  "Copyright" also means copyright-like laws that apply to other kinds of
works, such as semiconductor masks.

  "The Program" refers to any copyrightable work licensed under this
License.  Each licensee is addressed as "you".  "Licensees" and
"recipients" may be individuals or organizations.

  To "modify" a work means to copy from or adapt all or part of the work
in a fashion requiring copyright permission, other than the making of an
exact copy.  The resulting work is called a "modified version" of the
earlier work or a work "based on" the earlier work.

  A "covered work" means either the unmodified Program or a work based
on the Program.

  To "propagate" a work means to do anything with it that, without
permission, would make you directly or secondarily liable for
infringement under applicable copyright law, except executing it on a
computer or modifying a private copy.  Propagation includes copying,
distribution (with or without modification), making available to the
public, and in some countries other activities as well.

  To "convey" a work means any kind of propagation that enables other
parties to make or receive copies.  Mere interaction with a user through
a computer network, with no transfer of a copy, is not conveying.

  An interactive user interface displays "Appropriate Legal Notices"
to the extent that it includes a convenient and prominently visible
feature that (1) displays an appropriate copyright notice, and (2)
tells the user that there is no warranty for the work (except to the
extent that warranties are provided), that licensees may convey the
work under this License, and how to view a copy of this License.  If
the interface presents a list of user commands or options, such as a
menu, a prominent item in the list meets this criterion.

  1. Source Code.

  The "source code" for a work means the preferred form of the work
for making modifications to it.  "Object code" means any non-source
form of a work.

  A "Standard Interface" means an interface that either is an official
standard defined by a recognized standards body, or, in the case of
interfaces specified for a particular programming language, one that
is widely used among developers working in that language.

  The "System Libraries" of an executable work include anything, other
than the work as a whole, that (a) is included in the normal form of
packaging a Major Component, but which is not part of that Major
Component, and (b) serves only to enable use of the work with that
Major Component, or to implement a Standard Interface for which an
implementation is available to the public in source code form.  A
"Major Component", in this context, means a major essential component
(kernel, window system, and so on) of the specific operating system
(if any) on which the executable work runs, or a compiler used to
produce the work, or an object code interpreter used to run it.

  The "Corresponding Source" for a work in object code form means all
the source code needed to generate, install, and (for an executable
work) run the object code and to modify the work, including scripts to
control those activities.  However, it does not include the work's
System Libraries, or general-purpose tools or generally available free
programs which are used unmodified in performing those activities but
which are not part of the work.  For example, Corresponding Source
includes interface definition files associated with source files for
the work, and the source code for shared libraries and dynamically
linked subprograms that the work is specifically designed to require,
such as by intimate data communication or control flow between those
subprograms and other parts of the work.

  The Corresponding Source need not include anything that users
can regenerate automatically from other parts of the Corresponding
Source.

  The Corresponding Source for a work in source code form is that
same work.

  2. Basic Permissions.

  All rights granted under this License are granted for the term of
copyright on the Program, and are irrevocable provided the stated
conditions are met.  This License explicitly affirms your unlimited
permission to run the unmodified Program.  The output from running a
covered work is covered by this License only if the output, given its
content, constitutes a covered work.  This License acknowledges your
rights of fair use or other equivalent, as provided by copyright law.

  You may make, run and propagate covered works that you do not
convey, without conditions so long as your license otherwise remains
in force.  You may convey covered works to others for the sole purpose
of having them make modifications exclusively for you, or provide you
with facilities for running those works, provided that you comply with
the terms of this License in conveying all material for which you do
not control copyright.  Those thus making or running the covered works
for you must do so exclusively on your behalf, under your direction
and control, on terms that prohibit them from making any copies of
your copyrighted material outside their relationship with you.

  Conveying under any other circumstances is permitted solely under
the conditions stated below.  Sublicensing is not allowed; section 10
makes it unnecessary.

  3. Protecting Users' Legal Rights From Anti-Circumvention Law.

  No covered work shall be deemed part of an effective technological
measure under any applicable law fulfilling obligations under article
11 of the WIPO copyright treaty adopted on 20 December 1996, or
similar laws prohibiting or restricting circumvention of such
measures.

  When you convey a covered work, you waive any legal power to forbid
circumvention of technological measures to the extent such circumvention
is effected by exercising rights under this License with respect to
the covered work, and you disclaim any intention to limit operation or
modification of the work as a means of enforcing, against the work's
users, your or third parties' legal rights to forbid circumvention of
technological measures.

  4. Conveying Verbatim Copies.

  You may convey verbatim copies of the Program's source code as you
receive it, in any medium, provided that you conspicuously and
appropriately publish on each copy an appropriate copyright notice;
keep intact all notices stating that this License and any
non-permissive terms added in accord with section 7 apply to the code;
keep intact all notices of the absence of any warranty; and give all
recipients a copy of this License along with the Program.

  You may charge any price or no price for each copy that you convey,
and you may offer support or warranty protection for a fee.

  5. Conveying Modified Source Versions.

  You may convey a work based on the Program, or the modifications to
produce it from the Program, in the form of source code under the
terms of section 4, provided that you also meet all of these conditions:

    a) The work must carry prominent notices stating that you modified
    it, and giving a relevant date.

    b) The work must carry prominent notices stating that it is
    released under this License and any conditions added under section
    7.  This requirement modifies the requirement in section 4 to
    "keep intact all notices".

    c) You must license the entire work, as a whole, under this
    License to anyone who comes into possession of a copy.  This
    License will therefore apply, along with any applicable section 7
    additional terms, to the whole of the work, and all its parts,
    regardless of how they are packaged.  This License gives no
    permission to license the work in any other way, but it does not
    invalidate such permission if you have separately received it.

    d) If the work has interactive user interfaces, each must display
    Appropriate Legal Notices; however, if the Program has interactive
    interfaces that do not display Appropriate Legal Notices, your
    work need not make them do so.

  A compilation of a covered work with other separate and independent
works, which are not by their nature extensions of the covered work,
and which are not combined with it such as to form a larger program,
in or on a volume of a storage or distribution medium, is called an
"aggregate" if the compilation and its resulting copyright are not
used to limit the access or legal rights of the compilation's users
beyond what the individual works permit.  Inclusion of a covered work
in an aggregate does not cause this License to apply to the other
parts of the aggregate.

  6. Conveying Non-Source Forms.

  You may convey a covered work in object code form under the terms
of sections 4 and 5, provided that you also convey the
machine-readable Corresponding Source under the terms of this License,
in one of these ways:

    a) Convey the object code in, or embodied in, a physical product
    (including a physical distribution medium), accompanied by the
    Corresponding Source fixed on a durable physical medium
    customarily used for software interchange.

    b) Convey the object code in, or embodied in, a physical product
    (including a physical distribution medium), accompanied by a
    written offer, valid for at least three years and valid for as
    long as you offer spare parts or customer support for that product
    model, to give anyone who possesses the object code either (1) a
    copy of the Corresponding Source for all the software in the
    product that is covered by this License, on a durable physical
    medium customarily used for software interchange, for a price no
    more than your reasonable cost of physically performing this
    conveying of source, or (2) access to copy the
    Corresponding Source from a network server at no charge.

    c) Convey individual copies of the object code with a copy of the
    written offer to provide the Corresponding Source.  This
    alternative is allowed only occasionally and noncommercially, and
    only if you received the object code with such an offer, in accord
    with subsection 6b.

    d) Convey the object code by offering access from a designated
    place (gratis or for a charge), and offer equivalent access to the
    Corresponding Source in the same way through the same place at no
    further charge.  You need not require recipients to copy the
    Corresponding Source along with the object code.  If the place to
    copy the object code is a network server, the Corresponding Source
    may be on a different server (operated by you or a third party)
    that supports equivalent copying facilities, provided you maintain
    clear directions next to the object code saying where to find the
    Corresponding Source.  Regardless of what server hosts the
    Corresponding Source, you remain obligated to ensure that it is
    available for as long as needed to satisfy these requirements.

    e) Convey the object code using peer-to-peer transmission, provided
    you inform other peers where the object code and Corresponding
    Source of the work are being offered to the general public at no
    charge under subsection 6d.

  A separable portion of the object code, whose source code is excluded
from the Corresponding Source as a System Library, need not be
included in conveying the object code work.

  A "User Product" is either (1) a "consumer product", which means any
tangible personal property which is normally used for personal, family,
or household purposes, or (2) anything designed or sold for incorporation
into a dwelling.  In determining whether a product is a consumer product,
doubtful cases shall be resolved in favor of coverage.  For a particular
product received by a particular user, "normally used" refers to a
typical or common use of that class of product, regardless of the status
of the particular user or of the way in which the particular user
actually uses, or expects or is expected to use, the product.  A product
is a consumer product regardless of whether the product has substantial
commercial, industrial or non-consumer uses, unless such uses represent
the only significant mode of use of the product.

  "Installation Information" for a User Product means any methods,
procedures, authorization keys, or other information required to install
and execute modified versions of a covered work in that User Product from
a modified version of its Corresponding Source.  The information must
suffice to ensure that the continued functioning of the modified object
code is in no case prevented or interfered with solely because
modification has been made.

  If you convey an object code work under this section in, or with, or
specifically for use in, a User Product, and the conveying occurs as
part of a transaction in which the right of possession and use of the
User Product is transferred to the recipient in perpetuity or for a
fixed term (regardless of how the transaction is characterized), the
Corresponding Source conveyed under this section must be accompanied
by the Installation Information.  But this requirement does not apply
if neither you nor any third party retains the ability to install
modified object code on the User Product (for example, the work has
been installed in ROM).

  The requirement to provide Installation Information does not include a
requirement to continue to provide support service, warranty, or updates
for a work that has been modified or installed by the recipient, or for
the User Product in which it has been modified or installed.  Access to a
network may be denied when the modification itself materially and
adversely affects the operation of the network or violates the rules and
protocols for communication across the network.

  Corresponding Source conveyed, and Installation Information provided,
in accord with this section must be in a format that is publicly
documented (and with an implementation available to the public in
source code form), and must require no special password or key for
unpacking, reading or copying.

  7. Additional Terms.

  "Additional permissions" are terms that supplement the terms of this
License by making exceptions from one or more of its conditions.
Additional permissions that are applicable to the entire Program shall
be treated as though they were included in this License, to the extent
that they are valid under applicable law.  If additional permissions
apply only to part of the Program, that part may be used separately
under those permissions, but the entire Program remains governed by
this License without regard to the additional permissions.

  When you convey a copy of a covered work, you may at your option
remove any additional permissions from that copy, or from any part of
it.  (Additional permissions may be written to require their own
removal in certain cases when you modify the work.)  You may place
additional permissions on material, added by you to a covered work,
for which you have or can give appropriate copyright permission.

  Notwithstanding any other provision of this License, for material you
add to a covered work, you may (if authorized by the copyright holders of
that material) supplement the terms of this License with terms:

    a) Disclaiming warranty or limiting liability differently from the
    terms of sections 15 and 16 of this License; or

    b) Requiring preservation of specified reasonable legal notices or
    author attributions in that material or in the Appropriate Legal
    Notices displayed by works containing it; or

    c) Prohibiting misrepresentation of the origin of that material, or
    requiring that modified versions of such material be marked in
    reasonable ways as different from the original version; or

    d) Limiting the use for publicity purposes of names of licensors or
    authors of the material; or

    e) Declining to grant rights under trademark law for use of some
    trade names, trademarks, or service marks; or

    f) Requiring indemnification of licensors and authors of that
    material by anyone who conveys the material (or modified versions of
    it) with contractual assumptions of liability to the recipient, for
    any liability that these contractual assumptions directly impose on
    those licensors and authors.

  All other non-permissive additional terms are considered "further
restrictions" within the meaning of section 10.  If the Program as you
received it, or any part of it, contains a notice stating that it is
governed by this License along with a term that is a further
restriction, you may remove that term.  If a license document contains
a further restriction but permits relicensing or conveying under this
License, you may add to a covered work material governed by the terms
of that license document, provided that the further restriction does
not survive such relicensing or conveying.

  If you add terms to a covered work in accord with this section, you
must place, in the relevant source files, a statement of the
additional terms that apply to those files, or a notice indicating
where to find the applicable terms.

  Additional terms, permissive or non-permissive, may be stated in the
form of a separately written license, or stated as exceptions;
the above requirements apply either way.

  8. Termination.

  You may not propagate or modify a covered work except as expressly
provided under this License.  Any attempt otherwise to propagate or
modify it is void, and will automatically terminate your rights under
this License (including any patent licenses granted under the third
paragraph of section 11).

  However, if you cease all violation of this License, then your
license from a particular copyright holder is reinstated (a)
provisionally, unless and until the copyright holder explicitly and
finally terminates your license, and (b) permanently, if the copyright
holder fails to notify you of the violation by some reasonable means
prior to 60 days after the cessation.

  Moreover, your license from a particular copyright holder is
reinstated permanently if the copyright holder notifies you of the
violation by some reasonable means, this is the first time you have
received notice of violation of this License (for any work) from that
copyright holder, and you cure the violation prior to 30 days after
your receipt of the notice.

  Termination of your rights under this section does not terminate the
licenses of parties who have received copies or rights from you under
this License.  If your rights have been terminated and not permanently
reinstated, you do not qualify to receive new licenses for the same
material under section 10.

  9. Acceptance Not Required for Having Copies.

  You are not required to accept this License in order to receive or
run a copy of the Program.  Ancillary propagation of a covered work
occurring solely as a consequence of using peer-to-peer transmission
to receive a copy likewise does not require acceptance.  However,
nothing other than this License grants you permission to propagate or
modify any covered work.  These actions infringe copyright if you do
not accept this License.  Therefore, by modifying or propagating a
covered work, you indicate your acceptance of this License to do so.

  10. Automatic Licensing of Downstream Recipients.

  Each time you convey a covered work, the recipient automatically
receives a license from the original licensors, to run, modify and
propagate that work, subject to this License.  You are not responsible
for enforcing compliance by third parties with this License.

  An "entity transaction" is a transaction transferring control of an
organization, or substantially all assets of one, or subdividing an
organization, or merging organizations.  If propagation of a covered
work results from an entity transaction, each party to that
transaction who receives a copy of the work also receives whatever
licenses to the work the party's predecessor in interest had or could
give under the previous paragraph, plus a right to possession of the
Corresponding Source of the work from the predecessor in interest, if
the predecessor has it or can get it with reasonable efforts.

  You may not impose any further restrictions on the exercise of the
rights granted or affirmed under this License.  For example, you may
not impose a license fee, royalty, or other charge for exercise of
rights granted under this License, and you may not initiate litigation
(including a cross-claim or counterclaim in a lawsuit) alleging that
any patent claim is infringed by making, using, selling, offering for
sale, or importing the Program or any portion of it.

  11. Patents.

  A "contributor" is a copyright holder who authorizes use under this
License of the Program or a work on which the Program is based.  The
work thus licensed is called the contributor's "contributor version".

  A contributor's "essential patent claims" are all patent claims
owned or controlled by the contributor, whether already acquired or
hereafter acquired, that would be infringed by some manner, permitted
by this License, of making, using, or selling its contributor version,
but do not include claims that would be infringed only as a
consequence of further modification of the contributor version.  For
purposes of this definition, "control" includes the right to grant
patent sublicenses in a manner consistent with the requirements of
this License.

  Each contributor grants you a non-exclusive, worldwide, royalty-free
patent license under the contributor's essential patent claims, to
make, use, sell, offer for sale, import and otherwise run, modify and
propagate the contents of its contributor version.

  In the following three paragraphs, a "patent license" is any express
agreement or commitment, however denominated, not to enforce a patent
(such as an express permission to practice a patent or covenant not to
sue for patent infringement).  To "grant" such a patent license to a
party means to make such an agreement or commitment not to enforce a
patent against the party.

  If you convey a covered work, knowingly relying on a patent license,
and the Corresponding Source of the work is not available for anyone
to copy, free of charge and under the terms of this License, through a
publicly available network server or other readily accessible means,
then you must either (1) cause the Corresponding Source to be so
available, or (2) arrange to deprive yourself of the benefit of the
patent license for this particular work, or (3) arrange, in a manner
consistent with the requirements of this License, to extend the patent
license to downstream recipients.  "Knowingly relying" means you have
actual knowledge that, but for the patent license, your conveying the
covered work in a country, or your recipient's use of the covered work
in a country, would infringe one or more identifiable patents in that
country that you have reason to believe are valid.

  If, pursuant to or in connection with a single transaction or
arrangement, you convey, or propagate by procuring conveyance of, a
covered work, and grant a patent license to some of the parties
receiving the covered work authorizing them to use, propagate, modify
or convey a specific copy of the covered work, then the patent license
you grant is automatically extended to all recipients of the covered
work and works based on it.

  A patent license is "discriminatory" if it does not include within
the scope of its coverage, prohibits the exercise of, or is
conditioned on the non-exercise of one or more of the rights that are
specifically granted under this License.  You may not convey a covered
work if you are a party to an arrangement with a third party that is
in the business of distributing software, under which you make payment
to the third party based on the extent of your activity of conveying
the work, and under which the third party grants, to any of the
parties who would receive the covered work from you, a discriminatory
patent license (a) in connection with copies of the covered work
conveyed by you (or copies made from those copies), or (b) primarily
for and in connection with specific products or compilations that
contain the covered work, unless you entered into that arrangement,
or that patent license was granted, prior to 28 March 2007.

  Nothing in this License shall be construed as excluding or limiting
any implied license or other defenses to infringement that may
otherwise be available to you under applicable patent law.

  12. No Surrender of Others' Freedom.

  If conditions are imposed on you (whether by court order, agreement or
otherwise) that contradict the conditions of this License, they do not
excuse you from the conditions of this License.  If you cannot convey a
covered work so as to satisfy simultaneously your obligations under this
License and any other pertinent obligations, then as a consequence you may
not convey it at all.  For example, if you agree to terms that obligate you
to collect a royalty for further conveying from those to whom you convey
the Program, the only way you could satisfy both those terms and this
License would be to refrain entirely from conveying the Program.

  13. Use with the GNU Affero General Public License.

  Notwithstanding any other provision of this License, you have
permission to link or combine any covered work with a work licensed
under version 3 of the GNU Affero General Public License into a single
combined work, and to convey the resulting work.  The terms of this
License will continue to apply to the part which is the covered work,
but the special requirements of the GNU Affero General Public License,
section 13, concerning interaction through a network will apply to the
combination as such.

  14. Revised Versions of this License.

  The Free Software Foundation may publish revised and/or new versions of
the GNU General Public License from time to time.  Such new versions will
be similar in spirit to the present version, but may differ in detail to
address new problems or concerns.

  Each version is given a distinguishing version number.  If the
Program specifies that a certain numbered version of the GNU General
Public License "or any later version" applies to it, you have the
option of following the terms and conditions either of that numbered
version or of any later version published by the Free Software
Foundation.  If the Program does not specify a version number of the
GNU General Public License, you may choose any version ever published
by the Free Software Foundation.

  If the Program specifies that a proxy can decide which future
versions of the GNU General Public License can be used, that proxy's
public statement of acceptance of a version permanently authorizes you
to choose that version for the Program.

  Later license versions may give you additional or different
permissions.  However, no additional obligations are imposed on any
author or copyright holder as a result of your choosing to follow a
later version.

  15. Disclaimer of Warranty.

  THERE IS NO WARRANTY FOR THE PROGRAM, TO THE EXTENT PERMITTED BY
APPLICABLE LAW.  EXCEPT WHEN OTHERWISE STATED IN WRITING THE COPYRIGHT
HOLDERS AND/OR OTHER PARTIES PROVIDE THE PROGRAM "AS IS" WITHOUT WARRANTY
OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO,
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
PURPOSE.  THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE PROGRAM
IS WITH YOU.  SHOULD THE PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF
ALL NECESSARY SERVICING, REPAIR OR CORRECTION.

  16. Limitation of Liability.

  IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING
WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MODIFIES AND/OR CONVEYS
THE PROGRAM AS PERMITTED ABOVE, BE LIABLE TO YOU FOR DAMAGES, INCLUDING ANY
GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE
USE OR INABILITY TO USE THE PROGRAM (INCLUDING BUT NOT LIMITED TO LOSS OF
DATA OR DATA BEING RENDERED INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD
PARTIES OR A FAILURE OF THE PROGRAM TO OPERATE WITH ANY OTHER PROGRAMS),
EVEN IF SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF
SUCH DAMAGES.

  17. Interpretation of Sections 15 and 16.

  If the disclaimer of warranty and limitation of liability provided
above cannot be given local legal effect according to their terms,
reviewing courts shall apply local law that most closely approximates
an absolute waiver of all civil liability in connection with the
Program, unless a warranty or assumption of liability accompanies a
copy of the Program in return for a fee.

                     END OF TERMS AND CONDITIONS

            How to Apply These Terms to Your New Programs

  If you develop a new program, and you want it to be of the greatest
possible use to the public, the best way to achieve this is to make it
free software which everyone can redistribute and change under these terms.

  To do so, attach the following notices to the program.  It is safest
to attach them to the start of each source file to most effectively
state the exclusion of warranty; and each file should have at least
the "copyright" line and a pointer to where the full notice is found.

    <one line to give the program's name and a brief idea of what it does.>
    Copyright (C) <year>  <name of author>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

Also add information on how to contact you by electronic and paper mail.

  If the program does terminal interaction, make it output a short
notice like this when it starts in an interactive mode:

    <program>  Copyright (C) <year>  <name of author>
    This program comes with ABSOLUTELY NO WARRANTY; for details type `show w'.
    This is free software, and you are welcome to redistribute it
    under certain conditions; type `show c' for details.

The hypothetical commands `show w' and `show c' should show the appropriate
parts of the General Public License.  Of course, your program's commands
might be different; for a GUI interface, you would use an "about box".

  You should also get your employer (if you work as a programmer) or school,
if any, to sign a "copyright disclaimer" for the program, if necessary.
For more information on this, and how to apply and follow the GNU GPL, see
<http://www.gnu.org/licenses/>.

  The GNU General Public License does not permit incorporating your program
into proprietary programs.  If your program is a subroutine library, you
may consider it more useful to permit linking proprietary applications with
the library.  If this is what you want to do, use the GNU Lesser General
Public License instead of this License.  But first, please read
<http://www.gnu.org/philosophy/why-not-lgpl.html>.
"""
import abc
import copy
import csv
import gzip
import logging
import os
import pickle

from urllib.request import urlretrieve

import ConfigSpace.util
import lockfile
import numpy as np
import scipy.stats
import sklearn.ensemble
import sklearn.metrics
import sklearn.model_selection
import sklearn.pipeline
import sklearn.preprocessing
import sklearn.utils.validation

from ConfigSpace import CategoricalHyperparameter
from ConfigSpace import Configuration
from ConfigSpace import ConfigurationSpace
from ConfigSpace import EqualsCondition
from ConfigSpace import UniformFloatHyperparameter
from ConfigSpace import UniformIntegerHyperparameter


__version__ = 0.2


# Differences to the paper are in comments at the end of the line

_expected_amount_of_data = {
    "GLMNET": {
        3: 15546,  # Paper says that there are 15547 entries, but one of them has a negative runtime
        31: 15528,  # Paper says that there are 15547 entries, but 19 of them have a negative runtime
        37: 15488,  # Paper says that there are 15546 entries, but 58 of them have a negative runtime
        44: 15527,  # Paper says that there are 15547 entries, but 20 of them have a negative runtime
        50: 15545,  # Paper says that there are 15547 entries, but 2 of them have a negative runtime
        151: 15547,
        312: 6613,
        333: 15441,  # Paper says that there are 15547 entries, but 105 of them have a negative runtime
        334: 15486,  # Paper says that there are 15547 entries, but 61 of them have a negative runtime
        335: 15516,  # Paper says that there are 15547 entries, but 31 of them have a negative runtime
        1036: 14937,
        1038: 15547,
        1043: 6365,  # Paper says that there are 6466 entries, but 101 of them have a negative runtime
        1046: 15462,
        1049: 7396,  # Paper says that there are 7423 entries, but 27 of them have a negative runtime
        1050: 15521,  # Paper says that there are 15521 entries, but 26 of them have a negative runtime
        1063: 15518,  # Paper says that there are 15518 entries, but 29 of them have a negative runtime
        1067: 15523,  # Paper says that there are 15518 entries, but 23 of them have a negative runtime
        1068: 15546,
        1120: 15531,
        1176: 13005,  # Paper does not mention this dataset
        1461: 6970,
        1462: 8955,
        1464: 15531,  # Paper says that there are 15518 entries, but 16 of them have a negative runtime
        1467: 15387,  # Paper says that there are 15518 entries, but 160 of them have a negative runtime
        1471: 15479,  # Paper says that there are 15518 entries, but 68 of them have a negative runtime
        1479: 15546,
        1480: 15000,  # Paper says that there are 15518 entries, but 24 of them have a negative runtime
        1485: 8247,
        1486: 3866,
        1487: 15543,  # Paper says that there are 15518 entries, but 4 of them have a negative runtime
        1489: 15522,  # Paper says that there are 15518 entries, but 25 of them have a negative runtime
        1494: 15515,  # Paper says that there are 15518 entries, but 15 of them have a negative runtime
        1504: 15527,  # Paper says that there are 15518 entries, but 20 of them have a negative runtime
        1510: 15406,  # Paper says that there are 15518 entries, but 141 of them have a negative runtime
        1570: 15452,  # Paper says that there are 15518 entries, but 94 of them have a negative runtime
        4134: 1493,
        4534: 2801,
    },
    "RPART": {
        3: 14624,  # Paper says that there are 14633 entries, but 9 of them have a negative runtime
        31: 14624,  # Paper says that there are 14633 entries, but 9 of them have a negative runtime
        37: 14598,  # Paper says that there are 14633 entries, but 35 of them have a negative runtime
        44: 14633,
        50: 14618,  # Paper says that there are 14633 entries, but 15 of them have a negative runtime
        151: 14632,
        312: 13455,
        333: 14585,  # Paper says that there are 14633 entries, but 47 of them have a negative runtime
        334: 14580,  # Paper says that there are 14633 entries, but 53 of them have a negative runtime
        335: 14625,  # Paper says that there are 14633 entries, but 8 of them have a negative runtime
        1036: 14633,
        1038: 5151,
        1043: 14633,
        1046: 14624,  # Paper says that there are 14633 entries, but 8 of them have a negative runtime
        1049: 14549,  # Paper says that there are 14633 entries, but 83 of them have a negative runtime
        1050: 14497,  # Paper says that there are 14633 entries, but 136 of them have a negative runtime
        1063: 14497,  # Paper says that there are 14633 entries, but 136 of them have a negative runtime
        1067: 14632,
        1068: 14633,
        1120: 7477,
        1176: 14632,  # Paper does not mention this dataset
        1461: 14073,
        1462: 14536,  # Paper says that there are 14633 entries, but 97 of them have a negative runtime
        1464: 14609,  # Paper says that there are 14633 entries, but 23 of them have a negative runtime
        1467: 14626,  # Paper says that there are 14633 entries, but 7 of them have a negative runtime
        1471: 14616,  # Paper says that there are 14633 entries, but 17 of them have a negative runtime
        1479: 14633,
        1480: 14576,  # Paper says that there are 14633 entries, but 57 of them have a negative runtime
        1485: 10923,
        1486: 11389,
        1487: 6005,
        1489: 14628,  # Paper says that there are 14633 entries, but 5 of them have a negative runtime
        1494: 14632,
        1504: 14629,  # Paper says that there are 14633 entries, but 4 of them have a negative runtime
        1510: 14561,  # Paper says that there are 14633 entries, but 72 of them have a negative runtime
        1570: 14515,  # Paper says that there are 14633 entries, but 117 of them have a negative runtime
        4134: 3947,
        4534: 3231,
    },
    "SVM": {
        3: 19644,
        31: 19644,
        37: 15985,
        44: 19644,
        50: 19644,
        151: 2384,
        312: 18740,
        333: 19634,  # Paper says that there are 19644 entries, but 10 of them have a negative runtime
        334: 19629,  # Paper says that there are 19644 entries, but 15 of them have a negative runtime
        335: 15123,
        1036: 2338,
        1038: 5716,
        1043: 10121,
        1046: 5422,
        1049: 12064,
        1050: 19644,
        1063: 19644,
        1067: 10229,
        1068: 13893,
        1120: 3908,
        1176: 14451,  # Paper does not mention this dataset
        1461: 2678,
        1462: 6320,
        1464: 19644,
        1467: 4441,
        1471: 9725,
        1479: 19644,
        1480: 19644,
        1485: 10334,
        1486: 1490,
        1487: 19644,
        1489: 17298,
        1494: 19644,
        1504: 19644,
        1510: 19644,
        1570: 19644,
        4134: 560,
        4534: 2476,
    },
    "Ranger": {
        3: 15135,  # Paper says 15139, but for 4 instances min.node.size > 1
        31: 14965,  # Paper says 15139, but for 158 instances min.node.size > 1 and for 16 instances mtry > 1
        37: 15060,  # Paper says 15139, but for 79 instances min.node.size > 1
        44: 15129,  # Paper says 15139, but for 79 instances min.node.size > 1
        50: 13357,  # Paper says 15139, but for 219 instances min.node.size > 1 and for 219 instances mtry > 1563
        151: 12381,  # Paper says 12517, but for 136 instances mtry > 1
        312: 12937,  # Paper says 12985, but for 48 instances min.node.size > 1
        333: 15066,  # Paper says 15139, but for 73 instances min.node.size > 1
        334: 14441,  # Paper says 14492, but for 51 instances min.node.size > 1
        335: 14295,  # Paper says 15139, but for 299 instances min.node.size > 1 and for 219 instances mtry > 545
        1036: 7394,  # Paper says 15139, but for 3 instances min.node.size > 1
        1038: 4827,
        1043: 3788,
        1046: 8838,  # Paper says 8842, but for 4 instances min.node.size > 1
        1049: 14819,  # Paper says 15139, but for 320 instances min.node.size > 1
        1050: 11328,  # Paper says 11357, but for 29 instances min.node.size > 1
        1063: 7883,  # Paper says 7914, but for 29 instances min.node.size > 1
        1067: 7364,  # Paper says 7386, but for 22 instances min.node.size > 1
        1068: 8135,  # Paper says 8173, but for 38 instances min.node.size > 1
        1120: 9760,
        1176: 15117,  # Paper does not mention this dataset
        1461: 14279,  # Paper says 14323, but for 44 instances mtry > 1
        1462: 15103,  # Paper says 15139, but for 36 instances min.node.size > 1
        1464: 15034,  # Paper says 15139, but for 105 instances min.node.size > 1
        1467: 14896,  # Paper says 15139, but for 243 instances min.node.size > 1
        1471: 13522,  # Paper says 13523, but for 1 instances min.node.size > 1
        1479: 15092,  # Paper says 15140, but for 48 instances min.node.size > 1
        1480: 15074,  # Paper says 15139, but for 65 instances min.node.size > 1
        1485: 15015,  # Paper says 15139, but for 124 instances min.node.size > 1
        1486: 15139,
        1487: 15108,  # Paper says 15139, but for 31 instances min.node.size > 1
        1489: 15137,  # Paper says 15139, but for 2 instances min.node.size > 1
        1494: 14807,  # Paper says 15139, but for 332 instances min.node.size > 1
        1504: 14938,  # Paper says 15140, but for 202 instances min.node.size > 1
        1510: 15071,  # Paper says 15139, but for 68 instances min.node.size > 1
        1570: 15136,  # Paper says 15139, but for 3 instances min.node.size > 1
        4134: 14472,  # Paper says 14516, but for 44 instances min.node.size > 1
        4534: 15129,  # Paper says 14516, but for 10 instances min.node.size > 1
    },
    "XGBoost": {
        3: 16867,
        31: 16867,
        37: 16866,
        44: 16867,
        50: 16866,
        151: 16272,  # should be 16866, but eta, colsample_by_tree and colsample_by_level sometimes missing
        312: 15886,
        333: 16865,  # should be 16867, but 2 samples have negative runtime
        334: 16866,  # should be 16867, but 1 sample has a negative runtime
        335: 10002,
        1036: 2581,
        1038: 1370,
        1043: 16867,
        1046: 11812,
        1049: 4453,
        1050: 13758,
        1063: 16865,  # should be 16866, but 1 sample has a negative runtime
        1067: 16866,
        1068: 16866,
        1120: 8143,
        1176: 13047,  # Paper does not mention this dataset
        1461: 2215,
        1462: 16859,  # should be 16867, but 8 sample have a negative runtime
        1464: 16865,  # should be 16867, but 2 sample have a negative runtime
        1467: 16865,  # should be 16866, but 1 sample has a negative runtime
        1471: 16866,
        1479: 16867,
        1480: 16254,
        1485: 9237,
        1486: 5813,
        1487: 11194,
        1489: 16867,
        1494: 16867,
        1504: 16867,
        1510: 16867,
        1570: 16866,  # should be 16867, but 1 sample has a negative runtime
        4134: 2222,
        4534: 947,
    },
}


_DATA_DIR = os.path.expanduser("~/.hpolib/")


def get_rng(rng=None, self_rng=None):
    """ helper function to obtain RandomState.
    returns RandomState created from rng
    if rng then return RandomState created from rng
    if rng is None returns self_rng
    if self_rng and rng is None return random RandomState

    :param rng: int or RandomState
    :param self_rng: RandomState
    :return: RandomState
    """

    if rng is not None:
        return create_rng(rng)
    elif rng is None and self_rng is not None:
        return create_rng(self_rng)
    else:
        return np.random.RandomState()


def create_rng(rng):
    """ helper to create rng from RandomState or int
    :param rng: int or RandomState
    :return: RandomState
    """
    if rng is None:
        return np.random.RandomState()
    elif type(rng) == np.random.RandomState:
        return rng
    elif int(rng) == rng:
        # As seed is sometimes -1 (e.g. if SMAC optimizes a
        # deterministic function
        rng = np.abs(rng)
        return np.random.RandomState(rng)
    else:
        raise ValueError(
            "%s is neither a number nor a RandomState. " "Initializing RandomState failed"
        )


class AbstractBenchmark(object, metaclass=abc.ABCMeta):
    def __init__(self, rng=None):
        """Interface for benchmarks.

        A benchmark contains of two building blocks, the target function and
        the configuration space. Furthermore it can contain additional
        benchmark-specific information such as the location and the function
        value of the global optima. New benchmarks should be derived from
        this base class or one of its child classes.
        """

        self.rng = create_rng(rng)
        self.configuration_space = self.get_configuration_space()

    @abc.abstractmethod
    def objective_function(self, configuration, **kwargs):
        """Objective function.

        Override this function to provide your benchmark function. This
        function will be called by one of the evaluate functions. For
        flexibility you have to return a dictionary with the only mandatory
        key being `function_value`, the objective function value for the
        configuration which was passed. By convention, all benchmarks are
        minimization problems.

        Parameters
        ----------
        configuration : dict-like

        Returns
        -------
        dict
            Must contain at least the key `function_value`.
        """
        pass

    @abc.abstractmethod
    def objective_function_test(self, configuration, **kwargs):
        """
        If there is a different objective function for offline testing, e.g
        testing a machine learning on a hold extra test set instead
        on a validation set override this function here.

        Parameters
        ----------
        configuration : dict-like

        Returns
        -------
        dict
            Must contain at least the key `function_value`.
        """
        pass

    def _check_configuration(foo):
        """ Decorator to enable checking the input configuration

            Uses the check_configuration of the ConfigSpace class to ensure
            that all specified values are valid, and no conditionals are violated

            Can be combined with the _configuration_as_array decorator.
        """

        def wrapper(self, configuration, **kwargs):
            if not isinstance(configuration, ConfigSpace.Configuration):
                try:
                    squirtle = {
                        k: configuration[i]
                        for (i, k) in enumerate(self.configuration_space)
                    }
                    wartortle = ConfigSpace.Configuration(
                        self.configuration_space, squirtle
                    )
                except Exception as e:
                    raise Exception(
                        "Error during the conversion of the provided "
                        "into a ConfigSpace.Configuration object"
                    ) from e
            else:
                wartortle = configuration
            self.configuration_space.check_configuration(wartortle)
            return foo(self, configuration, **kwargs)

        return wrapper

    def _configuration_as_array(foo, data_type=np.float):
        """ Decorator to allow the first input argument to 'objective_function' to be an array.

            For all continuous benchmarks it is often required that the input to the benchmark
            can be a (NumPy) array. By adding this to the objective function, both inputs types,
            ConfigSpace.Configuration and array, are possible.

            Can be combined with the _check_configuration decorator.
        """

        def wrapper(self, configuration, **kwargs):
            if isinstance(configuration, ConfigSpace.Configuration):
                blastoise = np.array(
                    [configuration[k] for k in configuration], dtype=data_type
                )

            else:
                blastoise = configuration
            return foo(self, blastoise, **kwargs)

        return wrapper

    def __call__(self, configuration, **kwargs):
        """ Provides interface to use, e.g., SciPy optimizers """
        return self.objective_function(configuration, **kwargs)["function_value"]

    def test(self, n_runs=5, *args, **kwargs):
        """ Draws some random configuration and call objective_fucntion(_test).

        Parameters
        ----------
        n_runs : int
            number of random configurations to draw and evaluate
        """
        train_rvals = []
        test_rvals = []

        for _ in range(n_runs):
            configuration = self.configuration_space.sample_configuration()
            train_rvals.append(self.objective_function(configuration, *args, **kwargs))
            test_rvals.append(
                self.objective_function_test(configuration, *args, **kwargs)
            )

        return train_rvals, test_rvals

    @staticmethod
    @abc.abstractmethod
    def get_configuration_space():
        """ Defines the configuration space for each benchmark.

        Returns
        -------
        ConfigSpace.ConfigurationSpace
            A valid configuration space for the benchmark's parameters
        """
        raise NotImplementedError()

    @staticmethod
    @abc.abstractmethod
    def get_meta_information():
        """ Provides some meta information about the benchmark.

        Returns
        -------
        dict
            some human-readable information

        """
        raise NotImplementedError()


class ExploringOpenML(AbstractBenchmark):
    """Surrogate benchmarks based on the data from Automatic Exploration of Machine Learning
    Benchmarks on OpenML by Kühn et al..

    This is a base class that should not be used directly. Instead, use one of the automatically
    constructed classes at the bottom of the file. This benchmark does not contain the KKNN
    algorithm as it only allows for 30 different hyperparameter settings.

    Data is obtained from:
    https://figshare.com/articles/OpenML_R_Bot_Benchmark_Data_final_subset_/5882230
    """

    url = None

    def __init__(
        self, dataset_id, n_splits=10, n_iterations=30, rebuild=False, rng=None, n_jobs=1
    ):
        """

        Parameters
        ----------
        dataset_id: int
            Dataset Id as given in Table 2 of the paper by Kühn et al..
        n_splits : int
            Number of cross-validation splits for optimizing the surrogate hyperparameters.
        n_iterations : int
            Number of iterations of random search to construct a surrogate model
        rebuild : bool
            Whether to construct a new surrogate model if there is already one stored to disk.
            This is important because changing the ``n_splits`` and the ``n_iterations``
            arguments do not trigger a rebuild of the surrogate.
        rng: int/None/RandomState
            set up rng
        """

        super().__init__(rng=rng)
        self.logger = logging.getLogger(self.__class__.__name__)
        self.dataset_id = dataset_id
        self.classifier = self.__class__.__name__.split("_")[0]
        self.n_jobs = n_jobs

        surrogate_dir = os.path.join(_DATA_DIR, "ExploringOpenML", "surrogates")
        if not os.path.exists(surrogate_dir):
            os.makedirs(surrogate_dir, exist_ok=True)
        surrogate_file_name = os.path.join(
            surrogate_dir, "surrogate_%s_%d.pkl.gz" % (self.classifier, self.dataset_id),
        )
        while True:
            try:
                if rebuild or not os.path.exists(surrogate_file_name):
                    with lockfile.LockFile(surrogate_file_name, timeout=10):
                        self.construct_surrogate(dataset_id, n_splits, n_iterations)
                        with gzip.open(surrogate_file_name, "wb") as fh:
                            pickle.dump(
                                (
                                    self.regressor_loss,
                                    self.regressor_runtime,
                                    self.configurations,
                                    self.features,
                                    self.targets,
                                    self.runtimes,
                                ),
                                fh,
                            )
                    break
                else:
                    try:
                        with gzip.open(surrogate_file_name, "rb") as fh:
                            (
                                self.regressor_loss,
                                self.regressor_runtime,
                                self.configurations,
                                self.features,
                                self.targets,
                                self.runtimes,
                            ) = pickle.load(fh)
                        break
                    except:
                        with lockfile.LockFile(surrogate_file_name, timeout=10):
                            with gzip.open(surrogate_file_name, "rb") as fh:
                                (
                                    self.regressor_loss,
                                    self.regressor_runtime,
                                    self.configurations,
                                    self.features,
                                    self.targets,
                                    self.runtimes,
                                ) = pickle.load(fh)
                        break
            except lockfile.LockTimeout:
                self.logger.debug(
                    "Could not obtain file lock for %s", surrogate_file_name
                )

    def construct_surrogate(self, dataset_id, n_splits, n_iterations_rs):
        self.logger.info("Could not find surrogate pickle, constructing the surrogate.")

        save_to = os.path.join(_DATA_DIR, "ExploringOpenML")
        if not os.path.isdir(save_to):
            os.makedirs(save_to)
        csv_path = os.path.join(save_to, self.classifier + ".csv")
        if not os.path.exists(csv_path):
            self.logger.info(
                "Could not find surrogate data, downloading from %s", self.url
            )
            urlretrieve(self.url, csv_path)
            self.logger.info("Finished downloading surrogate data.")

        evaluations = []
        line_no = []

        self.logger.info("Starting to read in surrogate data.")
        with open(csv_path) as fh:
            csv_reader = csv.DictReader(fh)
            for i, line in enumerate(csv_reader):
                if int(line["data_id"]) != dataset_id:
                    continue
                evaluations.append(line)
                line_no.append(i)
        hyperparameters_names = [
            hp.name for hp in self.configuration_space.get_hyperparameters()
        ]
        categorical_features = [
            i
            for i in range(len(self.configuration_space.get_hyperparameters()))
            if isinstance(
                self.configuration_space.get_hyperparameters()[i],
                CategoricalHyperparameter,
            )
        ]
        target_features = "auc"
        configurations = []
        features = []
        targets = []
        runtimes = []
        for i, evaluation in enumerate(evaluations):
            number_of_features = float(evaluation["NumberOfFeatures"])
            number_of_datapoints = float(evaluation["NumberOfInstances"])
            config = {
                key: value
                for key, value in evaluation.items()
                if key in hyperparameters_names and value != "NA"
            }
            # Do some specific transformations
            if self.classifier == "Ranger":
                config["mtry"] = float(config["mtry"]) / number_of_features
                config["min.node.size"] = np.log(float(config["min.node.size"])) / np.log(
                    number_of_datapoints
                )
                if config["min.node.size"] > 1.0:
                    # MF: according to Philipp it is unclear why this is in
                    # the data
                    continue
                if config["mtry"] > 1.0:
                    # MF: according to Philipp it is unclear why this is in
                    # the data
                    continue
            elif self.classifier == "XGBoost":
                if "eta" not in config:
                    # MF: according to Philipp, the algorithm was run in the
                    # default and the OpenML R package did not upload the
                    # default in one of the earliest versions
                    continue
                elif "colsample_bytree" not in config and config["booster"] == "gbtree":
                    # MF: according to Philipp, the algorithm was run in the
                    # default and the OpenML R package did not upload the
                    # default in one of the earliest versions
                    continue
                elif "colsample_bylevel" not in config and config["booster"] == "gbtree":
                    # MF: according to Philipp, the algorithm was run in the
                    # default and the OpenML R package did not upload the
                    # default in one of the earliest versions
                    continue
            if float(evaluation[target_features]) > 1:
                raise ValueError(i, evaluation)
            # For unknown reasons, the runtimes can be negative...
            if float(evaluation["runtime"]) < 0:
                continue
            config = ConfigSpace.util.fix_types(
                configuration=config, configuration_space=self.configuration_space,
            )
            try:
                config = ConfigSpace.util.deactivate_inactive_hyperparameters(
                    configuration_space=self.configuration_space, configuration=config,
                )
            except ValueError as e:
                print(line_no[i], config, evaluation)
                raise e
            self.configuration_space.check_configuration(config)
            array = config.get_array()
            features.append(array)
            configurations.append(config)
            # HPOlib is about minimization!
            targets.append(1 - float(evaluation[target_features]))
            runtimes.append(float(evaluation["runtime"]))

        features = np.array(features)
        targets = np.array(targets) + 1e-10
        runtimes = np.array(runtimes) + 1e-10
        features = self.impute(features)
        self.logger.info("Finished reading in surrogate data.")

        if len(configurations) != _expected_amount_of_data[self.classifier][dataset_id]:
            raise ValueError(
                "Expected %d configurations for classifier %s on dataset %d, but found only %d!"
                % (
                    _expected_amount_of_data[self.classifier][dataset_id],
                    self.classifier,
                    self.dataset_id,
                    len(configurations),
                )
            )

        self.configurations = configurations
        self.features = features
        self.targets = targets
        self.runtimes = runtimes

        self.logger.info("Start building the surrogate, this can take a few minutes...")
        cv = sklearn.model_selection.KFold(
            n_splits=n_splits, random_state=1, shuffle=True
        )
        cs = ConfigurationSpace()
        min_samples_split = UniformIntegerHyperparameter(
            "min_samples_split", lower=2, upper=20, log=True,
        )
        min_samples_leaf = UniformIntegerHyperparameter(
            "min_samples_leaf", 1, 20, log=True
        )
        max_features = UniformFloatHyperparameter("max_features", 0.5, 1.0)
        bootstrap = CategoricalHyperparameter("bootstrap", [True, False])
        cs.add_hyperparameters(
            [min_samples_split, min_samples_leaf, max_features, bootstrap,]
        )
        # This makes HPO deterministic
        cs.seed(1)
        highest_correlations_loss = -np.inf
        highest_correlations_loss_by_fold = np.array((n_splits,)) * -np.inf
        highest_correlations_runtime = -np.inf
        highest_correlations_runtime_by_fold = np.array((n_splits,)) * -np.inf
        best_config_loss = cs.get_default_configuration()
        best_config_runtime = cs.get_default_configuration()
        for n_iterations in range(n_iterations_rs):
            self.logger.debug(
                "Random search iteration %d/%d.", n_iterations, n_iterations_rs
            )
            check_loss = True
            new_config_loss = cs.sample_configuration()
            new_config_runtime = copy.deepcopy(new_config_loss)
            regressor_loss = self.get_unfitted_regressor(
                new_config_loss, categorical_features, 25,
            )
            regressor_runtime = self.get_unfitted_regressor(
                new_config_runtime, categorical_features, 25,
            )

            rank_correlations_loss = np.ones((n_splits,)) * -np.NaN
            rank_correlations_runtime = np.ones((n_splits,)) * -np.NaN
            for n_fold, (train_idx, test_idx) in enumerate(cv.split(features, targets)):
                train_features = features[train_idx]
                train_targets_loss = targets[train_idx]
                train_targets_runtime = runtimes[train_idx]

                regressor_loss.fit(train_features, np.log(train_targets_loss))
                regressor_runtime.fit(train_features, np.log(train_targets_runtime))

                test_features = features[test_idx]

                y_hat_loss = np.exp(regressor_loss.predict(test_features))
                y_hat_runtime = np.exp(regressor_runtime.predict(test_features))

                test_targets_loss = targets[test_idx]
                spearman_rank_loss = scipy.stats.spearmanr(test_targets_loss, y_hat_loss)[
                    0
                ]
                rank_correlations_loss[n_fold] = spearman_rank_loss

                test_targets_runtime = runtimes[test_idx]
                spearman_rank_runtime = scipy.stats.spearmanr(
                    test_targets_runtime, y_hat_runtime,
                )[0]
                rank_correlations_runtime[n_fold] = spearman_rank_runtime

                if (
                    (
                        np.nanmean(highest_correlations_loss) * 0.99
                        > np.nanmean(rank_correlations_loss)
                    )
                    and (
                        (
                            np.nanmean(highest_correlations_loss_by_fold[: n_splits + 1])
                            * (0.99 + n_fold * 0.001)
                        )
                        > np.nanmean(rank_correlations_loss[: n_splits + 1])
                    )
                    and (
                        np.nanmean(highest_correlations_runtime) * 0.99
                        > np.nanmean(rank_correlations_runtime)
                    )
                    and (
                        (
                            np.nanmean(
                                highest_correlations_runtime_by_fold[: n_splits + 1]
                            )
                            * (0.99 + n_fold * 0.001)
                        )
                        > np.nanmean(rank_correlations_runtime[: n_splits + 1])
                    )
                ):
                    check_loss = False
                    break

            if check_loss and np.mean(rank_correlations_loss) > highest_correlations_loss:
                highest_correlations_loss = np.mean(rank_correlations_loss)
                highest_correlations_loss_by_fold = rank_correlations_loss
                best_config_loss = new_config_loss
            if (
                check_loss
                and np.mean(rank_correlations_runtime) > highest_correlations_runtime
            ):
                highest_correlations_runtime = np.mean(rank_correlations_runtime)
                highest_correlations_runtime_by_fold = rank_correlations_runtime
                best_config_runtime = new_config_runtime

        regressor_loss = self.get_unfitted_regressor(
            best_config_loss, categorical_features, 500
        )
        regressor_loss.fit(
            X=features, y=np.log(targets),
        )
        regressor_runtime = self.get_unfitted_regressor(
            best_config_runtime, categorical_features, 500,
        )
        regressor_runtime.fit(X=features, y=np.log(runtimes))
        self.logger.info("Finished building the surrogate.")

        self.regressor_loss = regressor_loss
        self.regressor_runtime = regressor_runtime

    def get_unfitted_regressor(self, config, categorical_features, n_trees):
        return sklearn.pipeline.Pipeline(
            [
                (
                    "ohe",
                    sklearn.preprocessing.OneHotEncoder(
                        categorical_features=categorical_features, sparse=False,
                    ),
                ),
                (
                    "poly",
                    sklearn.preprocessing.PolynomialFeatures(
                        degree=2, interaction_only=True, include_bias=False,
                    ),
                ),
                (
                    "estimator",
                    sklearn.ensemble.RandomForestRegressor(
                        n_estimators=n_trees,
                        n_jobs=self.n_jobs,
                        random_state=1,
                        **config.get_dictionary(),
                    ),
                ),
            ]
        )

    def impute(self, features):
        features = features.copy()
        for i, hp in enumerate(self.configuration_space.get_hyperparameters()):
            nan_rows = ~np.isfinite(features[:, i])
            features[nan_rows, i] = -1
        return features

    @AbstractBenchmark._check_configuration
    def objective_function(self, x, **kwargs):
        x = x.get_array().reshape((1, -1))
        x = self.impute(x)
        y = self.regressor_loss.predict(x)
        y = y[0]
        runtime = self.regressor_runtime.predict(x)
        runtime = runtime[0]
        # Untransform and round to the resolution of the data file.
        y = np.round(np.exp(y) - 1e-10, 6)
        runtime = np.round(np.exp(runtime) - 1e-10, 6)
        return {"function_value": y, "cost": runtime}

    @AbstractBenchmark._check_configuration
    def objective_function_test(self, x, **kwargs):
        raise NotImplementedError()

    @staticmethod
    def get_meta_information():
        return {
            "num_function_evals": 50,
            "name": "Exploring_OpenML",
            "references": [
                """@article{kuhn_arxiv2018a,
    title = {Automatic {Exploration} of {Machine} {Learning} {Experiments} on {OpenML}},
    journal = {arXiv:1806.10961 [cs, stat]},
    author = {Daniel Kühn and Philipp Probst and Janek Thomas and Bernd Bischl},
    year = {2018},
    }""",
                """@inproceedings{eggensperger_aaai2015a,
   author = {Katharina Eggensperger and Frank Hutter and Holger H. Hoos and Kevin Leyton-Brown},
   title = {Efficient Benchmarking of Hyperparameter Optimizers via Surrogates},
   booktitle = {Proceedings of the Twenty-Ninth AAAI Conference on Artificial Intelligence},
   conference = {AAAI Conference},
   year = {2015},
}

    """,
            ],
        }

    def get_empirical_f_opt(self):
        """Return the empirical f_opt.

        Because ``get_meta_information`` is a static function it has no access to the actual
        function values predicted by the surrogate. This helper function gives access.

        Returns
        -------
        Configuration
        """
        ms = []
        for t in self.regressor_loss.steps[-1][-1].estimators_:
            ms.append(np.min(t.tree_.value))
        return np.exp(np.mean(ms))

    def get_empirical_f_max(self):
        """Return the empirical f_max.

        This is the configuration resulting in the worst predictive performance. Necessary to
        compute the average distance to the minimum metric typically used by Wistuba,
        Schilling and Schmidt-Thieme.

        Returns
        -------
        Configuration
        """
        ms = []
        for t in self.regressor_loss.steps[-1][-1].estimators_:
            ms.append(np.max(t.tree_.value))
        return np.exp(np.mean(ms))

    def get_rs_difficulty(self, diff, n_evals, seed=None):
        rng = sklearn.utils.validation.check_random_state(seed)

        # Use the (optimistic) best value possible, but 0.5 as worst possible because an AUC of
        # 0.5 is random and we don't really care about anything below 0.5.
        f_opt = self.get_empirical_f_opt()
        f_max = 0.5
        difference = f_max - f_opt
        difference = difference if difference != 0 else 1
        cs = self.get_configuration_space()
        cs.seed(rng.randint(100000))
        configurations = cs.sample_configuration(n_evals * 100)
        scores = np.array(
            [
                self.objective_function(config)["function_value"]
                for config in configurations
            ]
        )

        # Compute a bootstrapped score
        lower = 0
        for i in range(100000):
            subsample = rng.choice(scores, size=n_evals, replace=False)
            min_score = np.min(subsample)
            rescaled_min_score = (min_score - f_opt) / difference
            if rescaled_min_score < diff:
                lower += 1
        return lower / 100000


class GLMNET(ExploringOpenML):
    url = "https://ndownloader.figshare.com/files/10462300"

    @staticmethod
    def get_configuration_space():
        cs = ConfigurationSpace()
        alpha = UniformFloatHyperparameter("alpha", lower=0, upper=1)
        lambda_ = UniformFloatHyperparameter(
            "lambda", lower=2 ** -10, upper=2 ** 10, log=True,
        )
        cs.add_hyperparameters([alpha, lambda_])
        return cs


class RPART(ExploringOpenML):
    url = "https://ndownloader.figshare.com/files/10462309"

    @staticmethod
    def get_configuration_space():
        cs = ConfigurationSpace()
        alpha = UniformFloatHyperparameter("cp", lower=0, upper=1)
        maxdepth = UniformIntegerHyperparameter("maxdepth", lower=1, upper=30,)
        minbucket = UniformIntegerHyperparameter("minbucket", lower=1, upper=60,)
        minsplit = UniformIntegerHyperparameter("minsplit", lower=1, upper=60,)
        cs.add_hyperparameters([alpha, maxdepth, minbucket, minsplit])
        return cs


class SVM(ExploringOpenML):
    url = "https://ndownloader.figshare.com/files/10462312"

    @staticmethod
    def get_configuration_space():
        cs = ConfigurationSpace()
        kernel = CategoricalHyperparameter(
            "kernel", choices=["linear", "polynomial", "radial"],
        )
        cost = UniformFloatHyperparameter("cost", 2 ** -10, 2 ** 10, log=True)
        gamma = UniformFloatHyperparameter("gamma", 2 ** -10, 2 ** 10, log=True)
        degree = UniformIntegerHyperparameter("degree", 2, 5)
        cs.add_hyperparameters([kernel, cost, gamma, degree])
        gamma_condition = EqualsCondition(gamma, kernel, "radial")
        degree_condition = EqualsCondition(degree, kernel, "polynomial")
        cs.add_conditions([gamma_condition, degree_condition])
        return cs


class Ranger(ExploringOpenML):

    url = "https://ndownloader.figshare.com/files/10462306"

    @staticmethod
    def get_configuration_space():
        cs = ConfigurationSpace()
        num_trees = UniformIntegerHyperparameter("num.trees", lower=1, upper=2000,)
        replace = CategoricalHyperparameter("replace", choices=["FALSE", "TRUE"],)
        sample_fraction = UniformFloatHyperparameter("sample.fraction", lower=0, upper=1,)
        mtry = UniformFloatHyperparameter("mtry", lower=0, upper=1,)
        respect_unordered_factors = CategoricalHyperparameter(
            "respect.unordered.factors", choices=["FALSE", "TRUE"],
        )
        min_node_size = UniformFloatHyperparameter("min.node.size", lower=0, upper=1,)
        cs.add_hyperparameters(
            [
                num_trees,
                replace,
                sample_fraction,
                mtry,
                respect_unordered_factors,
                min_node_size,
            ]
        )
        return cs


class XGBoost(ExploringOpenML):
    url = "https://ndownloader.figshare.com/files/10462315"

    @staticmethod
    def get_configuration_space():
        cs = ConfigurationSpace()
        nrounds = UniformIntegerHyperparameter("nrounds", lower=1, upper=5000,)
        eta = UniformFloatHyperparameter("eta", lower=2 ** -10, upper=2 ** 0, log=True,)
        subsample = UniformFloatHyperparameter("subsample", lower=0, upper=1,)
        booster = CategoricalHyperparameter("booster", choices=["gblinear", "gbtree"],)
        max_depth = UniformIntegerHyperparameter("max_depth", lower=1, upper=15,)
        min_child_weight = UniformFloatHyperparameter(
            "min_child_weight", lower=2 ** 0, upper=2 ** 7, log=True,
        )
        colsample_bytree = UniformFloatHyperparameter(
            "colsample_bytree", lower=0, upper=1,
        )
        colsample_bylevel = UniformFloatHyperparameter(
            "colsample_bylevel", lower=0, upper=1,
        )
        lambda_ = UniformFloatHyperparameter(
            "lambda", lower=2 ** -10, upper=2 ** 10, log=True,
        )
        alpha = UniformFloatHyperparameter(
            "alpha", lower=2 ** -10, upper=2 ** 10, log=True,
        )
        cs.add_hyperparameters(
            [
                nrounds,
                eta,
                subsample,
                booster,
                max_depth,
                min_child_weight,
                colsample_bytree,
                colsample_bylevel,
                lambda_,
                alpha,
            ]
        )
        colsample_bylevel_condition = EqualsCondition(
            colsample_bylevel, booster, "gbtree",
        )
        colsample_bytree_condition = EqualsCondition(colsample_bytree, booster, "gbtree",)
        max_depth_condition = EqualsCondition(max_depth, booster, "gbtree",)
        min_child_weight_condition = EqualsCondition(min_child_weight, booster, "gbtree",)
        cs.add_conditions(
            [
                colsample_bylevel_condition,
                colsample_bytree_condition,
                max_depth_condition,
                min_child_weight_condition,
            ]
        )
        return cs


all_datasets = [
    3,
    31,
    37,
    44,
    50,
    151,
    312,
    333,
    334,
    335,
    1036,
    1038,
    1043,
    1046,
    1049,
    1050,
    1063,
    1067,
    1068,
    1120,
    1461,
    1462,
    1464,
    1467,
    1471,
    1479,
    1480,
    1485,
    1486,
    1487,
    1489,
    1494,
    1504,
    1510,
    1570,
    4134,
    4534,
]
all_model_classes = [
    GLMNET,
    RPART,
    SVM,
    Ranger,
    XGBoost,
]

for model_class in all_model_classes:
    for dataset_id_ in all_datasets:
        benchmark_string = """class %s_%d(%s):
         def __init__(self, n_splits=10, n_iterations=30, rng=None, n_jobs=1):
             super().__init__(dataset_id=%d, n_splits=n_splits, n_iterations=n_iterations, rebuild=False, rng=rng, n_jobs=n_jobs)
    """ % (
            model_class.__name__,
            dataset_id_,
            model_class.__name__,
            dataset_id_,
        )

        exec(benchmark_string)
