from metagen.validation import validate_material_properties, PropertyValidationStatus
from metagen.processing import get_sim_results
from metagen.util import list_all_filtered, format_float
import json
import os
from tqdm import tqdm
import re
import base64
from enum import Enum
import itertools
import random
import copy
import numpy as np
from metagen.util import AWS_ACCOUNT_ID, open_file
from metagen.evaluation import IoU
from metagen.validation import load_voxels
from metagen.benchmarks_inverse_design import get_target_property_profile_from_properties, TargetType, PropertyType, property_references
from tqdm.contrib.concurrent import process_map
from dataclasses import dataclass


from enum import Enum
class LLMFormat(Enum):
    NOVA = 0
    LLAVA = 1
    OPENAI = 2

def data_path_join(data_root, *path):
    return os.path.join(data_root.rstrip('/'), os.path.join(*[p.lstrip('/') for p in path]))

def get_header(code):
    code = code[code.index("'''")+3:]
    header = code[:code.index("'''")].strip()
    return header

def get_program(code):
    code = code[code.index("'''")+3:]
    code = code[code.index("'''")+3:]
    if "# --- END:" in code:
        code = code[:code.index("# --- END:")].strip()
    return code.strip()

def split(data, split_sizes=[0.8, 0.1, 0.1], seed=42):
    """
    Splits a dataset sections
    
    Args:
        data (list): The dataset to be split.
        split_sizes (list): A list of slit sizes. If floats, they will be normalized. If ints, one extra split will be made for the rest of the data.
        seed (int): Random seed for reproducibility.
    
    Returns:
        splits (list): n lists with dataset splits
    """
    if any(isinstance(i, float) for i in split_sizes):
        split_sizes = [i/sum(split_sizes) for i in split_sizes]
        split_sizes = [int(i * len(data)) for i in split_sizes]
        if sum(split_sizes) != len(data):
            split_sizes[-1] += len(data) - sum(split_sizes)
    else:
        assert all(isinstance(i, int) for i in split_sizes), "Split sizes must be either all floats or all ints"
        if sum(split_sizes) > len(data):
            raise ValueError("Split sizes must be less than or equal to the length of the data")
        if sum(split_sizes) < len(data):
            split_sizes.append(len(data) - sum(split_sizes))
    
    data = copy.copy(data)
    random.seed(seed)
    random.shuffle(data)
    
    splits = []
    start = 0
    for size in split_sizes:
        end = start + size
        splits.append(data[start:end])
        start = end
    return splits

class Database:
    def __init__(self, data_root, database='models'):
        self.data_root = data_root
        database_path = os.path.join(data_root, database)
        database_code_paths = list_all_filtered(database_path, ['**/code.py'])
        self.database_sources = ['/'.join(p.split('/')[:-1])[len(data_root):] for p in database_code_paths]
    
    @property
    def sources(self):
        return self.database_sources
    
    def join(self, *path):
        return data_path_join(self.data_root, *path)
    
    def path(self, *path):
        return self.join(*path)
    
    def raw_code(self, source):
        code_path = self.path(source, 'code.py')
        with open(code_path, 'r') as f:
            code = f.read()
        return code
    def code(self, source):
        code = self.raw_code(source)
        return get_program(code)
    def raw_properties(self, source):
        properties_path = self.path(source, 'structure_info.json')
        with open(properties_path, 'r') as f:
            properties = json.load(f)
        return properties
    def properties(self, source):
        properties = self.raw_properties(source)
        return get_sim_results(properties,False)
    def renders(self, source):
        views = ['top', 'front', 'right', 'top_right']
        return {v : os.path.join(source, f'{v}.png') for v in views}
    def voxels(self, source):
        return os.path.join(source, 'vox_active_cells.txt')
    def mesh(self, source):
        return os.path.join(source, 'thickened_mc.obj')
    def validation_status(self, source): 
        properties = self.raw_properties(source)
        return validate_material_properties(properties)


code_output_format = """

# Output Format:
Generate a Metagen program within a python code block:

```python
from metagen import *

def make_structure(...) -> Structure:
    ...
```
"""


def inverse_design(source: str, db: Database, n: int = 3, min_k: int = 1, max_k: int = 6) -> list:
    properties = db.properties(source)
    profiles = get_target_property_profile_from_properties(properties, n, min_k, max_k)
    code = db.code(source)

    tasks = []
    for i, profile in enumerate(profiles):
        descriptions = []
        for p in profile:
            descriptions.append(random.sample(p['target_descriptions'],1)[0])
        random.shuffle(descriptions)

        num_targets = len(descriptions)

        adjectives = [d['description'] for d in descriptions if d['description_type'] == 'adjective']
        front_adjectives = []
        back_adjectives = []
        for a in adjectives:
            if random.random() < 0.5:
                front_adjectives.append(a)
            else:
                back_adjectives.append(a)
        props = [d['description'] for d in descriptions if d['description_type'] == 'property']
        verbs = [d['description'] for d in descriptions if d['description_type'] == 'verb']

        front_adjectives_str = ', '.join([p for p in front_adjectives])
        if len(back_adjectives) > 1:
            back_adjectives[-1] = 'and ' + back_adjectives[-1]
        back_sep = ', ' if len(back_adjectives) > 2 else ' '
        back_adjectives_str = back_sep.join([p for p in back_adjectives])

        pronoun = 'a'
        if len(front_adjectives) > 0:
            pronoun = 'an' if front_adjectives_str[0] in 'aeiou' else 'a'

        if len(front_adjectives_str) > 0:
            front_adjectives_str += ' '

        query = f"Write a metagen program that creates {pronoun} {front_adjectives_str}material"

        if len(verbs) > 1:
            verbs[-1] = 'and ' + verbs[-1]
        verb_sep = ', ' if len(verbs) > 2 else ' '
        verbs_str = 'that ' + verb_sep.join([p for p in verbs])

        if len(back_adjectives_str) > 0:
            query += " that is " + back_adjectives_str
            if len(verbs) > 0:
                query += ", " + verbs_str
            if len(props) > 0:
                query += ", with "
        else:
            if len(verbs) > 0:
                query += " " + verbs_str
                if len(props) > 0:
                    query += ", with "
            else:
                if len(props) > 0:
                    query += " with "
            
        if len(props) > 1:
            props[-1] = 'and ' + props[-1]
        query += ', '.join([p for p in props])
        query += "."

        tasks.append({
            'task_category': 'inverse_design',
            'task_type': f'{num_targets}_target_inverse_design',
            'label': f'{num_targets}_target_inverse_design_{source}_{i}',
            'source': source,
            'data': profile,
            'query': "# Task:\n" + query + code_output_format,
            'response': '```python\n' + code + '\n```',
        })
    return tasks


def reconstruction(source, db, n_views=4, model_args='...'):
    assert n_views in range(1,5)
    if db.validation_status(source).value not in [0, 3, 4]:
        return []
    lang_template = dsl_code_template.format(model_args=model_args)
    tasks = []
    code = db.code(source)
    images = db.renders(source)
    image_sets = list(itertools.combinations(images.items(), n_views))
    for image_set in image_sets:
        image_set = dict(image_set)

        viewpoints = '\n'.join([rendered_views_templates[k].format(image=v) for k,v in image_set.items()])

        query = generate_from_image_template.format(
            rendered_views=viewpoints,
            lang_template=lang_template
        )

        response = f"```python\n{code}\n```"

        view_labels = ':'.join([k for k in image_set.keys()])
        task = {
            'task_category': 'reconstruction',
            'task_type': f'{n_views}_view_reconstruction',
            'label': f'{n_views}_view_reconstruction_{view_labels}_{source}',
            'source': source,
            'data': image_set | {'code': code, 'mesh': db.mesh(source), 'voxels': db.voxels(source)},
            'query': query,
            'response': response
        }
        tasks.append(task)
    return tasks

def chamfer_distance(x, y):
    """Chamfer distance between two point clouds
    Parameters
    ----------
    x: numpy array [n_points_x, n_dims]
        first point cloud
    y: numpy array [n_points_y, n_dims]
        second point cloud
    Returns
    -------
    chamfer_dist: float
        computed bidirectional Chamfer distance:
            sum_{x_i \\in x}{\\min_{y_j \\in y}{||x_i-y_j||**2}} + sum_{y_j \\in y}{\\min_{x_i \\in x}{||x_i-y_j||**2}}
    """
    from sklearn.neighbors import NearestNeighbors # was getting circular import errors if imported at the module level
    x_nn = NearestNeighbors(n_neighbors=1, leaf_size=1, algorithm='kd_tree', metric='l2').fit(x)
    min_y_to_x = x_nn.kneighbors(y)[0]
    y_nn = NearestNeighbors(n_neighbors=1, leaf_size=1, algorithm='kd_tree', metric='l2').fit(y)
    min_x_to_y = y_nn.kneighbors(x)[0]
    chamfer_dist = (np.mean(min_y_to_x) + np.mean(min_x_to_y))/2

    return chamfer_dist

predict_from_image_template = """
# Task:
Analyze these views of a metamaterial, and predict its material properties.

# Inputs:

**Rendered View:**

- Angled (Front-Top-Right): <[{top_right}]>

# Output Format:

Output a json object, delimited by ```json ```, where the keys are material property names, and the values are the predicted material properties. Predict these properties (keys):
- "A" : Anisotropy (universal anisotropy index)
- "E" : Young's Modulus relative to E_base
- "K" : Bulk modulus relative to E_base
- "G": Shear modulus relative to E_base
- "nu": Isotropic Poisson ratio
- "V" : Relative Density (Volume Fraction)
"""

predict_from_images_and_code_template = """
# Task:
Analyze these views of a metamaterial, and the Metagen program, and predict its material properties.

# Inputs:

**Metagen Program:**

{code}

**Rendered Views:**
- Top: <[{top}]>
- Front: <[{front}]>
- Right: <[{right}]>
- Angled (Front-Top-Right): <[{top_right}]>

# Output Format:

Output a json object, delimited by ```json ```, where the keys are material property names, and the values are the predicted material properties. Predict these properties (keys):
- "A" : Anisotropy (universal anisotropy index)
- "E" : Young's Modulus relative to E_base
- "K" : Bulk modulus relative to E_base
- "G": Shear modulus relative to E_base
- "nu": Isotropic Poisson ratio
- "V" : Relative Density (Volume Fraction)
"""

def material_understanding(source: str, db: Database) -> list:
    code = db.code(source)
    renders = db.renders(source)
    query_inputs = {'code': code} | renders
    image_and_code_query = predict_from_images_and_code_template.format(**query_inputs)
    image_query = predict_from_image_template.format(**renders)
    properties = db.properties(source)
    target_properties = {k: properties[k] for k in ['A', 'E', 'K', 'G', 'nu', 'V']}
    formatted_target_propeties = {k: format_float(v) for k,v in target_properties.items()}
    
    multiview_task = {
            'task_category': 'material_understanding',
            'task_type': 'multiview_and_code_material_understanding',
            'label': f'multiview_and_code_material_understanding_{source}',
            'source': source,
            'data': target_properties,
            'query': image_and_code_query,
            'response': "```json\n" + json.dumps(formatted_target_propeties,indent=2) + "\n```",
        }
    
    singleview_task = {
            'task_category': 'material_understanding',
            'task_type': 'single_view_material_understanding',
            'label': f'single_view_material_understanding_{source}',
            'source': source,
            'data': target_properties,
            'query': image_query,
            'response': "```json\n" + json.dumps(formatted_target_propeties,indent=2) + "\n```",
        }
    
    return [multiview_task, singleview_task]

def tasks_from_source(source: str, db: Database) -> list:
    tasks = []
    for i in range(1, 5): # Create all 1,2,3, and 4 view reconstruction tasks
        tasks += reconstruction(source, db, n_views=i)
    for i in range(1, 7): # Create one inverse design task for each of 1-6 target properties
        try:
            new_tasks = inverse_design(source, db, 1, i, i)
            tasks += new_tasks
        except Exception as e:
            continue # For now skip NaN errors -- we can regenerate this later if necessary
    tasks += material_understanding(source, db) # Create single and multiview+code material understanding tasks

    return tasks

def generate_benchmarks(db: Database, test_size: int = 500, validate_size: int = 50):
    #source_tasks = [tasks_from_source(source, db) for source in tqdm(db.sources, desc="Generating tasks", leave=False)]
    source_tasks = process_map(tasks_from_source, db.sources, [db]*len(db.sources), chunksize=10, desc="Generating tasks", total=len(db.sources), leave=False)
    all_tasks = [task for source in source_tasks for task in source]
    task_types = set([task['task_type'] for task in all_tasks])
    typed_tasks = {task_type: [] for task_type in task_types}
    for task in all_tasks:
        typed_tasks[task['task_type']].append(task)
    benchmark_dir = db.path('/benchmark')
    os.makedirs(benchmark_dir, exist_ok=True)
    train_path = db.path('/benchmark/train.txt')
    test_path = db.path('/benchmark/test.txt')
    validate_path = db.path('/benchmark/validate.txt')
    if os.path.exists(train_path) and os.path.exists(test_path) and os.path.exists(validate_path):
        with open(train_path, 'r') as f:
            train_sources = set([x.strip() for x in f.readlines()])
        with open(test_path, 'r') as f:
            test_sources = set([x.strip() for x in f.readlines()])
        with open(validate_path, 'r') as f:
            validate_sources = set([x.strip() for x in f.readlines()])
    else:
        test_sources, validate_sources, train_sources = split(db.sources, [test_size, validate_size])
        with open(train_path, 'w') as f:
            f.write('\n'.join(train_sources))
        with open(test_path, 'w') as f:
            f.write('\n'.join(test_sources))
        with open(validate_path, 'w') as f:
            f.write('\n'.join(validate_sources))
    
    # Write full benchmark set out
    full_benchmark_dir = db.path('/benchmark/omnitask')
    os.makedirs(full_benchmark_dir, exist_ok=True)
    with open(os.path.join(full_benchmark_dir, 'train.jsonl'), 'w') as f:
        for task in all_tasks:
            if task['source'] in train_sources:
                f.write(json.dumps(task) + '\n')
    with open(os.path.join(full_benchmark_dir, 'test.jsonl'), 'w') as f:
        for task in all_tasks:
            if task['source'] in test_sources:
                f.write(json.dumps(task) + '\n')
    with open(os.path.join(full_benchmark_dir, 'validate.jsonl'), 'w') as f:
        for task in all_tasks:
            if task['source'] in validate_sources:
                f.write(json.dumps(task) + '\n')
    
    # Write out task-specific benchmarks
    for task_type, tasks in typed_tasks.items():
        task_category = tasks[0]['task_category']
        task_dir = db.path(f'/benchmark/{task_category}', task_type)
        os.makedirs(task_dir, exist_ok=True)
        with open(os.path.join(task_dir, 'train.jsonl'), 'w') as f:
            for task in tasks:
                if task['source'] in train_sources:
                    f.write(json.dumps(task) + '\n')
        with open(os.path.join(task_dir, 'test.jsonl'), 'w') as f:
            for task in tasks:
                if task['source'] in test_sources:
                    f.write(json.dumps(task) + '\n')
        with open(os.path.join(task_dir, 'validate.jsonl'), 'w') as f:
            for task in tasks:
                if task['source'] in validate_sources:
                    f.write(json.dumps(task) + '\n')


def format_task(task: dict, llm: LLMFormat, db, test: bool = False, system_prompt: str = None, data_root: str = None):
    if llm == LLMFormat.NOVA:
        return format_nova(task,  db, test, system_prompt, data_root)
    elif llm == LLMFormat.LLAVA:
        return format_llava(task,  db, test, system_prompt)

def message_to_content(message: str, format: LLMFormat, db, test: bool, data_root: str = None) -> str:
    """
    Converts a message to the content format for the specified LLM format.
    
    Args:
        message (str): The message to be converted.
        format (LLMFormat): The LLM format to convert to.
    
    Returns:
        str: The converted message in the specified LLM format.
        images: A list of images in the message. Only returned if format is LLAVA.
    """

    contents = []
    images = []
    pattern = r'(<\[[^\]]+\]>)'
    parts = re.split(pattern, message)
    for part in parts:
        if re.match(pattern, part):
            db_image_path = part[2:-2]
            if data_root is not None:
                image_path = data_root.rstrip('/') + '/' + db_image_path.lstrip('/')
            else:
                image_path = db_image_path
            images.append(db_image_path)
            if format == LLMFormat.NOVA:
                if test:
                    local_path = db.path(db_image_path)
                    with open(local_path, "rb") as image_file:
                        image_data = image_file.read()
                        encoded_image = base64.b64encode(image_data).decode('utf-8')
                    contents.append({
                        'image':{
                            'format': 'png', 
                            'source': {
                                'bytes': encoded_image
                            }
                        }
                    })
                else:
                    contents.append({
                        'image':{
                            'format': 'png', 
                            'source': {
                                's3Location': {
                                    'uri': image_path, 
                                    'bucketOwner': AWS_ACCOUNT_ID
                                }
                            }
                        }
                    })
            elif format == LLMFormat.LLAVA:
                contents.append({'type': 'image'})
        else:
            if format == LLMFormat.NOVA:
                contents.append({'text': part})
            elif format == LLMFormat.LLAVA:
                contents.append({'type': 'text', 'text': part})
    if format == LLMFormat.LLAVA:
        return contents, images
    else:
        return contents
    

def format_nova(task: dict, db, test: bool = False, system_prompt: str = None, data_root: str = None):
    formatted_task = {}
    messages = []
    if system_prompt is not None:
        formatted_task['system'] = message_to_content(system_prompt, LLMFormat.NOVA, db, test, data_root)
    user_content = message_to_content(task['query'], LLMFormat.NOVA, db, test, data_root)
    messages.append({'role': 'user', 'content': user_content})
    if not test:
        assistant_content = message_to_content(task['response'], LLMFormat.NOVA, db, test, data_root)
        messages.append({'role': 'assistant', 'content': assistant_content})
    formatted_task['messages'] = messages
    formatted_task['schemaVersion'] = 'messages-v1'
    if test:
        formatted_task = {
            'id': task['label'],
            'modelInput': formatted_task
        }
    return formatted_task
def format_llava(task: dict, db, test: bool = False, system_prompt: str = None, data_root: str = None):
    messages = []
    images = []
    if system_prompt is not None:
        system_content, system_images = message_to_content(system_prompt, LLMFormat.LLAVA, db, test, data_root)
        messages.append({'role': 'system', 'content': system_content})
        images.extend(system_images)
    user_content, user_images = message_to_content(task['query'], LLMFormat.LLAVA, db, test, data_root)
    messages.append({'role': 'user', 'content': user_content})
    images.extend(user_images)
    
    assistant_content, assistant_images = message_to_content(task['response'], LLMFormat.LLAVA, db, test, data_root)
    messages.append({'role': 'assistant', 'content': assistant_content})
    images.extend(assistant_images)
    id = task['label']
    return {
        'id': id,
        'messages': messages,
        'images': images
    }


# ==========================================
# v3 Benchmark Evaluation Code
# ==========================================

from metagen.util import extract_and_classify_blocks

def extract_predicted_code_nova(db: Database, model: str, task: str, subtask: str = None):
    if subtask:    
        test_path = db.path('/benchmark',  task, subtask, 'test.jsonl')
        response_path = db.path('/workspace/inference_data', model, task, subtask, 'test_responses.jsonl')
        code_dir = db.path('/workspace/inference_data', model, task, subtask, 'predicted_code')
    else:
        test_path = db.path('/benchmark',  task, 'test.jsonl')
        response_path = db.path('/workspace/inference_data', model, task, 'test_responses.jsonl')
        code_dir = db.path('/workspace/inference_data', model, task, 'predicted_code')
    os.makedirs(code_dir, exist_ok=True)
    os.makedirs(os.path.dirname(response_path), exist_ok=True)
    with open(test_path, 'r') as f:
        test_data = [json.loads(line) for line in f.readlines()]
        label_to_index = {x['label']:i for i,x in enumerate(test_data)}
    with open(response_path, 'r') as f:
        response_data = [json.loads(line) for line in f.readlines()]
    assert all([label_to_index[r['modelInput']['id']] == i for i,r in enumerate(response_data)]), "Response data does not match test data"
    for i, r in enumerate(response_data):
        if test_data[i]['task_category'] == 'material_understanding':
            lang = 'json'
            ext = 'json'
        else:
            lang = 'python'
            ext = 'py'
        code_blocks = get_code_blocks_from_nova_inference(r, lang)
        if len(code_blocks) != 1:
            print('Not exactly one code block found')
            print(i)
            print(lang)
            print(r['modelOutput']['output']['message']['content'][0]['text'])
            print(json.dumps(r, indent=2))
            continue
        code_path = os.path.join(code_dir, f'{i}.{ext}')
        with open(code_path, 'w') as f:
            f.write(code_blocks[0])




def get_code_blocks_from_nova_inference(pred, lang='python'):
    return extract_and_classify_blocks(
        pred['modelOutput']['output']['message']['content'][0]['text'],
        [lang]
    )[lang]

"""
Put inference results in
/workspace/inference_data/model_name/(training_data_path)/
 - test_responses.jsonl
 - predicted_code/line_no.py (line from test.jsonl)
 - predicted_failures/line_no/program.py
 - predicted_successes/line_no/program.py
"""

@dataclass
class BenchmarkTask:
    task_category: str
    task_type: str
    label: str
    source: str
    data: dict
    query: str
    response: str

    def to_dict(self):
        return {
            'task_category': self.task_category,
            'task_type': self.task_type,
            'label': self.label,
            'source': self.source,
            'data': self.data,
            'query': self.query,
            'response': self.response
        }
    
    @classmethod
    def from_dict(cls, data):
        return cls(
            task_category=data['task_category'],
            task_type=data['task_type'],
            label=data['label'],
            source=data['source'],
            data=data['data'],
            query=data['query'],
            response=data['response']
        )

def evaluate_benchmarks(
      db: Database,
      output_path: str,
      model_results: list
):
    import pandas as pd

    all_metrics = []
    for model_result in model_results:
        all_metrics += evaluate_model_benchmarks(db, **model_result)
    table = pd.DataFrame.from_records(all_metrics)
    table.to_parquet(db.path(output_path))
    return table

def evaluate_task(data):
    i,packed_data = data
    task, model_name, offset, db, prediction_path, processed_path, skip = packed_data
    if skip:
        return []
    if task.task_category == 'material_understanding':
        pred_path = os.path.join(prediction_path, f"{str(i + offset)}.json")
        metrics = evaluate_material_understanding(task, pred_path, db)
    elif task.task_category == 'inverse_design':
        pred_path = os.path.join(processed_path, str(i + offset))
        metrics = evaluate_inverse_design(task, pred_path, db)
    elif task.task_category == 'reconstruction':
        pred_path = os.path.join(processed_path, str(i + offset))
        metrics = evaluate_reconstruction(task, pred_path, db)
    else:
        assert False, f"Unknown task category {task.task_category}"
    metrics = [m | {
        'Category': task.task_category,
        'Task': task.task_type,
        'Label': task.label,
        'Model': model_name,
        } for m in metrics]
    return metrics

def evaluate_model_benchmarks(
        db: Database, 
        test_path: str, 
        prediction_path: str,
        processed_path: str,
        model_name: str,
        index_at_one: bool = False):
    """
    Evaluate test predictions across all task type
    """

    # Check if DB or global paths and update if necessary
    assert os.path.exists(db.path(test_path)), f"Test path {test_path} does not exist"
    assert os.path.exists(db.path(prediction_path)), f"Prediction path {prediction_path} does not exist"

    with open(db.path(test_path), 'r') as f:
        test_data: list[BenchmarkTask] = [BenchmarkTask.from_dict(json.loads(line)) for line in f.readlines()]

    full_metrics = []
    offset = 1 if index_at_one else 0
    
    # Zip together all inputs needed for evaluate_task
    packed_tasks = zip(
        test_data, 
        [model_name]*len(test_data), 
        [offset]*len(test_data), 
        [db]*len(test_data),
        [prediction_path]*len(test_data),
        [processed_path]*len(test_data),
        [False]*len(test_data) # Skip in cases of lost data
    )

    num_tasks = len(test_data)

    # Workaround for o3 missing evaluations
    if model_name == 'OpenAIO3':
        with open(db.path('/workspace/inference_data/OpenAIO3/omnitask/test_responses.jsonl'), 'r') as f:
            responses = [json.loads(line) for line in f.readlines()]
        o3_labels = {r['custom_id'] for r in responses}
        repacked_tasks = []
        for pack in packed_tasks:
            if pack[0].label in o3_labels:
                repacked_tasks.append(pack)
            else:
                repacked_tasks.append(pack[:-1] + (True,)) # Add skip flag
        packed_tasks = repacked_tasks

    for metrics in process_map(evaluate_task, enumerate(packed_tasks), total=num_tasks, desc=f"Evaluating {model_name} on {test_path}"):
        full_metrics += metrics
    
    return full_metrics

def evaluate_material_understanding(task: BenchmarkTask, prediction_path: str, db: Database):
    if not os.path.exists(db.path(prediction_path)):
        # Invalid
        return [
            {'metric': 'Valid', 'value': 0.0}
        ]
    
    metrics = []
    targets = task.data
    try:
        with open(db.path(prediction_path), 'r') as f:
            pred_properties = json.load(f)
    except:
        return [
            {'metric': 'Valid', 'value': 0.0}
        ]
    total_normalized_error = 0
    valid = True
    for target_property, target_value in targets.items():
        if target_property not in pred_properties:
            valid = False
            continue
        pred_value = float(pred_properties[target_property])
        absolute_error, normalized_error = evaluate_target(target_value, pred_value, target_property, 'value')
        total_normalized_error += normalized_error
        metrics.append({'metric': f'{target_property} Absolute Error', 'value': absolute_error})
        metrics.append({'metric': f'{target_property} Normalized Error', 'value': normalized_error})
    metrics.append({'metric': 'Valid', 'value': 1.0 if valid else 0.0})
    metrics.append({'metric': 'Average Normalized Error', 'value': total_normalized_error / len(targets)})
    return metrics

def evaluate_inverse_design(task: BenchmarkTask, prediction_path: str, db: Database):
    if not os.path.exists(db.path(prediction_path)):
        # Invalid
        return [
            {'metric': 'Valid', 'value': 0.0}
        ]
    
    metrics = []
    targets = task.data
    pred_properties = db.properties(prediction_path)
    total_normalized_error = 0
    for target in targets:
        target_value = target['target_value']
        target_property = target['property']
        pred_value = pred_properties[target_property]
        absolute_error, normalized_error = evaluate_target(target_value, pred_value, target_property, target['target_type'])
        metrics.append({'metric': f'{target_property} Absolute Error', 'value': absolute_error})
        metrics.append({'metric': f'{target_property} Normalized Error', 'value': normalized_error})
        total_normalized_error += normalized_error
    metrics.append({'metric': 'Average Normalized Error', 'value': total_normalized_error / len(targets)})
    metrics.append({'metric': 'Valid', 'value': 1.0})
    for m in metrics:
        if np.isnan(m['value']) or m['value'] > np.e**10:
            # Invalid
            return [
                {'metric': 'Valid', 'value': 0.0}
            ]
    return metrics


def evaluate_target(target: float, pred: float, property: str, target_type: str):
    property_info = property_references[property]
    property_range = abs(property_info['dataset_coverage']['max'] - property_info['dataset_coverage']['min'])
    error = pred - target
    if target_type == 'upper_bound' and pred <= target:
        error = 0
    elif target_type == 'lower_bound' and target <= pred:
        error = 0
    return abs(error), abs(error) / property_range

def evaluate_reconstruction(task: BenchmarkTask, prediction_path: str, db: Database):
    
    if not os.path.exists(db.path(prediction_path)):
        # Invalid
        return [
            {'metric': 'Valid', 'value': 0.0},
        ]
    pred_voxel_path = db.voxels(db.path(prediction_path))
    pred_voxels = load_voxels(open_file(pred_voxel_path))
    gt_voxels = load_voxels(open_file(db.voxels(db.path(task.source))))
    pred_points = np.argwhere( pred_voxels == 1) / 99
    gt_points = np.argwhere( gt_voxels == 1) / 99

    chamfer_dist = chamfer_distance(pred_points, gt_points)
    iou = IoU(pred_voxels, gt_voxels)
    return [
            {'metric': 'Valid', 'value': 1.0},
            {'metric': 'Chamfer Distance', 'value': chamfer_dist},
            {'metric': 'IoU', 'value': iou}
        ]

universal_system_prompt_template = """You are an expert metamaterials assistant that generates and analyzes cellular metamaterial designs based on material properties, images, and programatic definitions in the Metagen metamaterial DSL.


# Procedural Description in a Metamaterial DSL:

{api_description}

# Material Analysis:
You can analyze the density, anisotropy, and elasticity properties of metamaterials. All metamaterials are assumed to be constucted from an isotropic base material with Poisson's ratio nu = 0.45.
The Young's Modulus of this base material is not specified, instead, the elastic moduli of the metamaterials -- Young's Modulus (E), Bulk Modulus (K), and Shear Modulus (G), are expressed relative to the base material Young's modulus (E_base). This means, for example, that relative Young's Moduli can range from 0 to 1. The material properties you can analyze are:

- E: Young's Modulus, Voigt-Reuss-Hill (VRH) average, relative to E_base
- E_1,E_2,E_3: Directional Young's Moduli, relative to E_base
- G: Shear Modulus (VRH average), relative to E_base
- G_23,G_13,G_12: Directional Shear Moduli, relative to E_base
- nu: Poisson ratio (VRH average)
- nu_12, nu_13, nu_23, nu_21, nu_31, nu_32: Directional Poisson ratios
- K: Bulk modulus (VRH average), relative to E_base
- A: Anisotropy (universal anisotropy index)
- V: Volume Fraction

# Material Images:

Images of metamaterials depict a base cell of the material rendered from four viewpoints:

- from the top
- from the front side
- from the right side
- from an angle at the upper-front-right

# Tasks:

You will be asked to perform several kinds of tasks:

- Reconstruction: from one or more images of a target material, reconstruct a Metagen program that generates the metamaterial in the images.
- Inverse Design: from a description of the properties of a desired materials, write a Metagen program that creates a metamaterial with those properties.
- Material Understanding: from images of a metamaterial and/or a Metagen program, analyze a material and predict its properties.
"""

code_api_description = """
Programs in Metagen are built in two stages: one that creates local geometric structure, and a second that patterns this structure throughout space. Each of these is further broken down into subparts.


==================================
    API description (Boilerplate)
==================================
Each program is given as a python file (.py).
This program must import the metagen package and define a function called "make_structure()", which returns the final Structure object defined by the program. 
If parameters are present in make_structure(), they MUST have a default value.
Specifically, the file structure is as follows: 


from metagen import *

def make_structure(...) -> Structure:
    <content>



==================================
    DSL description
==================================

======= Skeleton Creation ========
vertex(cpEntity, t)
    @description:
        Create a new vertex. This vertex is defined relative to its containing convex polytope (CP). It will only have an embedding in R3 once the CP has been embedded.
    @params:
        cpEntity    - an entity of a convex polytope (CP), referenced by the entity names.
        t           - [OPTIONAL] list of floats in range [0,1], used to interpolate to a specific position on the cpEntity.
                        If cpEntity is a corner, t is ignored.
                        If cpEntity is an edge, t must contain exactly 1 value. t is used for linear interpolation between the endpoints of cpEntity.
                        If cpEntity is a face, t must contain exactly 2 values. If cpEntity is a triangular face, t is used to interpolate via barycentric coordinates. If cpEntity is a quad face, bilinear interpolation is used.
                        
                        If the optional interpolant t is omitted for a non-corner entity, the returned point will be at the midpoint (for edge) or the centroid (for face) of the entity. Semantically, we encourage that t be excluded (1) if the structure would be invalid given a different non-midpoint t, or (2) if the structure would remain unchanged in the presence a different t (e.g., in the case of a conjugate TPMS, where only the entity selection matters).
    @returns:
        vertex      - the new vertex object 
    @example_usage:
        v0 = vertex(cuboid.edges.BACK_RIGHT, [0.5])
        v1 = vertex(cuboid.edges.TOP_LEFT)


Polyline(ordered_verts)
    @description:
        Creates a piecewise-linear path along the ordered input vertices. All vertices must be referenced to the same CP (e.g., all relative to cuboid entities). The resulting path will remain a polyline in any structures that include it.
    @params:
        ordered_verts   - a list of vertices, in the order you'd like them to be traversed. A closed loop may be created by repeating the zeroth element at the end of the list. No other vertex may be repeated. Only simple paths are permitted.
    @returns:
        polyline        - the new polyline object
    @example_usage:
        p0 = Polyline([v2, v3])
        p0 = Polyline([v0, v1, v2, v3, v4, v5, v0])


Curve(ordered_verts)
    @description:
        Creates a path along the ordered input vertices. This path will be smoothed at a later stage (e.g., to a Bezier curve), depending on the lifting procedures that are chosen. All input vertices must be referenced to the same CP (e.g., all relative to cuboid entities). 
    @params:
        ordered_verts   - a list of vertices, in the order you'd like them to be traversed. A closed loop may be created by repeating the zeroth element at the end of the list. No other vertex may be repeated. Only simple paths are permitted.
    @returns:
        curve           - the new curve object
    @example_usage:
        c0 = Curve([v2, v3])
        c0 = Curve([v0, v1, v2, v3, v4, v5, v0])

skeleton(entities)
    @description:
        Combines a set of vertices OR polylines/curves into a larger structure, over which additional information can be inferred. For example, within a skeleton, multiple open polylines/curves may string together to create a closed loop, a branched path, or a set of disconnected components.
    @params:
        entities        - a list of entities (vertices or polylines/curves) to be combined. A given skeleton must only have entities with the same dimension -- that is, it must consist of all points or all polylines/curves.
    @returns:
        skeleton        - the new skeleton object
    @example_usage:
        skel = skeleton([curve0, polyline1, curve2, polyline3])
        skel = skeleton([v0])


======= Lifting Procedures ========
UniformBeams(skel, thickness)
    @description:
        Procedure to lift the input skeleton to a 3D volumetric structure by instantiating a beam of the given thickness centered along each polyline/curve of the input skeleton.
    @requirements:
        The skeleton must contain only polylines and/or curves. The skeleton must not contain any standalone vertices.
    @params:
        skel            - the skeleton to lift
        thickness       - the diameter of the beams
    @returns:
        liftProc        - the lifted skeleton
    @example_usage:
        liftProcedure = UniformBeams(skel, 0.03)

SpatiallyVaryingBeams(skel, thicknessProfile)
    @description:
        Procedure to lift the input skeleton to a 3D volumetric structure by instantiating a beam of the given spatially-varying thickness profile centered along each polyline/curve of the input skeleton.
    @requirements:
        The skeleton must contain only polylines and/or curves. The skeleton must not contain any standalone vertices.
    @params:
        skel            - the skeleton to lift
        thicknessProfile- specifications for the diameter of the beams along each polyline/curve. Given as a list[list[floats]], where the each of the n inner lists gives the information for a single sample point along the polyline/curve. The first element in each inner list provides a position parameter t\\in[0,1] along the polyline/curve, and the second element specifies the thickness of the beam at position t
    @returns:
        liftProc        - the lifted skeleton
    @example_usage:
        liftProcedure = SpatiallyVaryingBeams(skel, 0.03)

UniformDirectShell(skel, thickness)
    @description:
        Procedure to lift the input skeleton to a 3D volumetric structure by inferring a surface that conforms to the boundary provided by the input skeleton. The surface is given by a simple thin shell model: the resulting surface is incident on the provided boundary while minimizing a weighted sum of bending and stretching energies. The boundary is fixed, though it may be constructed with a mix of polylines and curves (which are first interpolated into a spline, then fixed as part of the boundary). The skeleton must contain a single closed loop composed of one or more polylines and/or curves. The skeleton must not contain any standalone vertices.
    @requirements:

    @params:
        skel            - the skeleton to lift
        thickness       - the thickness of the shell. The final offset is thickness/2 to each side of the inferred surface.
    @returns:
        liftProc        - the lifted skeleton
    @example_usage:
        liftProcedure = UniformDirectShell(skel, 0.1)

UniformTPMSShellViaConjugation(skel, thickness)
    @description:
        Procedure to lift the input skeleton to a 3D volumetric structure by inferring a triply periodic minimal surface (TPMS) that conforms to the boundary constraints provided by the input skeleton. The surface is computed via the conjugate surface construction method. 
    @requirements: 
        The skeleton must contain a single closed loop composed of one or more polylines and/or curves. The skeleton must not contain any standalone vertices.
        Each vertex in the polylines/curves must live on a CP edge.
        Adjacent vertices must have a shared face. 
        The loop must touch every face of the CP at least once.
        If the CP has N faces, the loop must contain at least N vertices.
    @params:
        skel            - the skeleton to lift
        thickness       - the thickness of the shell. The final offset is thickness/2 to each side of the inferred surface.
    @returns:
        liftProc        - the lifted skeleton
    @example_usage:
        liftProcedure = UniformTPMSShellViaConjugation(skel, 0.03)

UniformTPMSShellViaMixedMinimal(skel, thickness)
    @description:
        Procedure to lift the input skeleton to a 3D volumetric structure by inferring a triply periodic minimal surface (TPMS) that conforms to the boundary constraints provided by the input skeleton. The surface is computed via mean curvature flow. All polyline boundary regions are considered fixed, but any curved regions may slide within their respective planes in order to reduce surface curvature during the solve.
    @requirements: 
        The skeleton must contain a single closed loop composed of one or more polylines and/or curves. The skeleton must not contain any standalone vertices.
        Each vertex in the polylines/curves must live on a CP edge.
        Adjacent vertices must have a shared face. 
    @params:
        skel            - the skeleton to lift
        thickness       - the thickness of the shell. The final offset is thickness/2 to each side of the inferred surface.
    @returns:
        liftProc        - the lifted skeleton
    @example_usage:
        liftProcedure = UniformTPMSShellViaMixedMinimal(skel, 0.03)

Spheres(skel, thickness)
    @description:
        Procedure to lift the input skeleton to a 3D volumetric structure by instantiating a sphere of the given radius centered at vertex p, for each vertex in the skeleton.
    @requirements:
        The skeleton must only contain standalone vertices; no polylines or curves can be used.
    @params:
        skel            - the skeleton to lift
        thickness       - the sphere radius 
    @returns:
        liftProc        - the lifted skeleton
    @example_usage:
        s_lift = Spheres(skel, 0.25)


======= Tile Creation ========
Tile(lifted_skeletons, embedding)
    @description:
        Procedure to embed a copy of the skeleton in R^3 using the provided embedding information. The embedding information can be computed by calling the "embed" method of the relevant CP. 
    @requirements:
        The embedding information must correspond to the same CP against which the vertices were defined. For example, if the vertices are defined relative to the cuboid, you must use the cuboid.embed() method.
    @params:
        lifted_skeletons- a list of lifted skeleton entities to embed in R^3. All entities must reside in the same CP type, and this type must have N corners.
        embedding       - information about how to embed the CP and its relative skeletons within R^3. Obtained using the CP's embed() method
    @returns:
        tile            - the new tile object
    @example_usage:
        embedding = cuboid.embed(side_len, side_len, side_len, cornerAtAABBMin=cuboid.corners.FRONT_BOTTOM_LEFT)
        s_tile = Tile([beams, shell], embedding)


======= Patterning Procedures ========
TetFullMirror()
    @description:
        Procedure which uses only mirrors to duplicate a tet-based tile such that it partitions R^3
    @params:
        N/A
    @returns:
        pat     - the patterning procedure
    @example_usage:
        pat = TetFullMirror()

TriPrismFullMirror()
    @description:
        Procedure which uses only mirrors to duplicate a triangular prism-based tile such that it partitions R^3
    @params:
        N/A
    @returns:
        pat     - the patterning procedure
    @example_usage:
        pat = TriPrismFullMirror()

CuboidFullMirror()
    @description:
        Procedure which uses only mirrors to duplicate an axis-aligned cuboid tile such that it fills a unit cube,  such that it partitions R^3. Eligible cuboid CPs must be such that all dimensions are 1/(2^k) for some positive integer k.
    @params:
        N/A
    @returns:
        pat     - the patterning procedure
    @example_usage:
        pat = CuboidFullMirror()

Identity()
    @description:
        No-op patterning procedure.
    @params:
        N/A
    @returns:
        pat     - the patterning procedure
    @example_usage:
        pat = Identity()

Custom(patternOp)
    @description:
        Environment used to compose a custom patterning procedure. Currently only implemented for the Cuboid CP.
    @params:
        patternOp- outermost pattern operation in the composition
    @returns:
        pat     - the complete patterning procedure
    @example_usage:
        pat = Custom(Rotate180([cuboid.edges.BACK_RIGHT, cuboid.edges.BACK_LEFT], True,
                        Rotate180([cuboid.edges.TOP_RIGHT], True)))

Mirror(entity, doCopy, patternOp)
    @description:
        Pattern operation specifying a mirror over the provided CP entity, which must be a CP Face. Can only be used inside of a Custom patterning environment.
    @params:
        entity   - CP Face that serves as the mirror plane. 
        doCopy   - boolean. When True, applies the operation to a copy of the input, such that the original and the transformed copy persist. When False, directly transforms the input.
        patternOp- [OPTIONAL] outermost pattern operation in the sub-composition, if any
    @returns:
        pat      - the composed patterning procedure, which may be used as is (within the Custom environment), or as the input for further composition
    @example_usage:
        pat = Custom(Mirror(cuboid.faces.TOP, True, 
                        Mirror(cuboid.faces.LEFT, True)))

Rotate180(entities, doCopy, patternOp)
    @description:
        Pattern operation specifying a 180 degree rotation about the provided CP entity. Can only be used inside of a Custom patterning environment.
    @params:
        entities - List of CP entities, which define the axis about which to rotate. If a single entity is provided, it must be a CP Edge. If multiple entities, they will be used to define a new entity that spans them. For example, if you provide two corners, the axis will go from one to the other. If you provide two CP Edges, the axis will reach from the midpoint of one to the midpoint of the other.
        doCopy   - boolean. When True, applies the operation to a copy of the input, such that the original and the transformed copy persist. When False, directly transforms the input.
        patternOp- [OPTIONAL] outermost pattern operation in the sub-composition, if any
    @returns:
        pat      - the composed patterning procedure, which may be used as is (within the Custom environment), or as the input for further composition
    @example_usage:
        pat = Custom(Rotate180([cuboid.edges.FRONT_LEFT, cuboid.edges.FRONT_RIGHT], True))

Translate(fromEntity, toEntity, doCopy, patternOp)
    @description:
        Pattern operation specifying a translation that effectively moves the fromEntity to the targetEntity. Can only be used inside of a Custom patterning environment.
    @params:
        fromEntity- CP Entity that serves as the origin of the translation vector. Currently only implemented for a CP Face.
        toEntity  - CP Entity that serves as the target of the translation vector. Currently only implemented for a CP Face.
        doCopy   - boolean. When True, applies the operation to a copy of the input, such that the original and the transformed copy persist. When False, directly transforms the input.
        patternOp- [OPTIONAL] outermost pattern operation in the sub-composition, if any
    @returns:
        pat      - the composed patterning procedure, which may be used as is (within the Custom environment), or as the input for further composition
    @example_usage:
        gridPat = Custom(Translate(cuboid.faces.LEFT, cuboid.faces.RIGHT, True,
                                Translate(cuboid.faces.FRONT, cuboid.faces.BACK, True)))


======= Structure Procedures ========
Structure(tile, pattern)
    @description:
        Combines local tile information (containing lifted skeletons) with the global patterning procedure to generate a complete metamaterial.
    @params:
        tile            - the tile object, which has (by construction) already been embedded in 3D space, along with all lifted skeletons it contains.
        pattern         - the patterning sequence to apply to extend this tile throughout space
    @returns:
        structure       - the new structure object
    @example_usage:
        obj = Structure(tile, pat)

Union(A, B)
    @description:
        Constructive solid geometry Boolean operation that computes the union of two input structures. The output of Union(A,B) is identical to Union(B,A)
    @params:
        A               - the first Structure to be unioned. This may be the output of Structure, Union, Subtract, or Intersect
        B               - the second Structure to be unioned. This may be the output of Structure, Union, Subtract, or Intersect
    @returns:
        structure       - the new structure object containing union(A,B)
    @example_usage:
        final_obj = Union(schwarzP_obj, Union(sphere_obj, beam_obj))

Subtract(A, B)
    @description:
        Constructive solid geometry Boolean operation that computes the difference (A - B) of two input structures. The relative input order is critical.
    @params:
        A               - the first Structure, from which B will be subtracted. This may be the output of Structure, Union, Subtract, or Intersect
        B               - the second Structure, to be subtracted from A. This may be the output of Structure, Union, Subtract, or Intersect
    @returns:
        structure       - the new structure object containing (A - B)
    @example_usage:
        final_obj = Subtract(c_obj, s_obj)

Intersect(A, B)
    @description:
        Constructive solid geometry Boolean operation that computes the intersection of two input structures, A and B. 
    @params:
        A               - the first Structure, which may be the output of Structure, Union, Subtract, or Intersect
        B               - the second Structure, which may be the output of Structure, Union, Subtract, or Intersect
    @returns:
        structure       - the new structure object containing the intersection of A and B
    @example_usage:
        final_obj = Intersect(c_obj, s_obj)




==================================
    Prebuilt Convex Polytopes
==================================
There are 3 prebuilt convex polytopes (CP) available for use: cuboid, triPrism, and tet. Each CP comprises a set of Entities, namely faces, edges and corners. 
For convenience, each individual entity can be referenced using the pattern <CP>.<entity_type>.<ENTITY_NAME>. 
For example, you can select a particular edge of the cuboid with the notation cuboid.edges.BOTTOM_RIGHT.
Each CP also has an embed() method which returns all necessary information to embed the CP within R^3.

The full list of entities and embed() method signatures for our predefined CPs are as follows:

tet.corners.{   BOTTOM_RIGHT,
                BOTTOM_LEFT,
                TOP_BACK,
                BOTTOM_BACK
            }
tet.edges.  {   BOTTOM_FRONT,
                TOP_LEFT,
                BACK,
                BOTTOM_RIGHT,
                TOP_RIGHT,
                BOTTOM_LEFT
            }
tet.faces.  {   BOTTOM,
                TOP,
                RIGHT,
                LEFT
            }
tet.embed(bounding_box_side_length)
    @description:
        Constructs the information required to embed the tet CP in R^3
    @params:
        bounding_box_side_length- length of axis-aligned bounding box containing the tet. Float in range [0,1]. Must be 1/2^k for some integer k
    @returns:
        embedding      - the embedding information. Specifically, the position in R^3 of all the CP corners. 
    @example_usage:
        side_len = 0.5 / num_tiling_unit_repeats_per_dim
        embedding = tet.embed(side_len)


triPrism.corners.{FRONT_BOTTOM_LEFT,
                FRONT_TOP,
                FRONT_BOTTOM_RIGHT,
                BACK_BOTTOM_LEFT,
                BACK_TOP,
                BACK_BOTTOM_RIGHT
            }
triPrism.edges.{FRONT_LEFT,
                FRONT_RIGHT,
                FRONT_BOTTOM,
                BACK_LEFT,
                BACK_RIGHT,
                BACK_BOTTOM,
                BOTTOM_LEFT,
                TOP,
                BOTTOM_RIGHT
            }
triPrism.faces.{FRONT_TRI,
                BACK_TRI,
                LEFT_QUAD,
                RIGHT_QUAD,
                BOTTOM_QUAD
            }
triPrism.embed(bounding_box_side_length)
    @description:
        Constructs the information required to embed the triangular prism CP in R^3
    @params:
        bounding_box_side_length - length of axis-aligned bounding box containing the triangular prism. Float in range [0,1]. Must be 1/2^k for some integer k
    @returns:
        embedding      - the embedding information. Specifically, the position in R^3 of all the CP corners. 
    @example_usage:
        side_len = 0.5 / num_tiling_unit_repeats_per_dim
        embedding = triPrism.embed(side_len)


cuboid.corners.{FRONT_BOTTOM_LEFT,
                FRONT_BOTTOM_RIGHT,
                FRONT_TOP_LEFT,
                FRONT_TOP_RIGHT,
                BACK_BOTTOM_LEFT,
                BACK_BOTTOM_RIGHT,
                BACK_TOP_LEFT,
                BACK_TOP_RIGHT
            }
cuboid.edges.{  FRONT_BOTTOM,
                FRONT_LEFT,
                FRONT_TOP,
                FRONT_RIGHT,
                BACK_BOTTOM,
                BACK_LEFT,
                BACK_TOP,
                BACK_RIGHT,
                BOTTOM_LEFT,
                TOP_LEFT,
                TOP_RIGHT,
                BOTTOM_RIGHT
            }
cuboid.faces.{  FRONT,
                BACK,
                TOP,
                BOTTOM,
                LEFT,
                RIGHT
            }
            
cuboid.embed(width, height, depth, cornerAtMinPt)
    @description:
        Constructs the information required to embed the cuboid CP in R^3
    @params:
        width          - length of cuboid side from left to right. float in range [0,1]. Must be 1/2^k for some integer k
        height         - length of cuboid side from top to bottom. float in range [0,1]. Must be 1/2^k for some integer k
        depth          - length of cuboid side from front to back. float in range [0,1]. Must be 1/2^k for some integer k
        cornerAtMinPt  - CP corner entity (e.g., cuboid.corners.FRONT_BOTTOM_LEFT) that should be collocated with the cuboid's minimum position in R^3
    @returns:
        embedding      - the embedding information. Specifically, the position in R^3 of all the CP corners. 
    @example_usage:
        side_len = 0.5 / num_tiling_unit_repeats_per_dim
        embedding = cuboid.embed(side_len, side_len, side_len, cornerAtAABBMin=cuboid.corners.FRONT_BOTTOM_LEFT)

cuboid.embed_via_minmax(aabb_min_pt, aabb_max_pt, cornerAtMinPt)
    @description:
        Constructs the information required to embed the cuboid CP in R^3
    @params:
        aabb_min_pt    - Minimum point of the cuboid, in R^3. Given as a list of length 3, where each component must be a float in range [0,1], with 1/2^k for some integer k
        aabb_max_pt    - Maximum point of the cuboid, in R^3. Given as a list of length 3, where each component must be a float in range [0,1], with 1/2^k for some integer k
        cornerAtMinPt  - CP corner entity (e.g., cuboid.corners.FRONT_BOTTOM_LEFT) that should be collocated with the cuboid's minimum position in R^3
    @returns:
        embedding      - the embedding information. Specifically, the position in R^3 of all the CP corners. 
    @example_usage:
        side_len = 0.5 / num_tiling_unit_repeats_per_dim
        embedding = cuboid.embed([0,0,0], [side_len, side_len, side_len], cuboid.corners.BACK_BOTTOM_RIGHT)
"""

omni_system_prompt = universal_system_prompt_template.format(api_description=code_api_description)

rendered_views_templates = {
    "top": "Top: <[{image}]>",
    "front": "Front: <[{image}]>",
    "right": "Right: <[{image}]>",
    "top_right": "Angled (Front-Top-Right): <[{image}]>"
}

generate_from_image_template = """
# Task:
Analyze these views of a metamaterial, then generate a metamaterial DSL procedure to reproduce it.

# Inputs:
**Rendered Views:**
{rendered_views}

# Output Format:
Generate a Metagen program within a python code block:

{lang_template}

"""

dsl_code_template = """```python
from metagen import *

def make_structure({model_args}) -> Structure:
    ...
```"""

def generate_formatted_benchmarks(db: Database, data_root = 's3://metagen-datasets/v3', system_prompt = omni_system_prompt):
    benchmark_dir = db.path('/benchmark')
    formatted_benchmark_dir = db.path('/workspace/training_data')
    llms = [('nova', LLMFormat.NOVA), ('llava', LLMFormat.LLAVA)]
    for dirpath, _, filenames in os.walk(benchmark_dir):
        for filename in filenames:
            if filename.endswith('.jsonl'):
                with open(os.path.join(dirpath, filename), 'r') as f:
                    tasks = [json.loads(line) for line in f.readlines()]
                is_test = 'test' in filename
                local_dir = dirpath[len(benchmark_dir)+1:]
                local_path = os.path.join(local_dir, filename)
                for llm_name, llm in llms:
                    converted_path = os.path.join(formatted_benchmark_dir, llm_name, local_path)
                    os.makedirs(os.path.dirname(converted_path), exist_ok=True)
                    print(f'Converting {dirpath + '/' + filename} to {llm_name} format')
                    with open(converted_path, 'w') as f:
                        for task in tasks:
                            formatted_task = format_task(task, llm, db, is_test, system_prompt, data_root)
                            f.write(json.dumps(formatted_task) + '\n')
