# Adapted from: https://github.com/locuslab/edge-of-stability
# Original paper: Cohen, J. M., Kaur, S., Li, Y., Kolter, J. Z., & Talwalkar, A. (2021).
# "Gradient Descent on Neural Networks Typically Occurs at the Edge of Stability", ICLR 2021.
# If you use this code, please cite the original work.
#
# Our modifications: lines 9, 17, 21, 25-33, 66-70, 83-89, 94-100, 103-119, 123-126, 136-138, 144-156, 162-163, 171-175, 208-213, 219-220

from os import makedirs
import os

from data import DATASETS
import torch
import torch.nn as nn
from torch.nn.utils import parameters_to_vector, vector_to_parameters
from torch.utils.data import Dataset
import argparse
import math

from archs import load_architecture
from utilities import get_flow_directory, get_loss_and_acc, compute_losses, \
    save_files, save_files_es, get_norm, AtParams, compute_gradient, get_hessian_eigenvalues, DEFAULT_PHYS_BS
from data import load_dataset, take_first


def int_or_str(value):
    try:
        return int(value)
    except ValueError:
        if value in {"nuc", "fro", "all"}:
            return value
        raise argparse.ArgumentTypeError(f"Invalid value: {value}. Must be an integer, 'nuc', or 'fro'.")


def rk_step(network: nn.Module, loss_fn: nn.Module, dataset: Dataset, step_size: float,
            physical_batch_size: int = DEFAULT_PHYS_BS):
    """ Take a Runge-Kutta step with a given step size. """
    theta = parameters_to_vector(network.parameters())

    def f(x: torch.Tensor):
        with AtParams(network, x):
            fx = -compute_gradient(network, loss_fn, dataset, physical_batch_size=physical_batch_size)
        return fx

    k1 = f(theta)
    k2 = f(theta + (step_size / 2) * k1)
    k3 = f(theta + (step_size / 2) * k2)
    k4 = f(theta + step_size * k3)

    theta_next = theta + (step_size / 6) * (k1 + 2 * k2 + 2 * k3 + k4)
    vector_to_parameters(theta_next, network.parameters())


def rk_advance_time(network: nn.Module, loss_fn: nn.Module, dataset: Dataset, T: float,
                    rk_step_size: float, physical_batch_size: int):
    """ Using the Runge-Kutta algorithm, numerically integrate the gradient flow ODE for time T, using a given
     Runge-Kutta step size."""
    T_remaining = T
    while T_remaining > 0:
        this_step_size = min(rk_step_size, T_remaining)
        rk_step(network, loss_fn, dataset, this_step_size, physical_batch_size)
        T_remaining -= rk_step_size


def main(dataset: str, arch_id: str, loss: str, max_time: float, tick: float, neigs=0, physical_batch_size=1000,
         abridged_size: int = 5000, eig_freq: int = -1, iterate_freq: int = -1, save_freq: int = -1, alpha: float = 1.0,
         nproj: int = 0, loss_goal: float = None, acc_goal: float = None, max_step_size: int = 999, seed: int = 0, 
         norm_input = "all", init_scaling: float = 1.0):
    directory = os.path.expanduser(get_flow_directory(dataset, arch_id, seed, loss, tick))
    if not init_scaling == 1.0:
        directory = directory + f"/scaling_{init_scaling}"
    print(f"output directory: {directory}")
    makedirs(directory, exist_ok=True)

    train_dataset, test_dataset = load_dataset(dataset, loss)
    abridged_train = take_first(train_dataset, abridged_size)

    loss_fn, acc_fn = get_loss_and_acc(loss)

    max_steps = int(max_time / tick)

    torch.manual_seed(seed)
    network = load_architecture(arch_id, dataset).cuda()
    if os.path.exists(directory+"/num_iterations"):
        num_iter = torch.load(directory+"/num_iterations", weights_only=False)
    else:
        num_iter = 0
        with torch.no_grad():  # Disable gradient tracking for this operation
            for param in network.parameters():
                param.mul_(init_scaling)

    torch.manual_seed(7)
    projectors = torch.randn(nproj, len(parameters_to_vector(network.parameters())))

    if norm_input == "all":
        norm_types = [1,"fro","nuc"]
    else:
        norm_types == [norm_input]

    if num_iter == 0:
        times, train_loss, test_loss, train_acc, test_acc = \
        torch.zeros(max_steps), torch.zeros(max_steps), torch.zeros(max_steps), \
        torch.zeros(max_steps), torch.zeros(max_steps)
        iterates = torch.zeros(math.ceil(max_steps / iterate_freq) if iterate_freq > 0 else 0, len(projectors))
        eigs = torch.zeros(math.ceil(max_steps / eig_freq) if eig_freq > 0 else 0, neigs)
        norms = torch.zeros(math.ceil(max_steps / eig_freq) if eig_freq > 0 else 0, len(norm_types))
    else:
        if not all([os.path.exists(directory+"/"+st) for st in ["full_model_final", "eigs", "iterates", "times", "test_acc", "train_acc", "train_loss", "norms"]]):
            raise FileNotFoundError("Number of past iterations is nonzero but a saved file does not exist")
        else:
            network = torch.load(directory+"/full_model_final", weights_only=False)
            times = torch.cat((torch.load(directory+"/times", weights_only=False)[:num_iter], torch.zeros(max_steps)))
            train_loss = torch.cat((torch.load(directory+"/train_loss", weights_only=False)[:num_iter], torch.zeros(max_steps)))
            test_loss = torch.cat((torch.load(directory+"/test_loss", weights_only=False)[:num_iter], torch.zeros(max_steps)))
            train_acc = torch.cat((torch.load(directory+"/train_acc", weights_only=False)[:num_iter], torch.zeros(max_steps)))
            test_acc = torch.cat((torch.load(directory+"/test_acc", weights_only=False)[:num_iter], torch.zeros(max_steps)))
            iterates = torch.cat((torch.load(directory+"/iterates", weights_only=False)[:math.ceil(num_iter / iterate_freq)], torch.zeros(math.ceil(num_iter + max_steps / iterate_freq) - math.ceil(num_iter / iterate_freq) if iterate_freq > 0 else 0, len(projectors))), dim=0)
            eigs = torch.cat((torch.load(directory+"/eigs", weights_only=False)[:math.ceil(num_iter / eig_freq)], torch.zeros(math.ceil(num_iter + max_steps / eig_freq) - math.ceil(num_iter / eig_freq) if eig_freq > 0 else 0, neigs)), dim=0)
            norms = torch.cat((torch.load(directory+"/norms", weights_only=False)[:math.ceil(num_iter / eig_freq)], torch.zeros(math.ceil(num_iter + max_steps / eig_freq) - math.ceil(num_iter / eig_freq) if eig_freq > 0 else 0, len(norm_types))), dim=0)
            print("Successfully loaded all files")

    sharpness = float('inf')

    ls, _ = compute_losses(network, [loss_fn, acc_fn], train_dataset, physical_batch_size)
    next_power = 10**(math.floor(math.log10(ls)) )

    for step in range(num_iter, num_iter+max_steps):
        times[step] = step * tick
        train_loss[step], train_acc[step] = compute_losses(network, [loss_fn, acc_fn], train_dataset,
                                                           physical_batch_size)
        test_loss[step], test_acc[step] = compute_losses(network, [loss_fn, acc_fn], test_dataset, physical_batch_size)

        if eig_freq != -1 and step % eig_freq == 0:
            eigs[step // eig_freq, :] = get_hessian_eigenvalues(network, loss_fn, abridged_train, neigs=neigs,
                                                                physical_batch_size=physical_batch_size)
            sharpness = eigs[step // eig_freq, 0]
            for i in range(len(norm_types)):
                norms[step // eig_freq, i] = get_norm(network, norm_types[i])
            print("sharpness: ", sharpness, ", norms: ", norms[step // eig_freq, :])

        if iterate_freq != -1 and step % iterate_freq == 0:
            iterates[step // iterate_freq, :] = projectors.mv(parameters_to_vector(network.parameters()).cpu().detach())

        if save_freq != -1 and step % save_freq == 0:
            save_files(directory, [("full_model_final", network), ("norms", norms[:(step // eig_freq)+1,:]),
                                   ("eigs", eigs[:(step // eig_freq)+1]), ("iterates", iterates[:(step // iterate_freq)+1]),
                                   ("train_loss", train_loss[:step+1]), ("test_loss", test_loss[:step+1]),
                                   ("train_acc", train_acc[:step+1]), ("test_acc", test_acc[:step+1]),
                                   ("times", times[:step+1]), ("num_iterations", step+1)])
            
        if train_loss[step] <= next_power:
            save_files_es(directory, next_power, [("full_model_final", network), ("norms", norms[:(step // eig_freq)+1,:]),
                                   ("eigs", eigs[:(step // eig_freq)+1]), ("iterates", iterates[:(step // iterate_freq)+1]),
                                   ("train_loss", train_loss[:step+1]), ("test_loss", test_loss[:step+1]),
                                   ("train_acc", train_acc[:step+1]), ("test_acc", test_acc[:step+1]),
                                   ("times", times[:step+1]), ("num_iterations", step+1)])
            next_power = next_power/10

        if (loss_goal is not None and train_loss[step] < loss_goal) or \
                (acc_goal is not None and train_acc[step] > acc_goal):
            break

        print(f"{times[step]:.4f}\t{train_loss[step]:.4f}\t{train_acc[step]:.4f}"
              f"\t{test_loss[step]:.4f}\t{test_acc[step]:.4f}", flush=True)

        if (loss_goal != None and train_loss[step] < loss_goal) or (acc_goal != None and train_acc[step] > acc_goal):
            break

        rk_step_size = min(alpha / sharpness, max_step_size)
        rk_advance_time(network, loss_fn, train_dataset, tick, rk_step_size, physical_batch_size)

    save_files(directory, [("full_model_final", network), ("norms", norms[:(step // eig_freq)+1,:]),
                                   ("eigs", eigs[:(step // eig_freq)+1]), ("iterates", iterates[:(step // iterate_freq)+1]),
                                   ("train_loss", train_loss[:step+1]), ("test_loss", test_loss[:step+1]),
                                   ("train_acc", train_acc[:step+1]), ("test_acc", test_acc[:step+1]),
                                   ("times", times[:step+1]), ("num_iterations", step+1)])


if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Train using gradient flow.")
    parser.add_argument("dataset", type=str, choices=DATASETS, help="which dataset to train")
    parser.add_argument("arch_id", type=str, help="which network architectures to train")
    parser.add_argument("loss", type=str, choices=["ce", "mse"], help="which loss function to use")
    parser.add_argument("tick", type=float,
                        help="the train / test losses and accuracies will be computed and saved every tick units of time")
    parser.add_argument("max_time", type=float, help="the maximum time (ODE time, not wall clock time) to train for")
    parser.add_argument("--alpha", type=float, default=1.0,
                        help=" the Runge-Kutta step size is min(alpha / [estimated sharpness], max_step_size).")
    parser.add_argument("--max_step_size", type=float, default=999,
                        help=" the Runge-Kutta step size is min(alpha / [estimated sharpness], max_step_size)")
    parser.add_argument("--seed", type=int, help="the random seed used when initializing the network weights",
                        default=0)
    parser.add_argument("--beta", type=float, help="momentum parameter (used if opt = polyak or nesterov)")
    parser.add_argument("--physical_batch_size", type=int,
                        help="the maximum number of examples that we try to fit on the GPU at once", default=1000)
    parser.add_argument("--acc_goal", type=float,
                        help="terminate training if the train accuracy ever crosses this value")
    parser.add_argument("--loss_goal", type=float, help="terminate training if the train loss ever crosses this value")
    parser.add_argument("--neigs", type=int, help="the number of top eigenvalues to compute")
    parser.add_argument("--eig_freq", type=int, default=-1,
                        help="the frequency at which we compute the top Hessian eigenvalues (-1 means never)")
    parser.add_argument("--nproj", type=int, default=0, help="the dimension of random projections")
    parser.add_argument("--iterate_freq", type=int, default=-1,
                        help="the frequency at which we save random projections of the iterates")
    parser.add_argument("--abridged_size", type=int, default=5000,
                        help="when computing top Hessian eigenvalues, use an abridged dataset of this size")
    parser.add_argument("--save_freq", type=int, default=-1,
                        help="the frequency at which we save resuls")
    parser.add_argument("--norm_type", type=int_or_str, default="all",
                        help="which type of norm should be computed during training: integer, nuc, or fro, use all for [1,2,nuc]")
    parser.add_argument("--es_acc_goal", type=float,
                        help="save model as early stopping if the train accuracy ever crosses this value")
    parser.add_argument("--es_loss_goal", type=float, help="save model as early stopping if the train loss ever crosses this value")
    parser.add_argument("--init_scaling", type=float, help="multiply initial weights by this", default=1.0)
    args = parser.parse_args()

    main(dataset=args.dataset, arch_id=args.arch_id, loss=args.loss, max_time=args.max_time, tick=args.tick,
         neigs=args.neigs, physical_batch_size=args.physical_batch_size, abridged_size=args.abridged_size,
         eig_freq=args.eig_freq, iterate_freq=args.iterate_freq, save_freq=args.save_freq, nproj=args.nproj,
         loss_goal=args.loss_goal, acc_goal=args.acc_goal, seed=args.seed, norm_input=args.norm_type,
         es_loss_goal=args.es_loss_goal, es_acc_goal=args.es_acc_goal, init_scaling=args.init_scaling)
