#!/bin/bash
#
# This script runs the main Deployment Gate experiment for the TRACE artifact.
# It trains a source model, fine-tunes 20 candidate models, and computes
# all gate scores, saving the results to a CSV file.
#
# Usage:
#   bash run_deployment_gate.sh
#
# The script will produce a timestamped output directory inside 'outputs/'.

set -e

echo "================================================================"
echo "Starting TRACE Deployment Gate Experiment on DomainNet"
echo "================================================================"

# --- Conda environment activation ---
# Make sure conda is initialized (adjust path if necessary)
if [ -f "/opt/conda/etc/profile.d/conda.sh" ]; then
    source "/opt/conda/etc/profile.d/conda.sh"
else
    # Add other common conda paths if needed
    echo "Conda initialization script not found at /opt/conda. Make sure conda is in your PATH."
fi

conda activate trace
echo "Activated conda environment: trace"


# --- GPU Verification ---
echo "--- Verifying GPU availability ---"
if ! command -v nvidia-smi &> /dev/null
then
    echo "WARNING: nvidia-smi could not be found. Running on CPU."
    DEVICE="cpu"
else
    nvidia-smi
    DEVICE="cuda"
fi
export CUDA_VISIBLE_DEVICES=${CUDA_VISIBLE_DEVICES:-0}
echo "--- Using GPU: $CUDA_VISIBLE_DEVICES ---"


# --- Experiment Runner ---
echo -e "\n--- Running Deployment Gate experiment ---"
# Assuming there is a runner script for the gate experiment
# at src/runners/run_gate.py
python -m src.runners.run_gate \
    --config configs/domainnet_gate.yaml \
    --device $DEVICE

echo "--- Deployment Gate experiment finished ---"


echo -e "\n================================================================"
echo "Experiment completed."
echo "Results are saved in the 'outputs/' directory."
echo "You can now run the post-processing script to generate the final table."
echo "================================================================"
