"""Configuration utilities for ARCOS experiments."""

import os
import yaml
from pathlib import Path
from typing import Any, Dict, Optional


class Config:
    """Configuration manager for ARCOS experiments."""
    
    def __init__(self, config_path: str):
        """Initialize configuration from YAML file.
        
        Args:
            config_path: Path to YAML configuration file
        """
        self.config_path = Path(config_path)
        if not self.config_path.exists():
            raise FileNotFoundError(f"Config file not found: {config_path}")
        
        with open(self.config_path, 'r') as f:
            self.config = yaml.safe_load(f)
        
        self._validate_config()
    
    def _validate_config(self):
        """Validate required configuration fields."""
        required_sections = ['dataset', 'acquisition', 'model', 'training', 'ot', 'metrics']
        for section in required_sections:
            if section not in self.config:
                raise ValueError(f"Missing required config section: {section}")
    
    def get(self, key: str, default: Any = None) -> Any:
        """Get configuration value using dot notation.
        
        Args:
            key: Configuration key (e.g., 'dataset.name')
            default: Default value if key not found
            
        Returns:
            Configuration value
        """
        keys = key.split('.')
        value = self.config
        
        try:
            for k in keys:
                value = value[k]
            return value
        except (KeyError, TypeError):
            return default
    
    def get_dataset_config(self) -> Dict[str, Any]:
        """Get dataset configuration section."""
        return self.config['dataset']
    
    def get_acquisition_config(self) -> Dict[str, Any]:
        """Get acquisition configuration section."""
        return self.config['acquisition']
    
    def get_model_config(self) -> Dict[str, Any]:
        """Get model configuration section."""
        return self.config['model']
    
    def get_training_config(self) -> Dict[str, Any]:
        """Get training configuration section."""
        return self.config['training']
    
    def get_ot_config(self) -> Dict[str, Any]:
        """Get optimal transport configuration section."""
        return self.config['ot']
    
    def get_metrics_config(self) -> Dict[str, Any]:
        """Get metrics configuration section."""
        return self.config['metrics']
    
    def get_system_config(self) -> Dict[str, Any]:
        """Get system configuration section."""
        return self.config.get('system', {})
    
    def get_output_config(self) -> Dict[str, Any]:
        """Get output configuration section."""
        return self.config.get('output', {})
    
    def get_feature_cache_config(self) -> Dict[str, Any]:
        """Get feature cache configuration section."""
        return self.config.get('feature_cache', {})
    
    def update(self, updates: Dict[str, Any]):
        """Update configuration with new values.
        
        Args:
            updates: Dictionary of updates in dot notation
        """
        for key, value in updates.items():
            keys = key.split('.')
            config_section = self.config
            
            # Navigate to the parent section
            for k in keys[:-1]:
                if k not in config_section:
                    config_section[k] = {}
                config_section = config_section[k]
            
            # Set the value
            config_section[keys[-1]] = value
    
    def save(self, output_path: Optional[str] = None):
        """Save configuration to file.
        
        Args:
            output_path: Output path, defaults to original path
        """
        if output_path is None:
            output_path = self.config_path
        
        with open(output_path, 'w') as f:
            yaml.dump(self.config, f, default_flow_style=False, indent=2)
    
    def __str__(self) -> str:
        """String representation of configuration."""
        return f"Config({self.config_path})"
    
    def __repr__(self) -> str:
        """String representation of configuration."""
        return self.__str__()


def load_config(config_path: str) -> Config:
    """Load configuration from file.
    
    Args:
        config_path: Path to configuration file
        
    Returns:
        Configuration object
    """
    return Config(config_path)
