"""Seed utilities for reproducible experiments."""

import random
import numpy as np
import torch
import os
from typing import Optional


def set_seed(seed: int, deterministic: bool = True):
    """Set random seeds for reproducibility.
    
    Args:
        seed: Random seed value
        deterministic: Whether to use deterministic algorithms
    """
    random.seed(seed)
    np.random.seed(seed)
    torch.manual_seed(seed)
    torch.cuda.manual_seed(seed)
    torch.cuda.manual_seed_all(seed)
    
    if deterministic:
        torch.backends.cudnn.deterministic = True
        torch.backends.cudnn.benchmark = False
        os.environ['PYTHONHASHSEED'] = str(seed)
    
    print(f"Set seed to {seed} (deterministic: {deterministic})")


def get_random_seed() -> int:
    """Get a random seed value.
    
    Returns:
        Random seed value
    """
    return random.randint(0, 2**32 - 1)


def set_deterministic_mode(deterministic: bool = True):
    """Set deterministic mode for PyTorch.
    
    Args:
        deterministic: Whether to use deterministic algorithms
    """
    torch.backends.cudnn.deterministic = deterministic
    torch.backends.cudnn.benchmark = not deterministic
    
    if deterministic:
        os.environ['PYTHONHASHSEED'] = '0'
    else:
        os.environ.pop('PYTHONHASHSEED', None)
    
    print(f"Set deterministic mode: {deterministic}")


def seed_worker(worker_id: int):
    """Seed worker for DataLoader reproducibility.
    
    Args:
        worker_id: Worker ID
    """
    worker_seed = torch.initial_seed() % 2**32
    np.random.seed(worker_seed)
    random.seed(worker_seed)
