from __future__ import annotations

import os
import json
from dataclasses import asdict, is_dataclass
from datetime import datetime
from pathlib import Path
from typing import Any, Dict

import yaml


def ensure_dir(path: str | Path) -> Path:
    p = Path(path)
    p.mkdir(parents=True, exist_ok=True)
    return p


def timestamp_now() -> str:
    return datetime.now().strftime("%Y%m%d_%H%M%S")


def dumps_jsonable(obj: Any) -> Any:
    if is_dataclass(obj):
        return asdict(obj)
    if isinstance(obj, (list, tuple)):
        return [dumps_jsonable(o) for o in obj]
    if isinstance(obj, dict):
        return {k: dumps_jsonable(v) for k, v in obj.items()}
    return obj


def save_yaml(obj: Dict[str, Any], path: str | Path) -> None:
    p = Path(path)
    ensure_dir(p.parent)
    with p.open("w") as f:
        yaml.safe_dump(dumps_jsonable(obj), f, sort_keys=False)


def save_json(obj: Dict[str, Any], path: str | Path) -> None:
    p = Path(path)
    ensure_dir(p.parent)
    with p.open("w") as f:
        json.dump(dumps_jsonable(obj), f, indent=2)


def save_text(text: str, path: str | Path) -> None:
    p = Path(path)
    ensure_dir(p.parent)
    with p.open("w") as f:
        f.write(text)


def save_csv(df_obj: Any, path: str | Path) -> None:
    """Save a pandas DataFrame-like to CSV."""
    p = Path(path)
    ensure_dir(p.parent)
    df_obj.to_csv(p, index=False)


