"""
Utilities for deterministic seeding across numpy, torch, and random.

Favor CPU to ensure bitwise reproducibility across environments.
"""
from __future__ import annotations

import os
import random
from typing import Optional

import numpy as np

try:
    import torch
except Exception:  # pragma: no cover - torch may not be available during some checks
    torch = None  # type: ignore


def seed_all(seed: int, deterministic_torch: bool = True) -> None:
    """Seed numpy, random, and torch.

    Parameters
    ----------
    seed: int
        Global seed value.
    deterministic_torch: bool
        If True, set PyTorch deterministic flags for reproducibility.
    """
    os.environ["PYTHONHASHSEED"] = str(seed)
    # Ensure deterministic cuBLAS behavior when using GPU with deterministic algorithms
    if deterministic_torch and "CUBLAS_WORKSPACE_CONFIG" not in os.environ:
        # Recommended configs: ':4096:8' (more memory) or ':16:8' (less memory)
        os.environ["CUBLAS_WORKSPACE_CONFIG"] = ":4096:8"
    random.seed(seed)
    np.random.seed(seed)

    if torch is not None:
        torch.manual_seed(seed)
        torch.use_deterministic_algorithms(deterministic_torch)
        if torch.cuda.is_available():
            torch.cuda.manual_seed_all(seed)
            torch.backends.cudnn.deterministic = deterministic_torch
            torch.backends.cudnn.benchmark = False


def numpy_rng(seed: int) -> np.random.Generator:
    """Return a new numpy Generator with the given seed."""
    return np.random.default_rng(seed)


def torch_device(prefer_cuda: bool = False) -> Optional["torch.device"]:
    """Return a torch.device, preferring CPU for reproducibility unless prefer_cuda.

    Parameters
    ----------
    prefer_cuda: bool
        If True and CUDA is available, return cuda device. Otherwise CPU.
    """
    if torch is None:
        return None
    if prefer_cuda and torch.cuda.is_available():
        return torch.device("cuda")
    return torch.device("cpu")


