import os
# os.environ['CUDA_VISIBLE_DEVICES'] = '7,5,6'

import numpy as np
import time
import torch
from tqdm import tqdm
from transformers import T5Tokenizer, T5ForConditionalGeneration, set_seed

from nltk.tokenize import sent_tokenize
from detectors import RoBERTaAIDetector
from my_utils.test_utils import get_generated_text_saving_dir
from my_utils.data_utils import save_list_to_tsv
from my_utils.my_dataloader import load_test_data

set_seed(42)

class DipperParaphraser(object):
    def __init__(self, model="kalpeshk2011/dipper-paraphraser-xxl", verbose=True):
        time1 = time.time()
        self.tokenizer = T5Tokenizer.from_pretrained(model) # 'google/t5-v1_1-xxl'
        self.model = T5ForConditionalGeneration.from_pretrained(model, device_map='balanced_low_0')
        if verbose:
            print(f"{model} model loaded in {time.time() - time1}")
        # self.model.cuda()
        self.model.eval()

    def paraphrase(self, input_text, lex_diversity, order_diversity, prefix="", sent_interval=3, **kwargs):
        """Paraphrase a text using the DIPPER model.

        Args:
            input_text (str): The text to paraphrase. Make sure to mark the sentence to be paraphrased between <sent> and </sent> blocks, keeping space on either side.
            lex_diversity (int): The lexical diversity of the output, choose multiples of 20 from 0 to 100. 0 means no diversity, 100 means maximum diversity.
            order_diversity (int): The order diversity of the output, choose multiples of 20 from 0 to 100. 0 means no diversity, 100 means maximum diversity.
            **kwargs: Additional keyword arguments like top_p, top_k, max_length.
        """
        assert lex_diversity in [0, 20, 40, 60, 80, 100], "Lexical diversity must be one of 0, 20, 40, 60, 80, 100."
        assert order_diversity in [0, 20, 40, 60, 80, 100], "Order diversity must be one of 0, 20, 40, 60, 80, 100."

        lex_code = int(100 - lex_diversity)
        order_code = int(100 - order_diversity)

        input_text = " ".join(input_text.split())
        sentences = sent_tokenize(input_text)
        prefix = " ".join(prefix.replace("\n", " ").split())
        output_text = ""

        for sent_idx in range(0, len(sentences), sent_interval):
            curr_sent_window = " ".join(sentences[sent_idx:sent_idx + sent_interval])
            final_input_text = f"lexical = {lex_code}, order = {order_code}"
            if prefix:
                final_input_text += f" {prefix}"
            final_input_text += f" <sent> {curr_sent_window} </sent>"

            final_input = self.tokenizer([final_input_text], return_tensors="pt")
            final_input = {k: v.cuda() for k, v in final_input.items()}

            with torch.inference_mode():
                outputs = self.model.generate(**final_input, **kwargs)
            outputs = self.tokenizer.batch_decode(outputs, skip_special_tokens=True)
            prefix += " " + outputs[0]
            output_text += " " + outputs[0]

        return output_text.rstrip()

if __name__ == "__main__":
    dataset = ''
    task_type = 'paraphrase'

    # load test
    test_data_list = load_test_data(dataset, task_type)

    dipper_model_path = "/path/to/para-paraphrase-ctx-t5-xxl"
    # run paraphrase
    dp = DipperParaphraser(model=dipper_model_path)

    new_text_list = []
    for ai_text in tqdm(test_data_list):
        # L60, O60,  sample with top_p=0.75
        new_text = dp.paraphrase(ai_text, lex_diversity=60, order_diversity=60, prefix="", do_sample=True, top_p=0.75, top_k=None, max_length=512)
        new_text_list.append(new_text)


    save_data_list = list(zip(test_data_list, new_text_list))
    save_data_list.insert(0, ['ai', 'new'])

    save_dir = get_generated_text_saving_dir(dataset, 'dipper')
    save_list_to_tsv(save_data_list, save_dir)
