import matplotlib.pyplot as plt

# These are the "Tableau 20" colors as RGB.
tableau20 = [(31, 119, 180), (174, 199, 232),
             (255, 127, 14), (255, 187, 120),
             (44, 160, 44), (152, 223, 138),
             (214, 39, 40), (255, 152, 150),
             (148, 103, 189), (197, 176, 213),
             (140, 86, 75), (196, 156, 148),
             (227, 119, 194), (247, 182, 210),
             (127, 127, 127), (199, 199, 199),
             (188, 189, 34), (219, 219, 141), (23, 190, 207), (158, 218, 229)]

# Scale the RGB values to the [0, 1] range, which is the format matplotlib accepts.
for i in range(len(tableau20)):
    r, g, b = tableau20[i]
    tableau20[i] = (r / 255., g / 255., b / 255.)

# You typically want your plot to be ~1.33x wider than tall. This plot is a rare
# exception because of the number of lines being plotted on it.
# Common sizes: (10, 7.5) and (12, 9)
plt.figure(figsize=(12, 8))

# Remove the plot frame lines. They are unnecessary chartjunk.
ax = plt.subplot(111)
# ax.spines["top"].set_visible(False)
# ax.spines["right"].set_visible(False)


# Make sure your axis ticks are large enough to be easily read.
# You don't want your viewers squinting to read your plot.
accuracy = [x / 100 for x in range(30, 100, 10)]
plt.yticks(accuracy, fontsize=14)

percentiles = [percentile for percentile in range(0, 91, 10)]
plt.xticks(percentiles, fontsize=14)

plt.xlabel('% of input pixels removed', fontsize=14)
plt.ylabel('Test Accuracy', fontsize=14)

# Limit the range of the plot to only where the data is. Avoid unnecessary whitespace.
plt.ylim(0.3, 1)
plt.xlim(0, 100)

# Provide tick lines across the plot to help your viewers trace along
# the axis ticks. Make sure that the lines are light and small so they
# don't obscure the primary data lines.
plt.grid(True)

# Remove the tick marks; they are unnecessary with the tick lines we just plotted.
plt.tick_params(axis="both", which="both", bottom="off", top="off",
                labelbottom="on", left="off", right="off", labelleft="on")

initial_accuracy = [84.36]
prune_grad_abs_66_5 = [75.130, 69.870, 66.540, 65.320, 54.860]
prune_pgd_abs_66_5  = [74.150, 69.020, 67.420, 63.960, 54.680]
prune_grad_abs_81_5 = [74.990, 69.340, 64.490, 58.090, 48.840]
prune_pgd_abs_81_5  = [74.640, 69.200, 65.920, 61.380, 50.970]
prune_grad_abs_86_5 = [74.560, 70.420, 65.260, 58.980, 50.410]
prune_pgd_abs_86_5 = [75.110, 67.330, 61.710, 56.900, 47.980]

def convert_to_accuracy(acc):
    return [a / 100 for a in acc]

grads = {
        'prune_grad_abs_66.5': (prune_grad_abs_66_5, 'PruneGrad_5%', 6, None, 'o'),
        'prune_pgd_abs_66.5': (prune_pgd_abs_66_5, 'PrunePGD_5%', 8, '--', '*'),
        'prune_grad_abs_81.5': (prune_grad_abs_81_5, 'PruneGrad_10%', 10, None, 'o'),
        'prune_pgd_abs_81.5': (prune_pgd_abs_81_5, 'PrunePGD_10%', 12, '--', '*'),
        'prune_grad_abs_86.5': (prune_grad_abs_86_5, 'PruneGrad_15%', 14, None, 'o'),
        'prune_pgd_abs_86.5': (prune_pgd_abs_86_5, 'PrunePGD_15%', 16, '--', '*'),
}

for entry in grads:
    accuracy, attribution_name, color_map_index, linestyle, marker = grads[entry]
    # Plot each line separately with its own color, using the Tableau 20
    # color set in order.
    plt.plot([0, 10, 30, 50, 70, 90][:len(accuracy) + 1],
             convert_to_accuracy(initial_accuracy + accuracy),
             lw=2,
             color=tableau20[color_map_index],
             label=attribution_name,
             linestyle=linestyle,
             marker=marker)

plt.legend()

# matplotlib's title() call centers the title on the plot, but not the graph,
# so I used the text() call to customize where the title goes.

# Make the title big enough so it spans the entire plot, but don't make it
# so big that it requires two lines to show.

# Note that if the title is descriptive enough, it is unnecessary to include
# axis labels; they are self-evident, in this plot's case.
# plt.text(50,
#          95,
#          "CIFAR10",
#          fontsize=17,
#          ha="center")

# Finally, save the figure as a PNG.
# You can also save it as a PDF, JPEG, etc.
# Just change the file extension in this call.
# bbox_inches="tight" removes all the extra whitespace on the edges of your plot.
plt.savefig("Roar-Cifar10-Resnet8-different-thresholds.png", bbox_inches="tight")
plt.show()
