import matplotlib.pyplot as plt

# These are the "Tableau 20" colors as RGB.
tableau20 = [(31, 119, 180), (174, 199, 232),
             (255, 127, 14), (255, 187, 120),
             (44, 160, 44), (152, 223, 138),
             (214, 39, 40), (255, 152, 150),
             (148, 103, 189), (197, 176, 213),
             (140, 86, 75), (196, 156, 148),
             (227, 119, 194), (247, 182, 210),
             (127, 127, 127), (199, 199, 199),
             (188, 189, 34), (219, 219, 141), (23, 190, 207), (158, 218, 229)]

# Scale the RGB values to the [0, 1] range, which is the format matplotlib accepts.
for i in range(len(tableau20)):
    r, g, b = tableau20[i]
    tableau20[i] = (r / 255., g / 255., b / 255.)

# You typically want your plot to be ~1.33x wider than tall. This plot is a rare
# exception because of the number of lines being plotted on it.
# Common sizes: (10, 7.5) and (12, 9)
plt.figure(figsize=(12, 8))

# Remove the plot frame lines. They are unnecessary chartjunk.
ax = plt.subplot(111)
# ax.spines["top"].set_visible(False)
# ax.spines["right"].set_visible(False)


# Make sure your axis ticks are large enough to be easily read.
# You don't want your viewers squinting to read your plot.
accuracy = [x / 100 for x in range(30, 100, 10)]
plt.yticks(accuracy, fontsize=14)

percentiles = [percentile for percentile in range(0, 91, 10)]
plt.xticks(percentiles, fontsize=14)

plt.xlabel('% of input pixels removed', fontsize=14)
plt.ylabel('Test Accuracy', fontsize=14)

# Limit the range of the plot to only where the data is. Avoid unnecessary whitespace.
plt.ylim(0.3, 1)
plt.xlim(0, 100)

# Provide tick lines across the plot to help your viewers trace along
# the axis ticks. Make sure that the lines are light and small so they
# don't obscure the primary data lines.
plt.grid(True)

# Remove the tick marks; they are unnecessary with the tick lines we just plotted.
plt.tick_params(axis="both", which="both", bottom="off", top="off",
                labelbottom="on", left="off", right="off", labelleft="on")

initial_accuracy = [84.36]
vanilla_saliency = [79.100, 81.100, 82.810, 82.260, 75.500]
gradxinput = [79.220, 79.850, 82.540, 81.580, 74.470]
GuidedBackprop = [79.240, 76.830, 78.020, 78.360, 74.780]
GradCam = [74.650, 69.990, 63.270, 58.030, 47.580]
RectGrad = [78.050, 79.710, 78.170, 78.340, 75.670]
IntegratedGradient = [80.800, 81.980, 83.370, 83.230, 80.900]
prune_grad = [74.560, 70.420, 65.260, 58.980, 50.410]
prune_pgd = [75.110, 67.330, 61.710, 56.900, 47.980]

def convert_to_accuracy(acc):
    return [a / 100 for a in acc]


grads = [
    ('VanillaGradient', 2, vanilla_saliency),
    ('IntegratedGradient', 4, IntegratedGradient),
    ('Grad*Input', 6, gradxinput),
    ('GuidedBackprop',8, GuidedBackprop),
    ('GradCam',10, GradCam),
    ('RectGrad',12, RectGrad),
    ('PruneGrad', 14, prune_grad),
    ('PrunePGD', 16, prune_pgd),
]

for entry in grads:
    attribution_name, color_map_index, attribution_val = entry
    # Plot each line separately with its own color, using the Tableau 20
    # color set in order.
    if attribution_name == 'PruneGrad':
        plt.plot([0, 10, 30, 50, 70, 90][:len(attribution_val)+1],
                 convert_to_accuracy(initial_accuracy + attribution_val),
                 lw=2,
                 color=tableau20[color_map_index],
                 label=attribution_name,
                 linestyle='--', marker='o')
    elif attribution_name == 'PrunePGD':
        plt.plot([0, 10, 30, 50, 70, 90][:len(attribution_val) + 1],
                 convert_to_accuracy(initial_accuracy + attribution_val),
                 lw=2,
                 color=tableau20[color_map_index],
                 label=attribution_name,
                 linestyle='--', marker='*')
    else:
        plt.plot([0, 10, 30, 50, 70, 90][:len(attribution_val) + 1],
                 convert_to_accuracy(initial_accuracy + attribution_val),
                 lw=2,
                 color=tableau20[color_map_index],
                 label=attribution_name)

plt.legend()

# matplotlib's title() call centers the title on the plot, but not the graph,
# so I used the text() call to customize where the title goes.

# Make the title big enough so it spans the entire plot, but don't make it
# so big that it requires two lines to show.

# Note that if the title is descriptive enough, it is unnecessary to include
# axis labels; they are self-evident, in this plot's case.
# plt.text(50,
#          95,
#          "CIFAR10",
#          fontsize=17,
#          ha="center")

# Finally, save the figure as a PNG.
# You can also save it as a PDF, JPEG, etc.
# Just change the file extension in this call.
# bbox_inches="tight" removes all the extra whitespace on the edges of your plot.
plt.savefig("Roar-Cifar10-Resnet8.png", bbox_inches="tight")
plt.show()
