from typing import Dict, Optional
import os

from .base_io import (
    File,
    TextFile,
    PickleFile,
    JsonFile,
    NpyFile,
    NpcFile,
    TensorFile,
    Word2VecFile
)
from .custom_io import (
    CodeDescriptionFile,
    LabelCSVFile
)


__all__ = ['Files']


class Files:
    mimic_code_descriptions: Dict[str, Optional[CodeDescriptionFile]] = {
        'diagnosis': None,
        'procedure': None
    }
    icd9_descriptions: Optional[TextFile] = None
    label_csvs: Dict[str, Dict[str, Optional[LabelCSVFile]]] = {
        'mimic3': {
            'train': None,
            'dev': None,
            'test': None
        },
        'mimic3-50': {
            'train': None,
            'dev': None,
            'test': None
        },
        'mimic3-50l': {
            'train': None,
            'dev': None,
            'test': None
        }
    }
    datasets: Dict[str, Dict[str, Optional[JsonFile]]] = {
        'mimic3': {
            'train': None,
            'dev': None,
            'test': None
        },
        'mimic3-50': {
            'train': None,
            'dev': None,
            'test': None
        },
        'mimic3-50l': {
            'train': None,
            'dev': None,
            'test': None
        }
    }
    section_datasets: Dict[str, Dict[str, Optional[JsonFile]]] = {
        'mimic3': {
            'train': None,
            'dev': None,
            'test': None
        },
        'mimic3-50': {
            'train': None,
            'dev': None,
            'test': None
        },
        'mimic3-50l': {
            'train': None,
            'dev': None,
            'test': None
        }
    }
    word2vec_model: Optional[Word2VecFile] = None
    word_count_dict: Optional[JsonFile] = None
    code_synonyms: Optional[JsonFile] = None

    sections: Optional[JsonFile] = None

    @classmethod
    def create_path(cls, io_config):
        saved_path = io_config.saved_path
        paths = [
            saved_path
        ]
        for path in paths:
            if not os.path.exists(path):
                os.makedirs(path)

    @classmethod
    def init_path(cls, io_config):
        cls.create_path(io_config)

        data_path, saved_path = io_config.data_path, io_config.saved_path
        cls.mimic_code_descriptions = {
            'diagnosis': CodeDescriptionFile(os.path.join(data_path, 'D_ICD_DIAGNOSES.csv')),
            'procedure': CodeDescriptionFile(os.path.join(data_path, 'D_ICD_PROCEDURES.csv'))
        }
        cls.icd9_descriptions = TextFile(os.path.join(data_path, 'ICD9_descriptions'))

        for version, template in zip(['mimic3', 'mimic3-50', 'mimic3-50l'],
                                     ['{mode}_full.csv', '{mode}_50.csv', '{mode}_50l.csv']):
            for mode in ['train', 'dev', 'test']:
                cls.label_csvs[version][mode] = LabelCSVFile(
                    os.path.join(data_path, 'notes', template.format(mode=mode))
                )

        for version in ['mimic3', 'mimic3-50', 'mimic3-50l']:
            for mode in ['train', 'dev', 'test']:
                cls.datasets[version][mode] = JsonFile(
                    os.path.join(data_path, 'notes', f'{version}_{mode}.json')
                )
        for version in ['mimic3', 'mimic3-50', 'mimic3-50l']:
            for mode in ['train', 'dev', 'test']:
                cls.section_datasets[version][mode] = JsonFile(
                    os.path.join(data_path, 'sections', f'{version}_{mode}.json')
                )
        cls.word2vec_model = Word2VecFile(os.path.join(data_path, 'embeddings', 'word2vec_sg0_100.model'))
        cls.word_count_dict = JsonFile(os.path.join(data_path, 'embeddings', 'word_count_dict.json'))
        cls.code_synonyms = JsonFile(os.path.join(data_path, 'icd_mimic3_random_sort.json'))

        cls.sections = JsonFile(os.path.join(data_path, 'sections.json'))
