import abc
import os
import json
import pickle

import numpy as np
import torch
import gensim


class File(abc.ABC):
    def __init__(self, path, verbose=False):
        self.path = path
        self.verbose = verbose

    def _load(self, path):
        raise NotImplementedError

    def _save(self, path, content):
        raise NotImplementedError

    def load(self):
        if not self.exist():
            return None
        if self.verbose:
            print(f'loading {self.path} ...')
        return self._load(self.path)

    def save(self, content):
        if self.verbose:
            print(f'saving {self.path} ...')
        self._save(self.path, content)

    def exist(self):
        return os.path.exists(self.path)


class TextFile(File):
    def _load(self, path):
        with open(path, encoding='utf-8') as f:
            lines = f.readlines()
        return lines

    def _save(self, path, content):
        with open(path, 'w', encoding='utf-8') as f:
            f.writelines(content)


class PickleFile(File):
    def _load(self, path):
        return pickle.load(open(path, 'rb'))

    def _save(self, path, content):
        pickle.dump(content, open(path, 'wb'))


class JsonFile(File):
    def _load(self, path):
        return json.load(open(path, 'r', encoding='utf-8'))

    def _save(self, path, content):
        json.dump(content, open(path, 'w', encoding='utf-8'), indent=4)


class NpyFile(File):
    def _load(self, path):
        return np.load(path)

    def _save(self, path, content):
        np.save(path, content)


class NpcFile(File):
    def _load(self, path):
        return np.load(path)['content']

    def _save(self, path, content):
        np.savez_compressed(path, content=content)


class TensorFile(File):
    def _load(self, path):
        return torch.load(path)

    def _save(self, path, content):
        torch.save(content, path)


class Word2VecFile(File):
    def _load(self, path):
        model = gensim.models.Word2Vec.load(path)
        return model

    def _save(self, path, model):
        model.save(path)

