import torch
import torch.nn as nn
import numpy as np

from parameters import get_matrix, identity_matrix
from nn_modules import Blind, UnBlind, Conv2d_sp, Linear_sp
from numpy.linalg import inv
from torchvision.models.utils import load_state_dict_from_url
from collections import OrderedDict

__all__ = [
    'VGG', 'vgg11', 'vgg11_bn', 'vgg13', 'vgg13_bn', 'vgg16', 'vgg16_bn',
    'vgg19_bn', 'vgg19',
]


model_urls = {
    'vgg11': 'https://download.pytorch.org/models/vgg11-bbd30ac9.pth',
    'vgg13': 'https://download.pytorch.org/models/vgg13-c768596a.pth',
    'vgg16': 'https://download.pytorch.org/models/vgg16-397923af.pth',
    'vgg19': 'https://download.pytorch.org/models/vgg19-dcbb9e9d.pth',
    'vgg11_bn': 'https://download.pytorch.org/models/vgg11_bn-6002323d.pth',
    'vgg13_bn': 'https://download.pytorch.org/models/vgg13_bn-abd245e5.pth',
    'vgg16_bn': 'https://download.pytorch.org/models/vgg16_bn-6c64b313.pth',
    'vgg19_bn': 'https://download.pytorch.org/models/vgg19_bn-c79401a0.pth',
}

class VGG(nn.Module):

    def __init__(self, features, num_classes=1000, init_weights=True, 
                 bm=None, um=None, gm=None, igm=None, 
                 bias_list=None, use_bias=True):
        super(VGG, self).__init__()
        self.features = features
        self.avgpool = nn.AdaptiveAvgPool2d((7, 7))
        self.bm = bm
        self.gm = gm
        self.um = um
        self.igm= igm
        bias_list.append(4096)
        bias_list.append(4096)
        bias_list.append(num_classes)
        self.bias_list = bias_list
        self.classifier = nn.Sequential(
            nn.Linear(512 * 7 * 7, 4096, bias=use_bias),
            nn.ReLU(False),
            nn.Dropout(),
            nn.Linear(4096, 4096, bias=use_bias),
            nn.ReLU(False),
            nn.Dropout(),
            nn.Linear(4096, num_classes, bias=use_bias)
        )
        print(len(self.bias_list))
        if init_weights:
            self.initialize_weights()

    def load_imageNet_weight(self, name):
        state_dict = load_state_dict_from_url(model_urls[name],
                                              progress=True)
        self_state_dict = self.state_dict()
        new_state_dict = OrderedDict()
        iter_old = iter(state_dict)
        iter_self= iter(self_state_dict)

        for _ in range(len(state_dict)):
            key = next(iter_self)
            old_key = next(iter_old)
            val = state_dict[old_key]
            new_state_dict[key] = val

        self.load_state_dict(new_state_dict)

    def copy_state_dict(self, state_dict):
        self_state_dict = self.state_dict()
        new_state_dict = OrderedDict()
        iter_old = iter(state_dict)
        iter_self= iter(self_state_dict)
        #print(len(state_dict), len(self_state_dict))
        for _ in range(len(state_dict)):
            key = next(iter_self)
            old_key = next(iter_old)
            #print(key, old_key)
            val = state_dict[old_key]
            new_state_dict[key] = val

        self.load_state_dict(new_state_dict)

    def print_stat(self):
        for l in self.features:
            if isinstance(l, Blind):
                print(l.mean, l.std, l.max, l.min)
        for l in self.classifier:
            if isinstance(l, Blind):
                print(l.mean, l.std, l.max, l.min)

    def add_classifier(self, list, blind_list):
        list.append(nn.Sequential(self.avgpool, nn.Flatten()))
        blind_list.append(1)
        for idx, layer in enumerate(self.classifier):
            val = 0
            if idx == 1 or idx == 3:
                list.append(nn.Sequential(layer))
                val = 1
            elif idx == 2 or idx == 4 or idx == 0:
                list.append(nn.Sequential(layer))

            blind_list.append(val)

    def addNoise(self):
        features_list = []
        classifier_list = []
        idx = 0
        for layer in self.features:
            if isinstance(layer, nn.Dropout):
                continue

            append_ = isinstance(layer, nn.Conv2d)
            if (append_):

                features_list.append(Blind(self.bm, self.igm))

            features_list.append(layer)
            if (append_):
                features_list.append(UnBlind(self.um, self.gm, self.bias_list[idx], use_bias=True))
                idx += 1

        for layer in self.classifier:
            append_ = isinstance(layer, nn.Linear)
            if (append_):
                
                classifier_list.append(Blind(self.bm, self.igm))

            classifier_list.append(layer)
            if (append_):
                classifier_list.append(UnBlind(self.um, self.gm, self.bias_list[idx], use_bias=True))
                idx += 1
        self.features = nn.Sequential(*features_list)
        self.classifier=nn.Sequential(*classifier_list)

    def forward(self, x):
        x = self.features(x)
        x = self.avgpool(x)
        x = torch.flatten(x, 1)
        x = self.classifier(x)
        return x


    def initialize_weights(self):
        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
                if m.bias is not None:
                    nn.init.constant_(m.bias, 0)
            elif isinstance(m, nn.BatchNorm2d):
                nn.init.constant_(m.weight, 1)
                nn.init.constant_(m.bias, 0)
            elif isinstance(m, nn.Linear):
                nn.init.normal_(m.weight, 0, 0.01)
                if m.bias is not None:
                    nn.init.constant_(m.bias, 0)
            elif isinstance(m, UnBlind):
                nn.init.constant_(m.bias, 0)



def make_layers(cfg, batch_norm=False, bm=None, um=None, gm=None, igm=None, use_bias=True):
    layers = []
    in_channels = 3
    bias_size = []
    for v in cfg:
        if v == 'M':
            layers += [nn.MaxPool2d(kernel_size=2, stride=2)]
        else:
            bias_size.append(v)
            conv2d = nn.Conv2d(in_channels, v, kernel_size=3, padding=1, bias=use_bias)
            if batch_norm:
                layers += [conv2d, nn.BatchNorm2d(v), nn.ReLU(inplace=False)]
            else:
                layers += [conv2d, nn.ReLU(inplace=False)]
            in_channels = v
    return nn.Sequential(*layers), bias_size


cfgs = {
    'A': [64, 'M', 128, 'M', 256, 256, 'M', 512, 512, 'M', 512, 512, 'M'],
    'B': [64, 64, 'M', 128, 128, 'M', 256, 256, 'M', 512, 512, 'M', 512, 512, 'M'],
    'D': [64, 64, 'M', 128, 128, 'M', 256, 256, 256, 'M', 512, 512, 512, 'M', 512, 512, 512, 'M'],
    'E': [64, 64, 'M', 128, 128, 'M', 256, 256, 256, 256, 'M', 512, 512, 512, 512, 'M', 512, 512, 512, 512, 'M'],
}


def _vgg(arch, cfg, batch_norm, pretrained, progress, device=None, dtype=None, use_bias=True, **kwargs):
    if pretrained:
        kwargs['init_weights'] = False

    print("if using bias", use_bias)
    bm, um, gm, igm = get_matrix(device, dtype)
    print(bm, um, gm, igm)
    features, bias_list = make_layers(cfgs[cfg], batch_norm=batch_norm, use_bias=use_bias)
    model = VGG(features, bm=bm, um=um, gm=gm, igm=igm, bias_list=bias_list, use_bias=use_bias, **kwargs)
    
    return model


def vgg11(pretrained=False, progress=True, **kwargs):
    r"""VGG 11-layer model (configuration "A") from                                                                              
    `"Very Deep Convolutional Networks For Large-Scale Image Recognition" <https://arxiv.org/pdf/1409.1556.pdf>`_                
    Args:                                                                                                                        
        pretrained (bool): If True, returns a model pre-trained on ImageNet                                                      
        progress (bool): If True, displays a progress bar of the download to stderr                                              
    """
    return _vgg('vgg11', 'A', False, pretrained, progress, **kwargs)


def vgg11_bn(pretrained=False, progress=True, **kwargs):
    r"""VGG 11-layer model (configuration "A") with batch normalization                                                          
    `"Very Deep Convolutional Networks For Large-Scale Image Recognition" <https://arxiv.org/pdf/1409.1556.pdf>`_                
    Args:                                                                                                                        
        pretrained (bool): If True, returns a model pre-trained on ImageNet                                                      
        progress (bool): If True, displays a progress bar of the download to stderr                                              
    """
    return _vgg('vgg11_bn', 'A', True, pretrained, progress, **kwargs)

def vgg13(pretrained=False, progress=True, **kwargs):
    r"""VGG 13-layer model (configuration "B")                                                                                   
    `"Very Deep Convolutional Networks For Large-Scale Image Recognition" <https://arxiv.org/pdf/1409.1556.pdf>`_                
    Args:                                                                                                                        
        pretrained (bool): If True, returns a model pre-trained on ImageNet                                                      
        progress (bool): If True, displays a progress bar of the download to stderr                                              
    """
    return _vgg('vgg13', 'B', False, pretrained, progress, **kwargs)


def vgg13_bn(pretrained=False, progress=True, **kwargs):
    r"""VGG 13-layer model (configuration "B") with batch normalization                                                          
    `"Very Deep Convolutional Networks For Large-Scale Image Recognition" <https://arxiv.org/pdf/1409.1556.pdf>`_                
    Args:                                                                                                                        
        pretrained (bool): If True, returns a model pre-trained on ImageNet                                                      
        progress (bool): If True, displays a progress bar of the download to stderr                                              
    """
    return _vgg('vgg13_bn', 'B', True, pretrained, progress, **kwargs)


def vgg16(pretrained=False, progress=True, dtype=None, device=None, **kwargs):
    r"""VGG 16-layer model (configuration "D")                                                                                   
    `"Very Deep Convolutional Networks For Large-Scale Image Recognition" <https://arxiv.org/pdf/1409.1556.pdf>`_                
    Args:                                                                                                                        
        pretrained (bool): If True, returns a model pre-trained on ImageNet                                                      
        progress (bool): If True, displays a progress bar of the download to stderr                                              
    """
    return _vgg('vgg16', 'D', False, pretrained, progress, device=device, dtype=dtype, **kwargs)

def vgg16_bn(pretrained=False, progress=True, **kwargs):
    r"""VGG 16-layer model (configuration "D") with batch normalization                                                          
    `"Very Deep Convolutional Networks For Large-Scale Image Recognition" <https://arxiv.org/pdf/1409.1556.pdf>`_                
    Args:                                                                                                                        
        pretrained (bool): If True, returns a model pre-trained on ImageNet                                                      
        progress (bool): If True, displays a progress bar of the download to stderr                                              
    """
    return _vgg('vgg16_bn', 'D', True, pretrained, progress, **kwargs)


def vgg19(pretrained=False, progress=True, dtype=None, device=None, **kwargs):
    r"""VGG 19-layer model (configuration "E")                                                                                   
    `"Very Deep Convolutional Networks For Large-Scale Image Recognition" <https://arxiv.org/pdf/1409.1556.pdf>`_                
    Args:                                                                                                                        
        pretrained (bool): If True, returns a model pre-trained on ImageNet                                                      
        progress (bool): If True, displays a progress bar of the download to stderr                                              
    """
    return _vgg('vgg19', 'E', False, pretrained, progress, device=device, dtype=dtype, **kwargs)


def vgg19_bn(pretrained=False, progress=True, **kwargs):
    r"""VGG 19-layer model (configuration 'E') with batch normalization                                                          
    `"Very Deep Convolutional Networks For Large-Scale Image Recognition" <https://arxiv.org/pdf/1409.1556.pdf>`_                
    Args:                                                                                                                        
        pretrained (bool): If True, returns a model pre-trained on ImageNet                                                      
        progress (bool): If True, displays a progress bar of the download to stderr                                              
    """
    return _vgg('vgg19_bn', 'E', True, pretrained, progress, **kwargs)