#include <iostream>
#include "tensorflow/core/framework/numeric_op.h"
#include "tensorflow/core/framework/op_kernel.h"
#include "tensorflow/core/framework/register_types.h"
#include "tensorflow/core/framework/tensor.h"
#include "tensorflow/core/framework/tensor_types.h"
#include "tensorflow/core/framework/common_shape_fns.h"
#include "tensorflow/core/common_runtime/threadpool_device.h"

#include "third_party/eigen3/unsupported/Eigen/CXX11/Tensor"
#include <dlfcn.h>

using namespace std;
using namespace tensorflow;

template <typename Device, typename T>
class ResnetActivationOp : public OpKernel {
  public:
    explicit ResnetActivationOp(OpKernelConstruction* context) : OpKernel(context) {
        OP_REQUIRES_OK(context, context->GetAttr("eid_low", &eid_low_));
        OP_REQUIRES_OK(context, context->GetAttr("eid_high", &eid_high_));
        lib_ = dlopen("App/enclave_bridge.so", RTLD_NOW);
        OP_REQUIRES(context, lib_ != NULL, errors::Unknown("Unable to load .so"));
        OP_REQUIRES_OK(context, context->GetAttr("mode", &act_mode_));

    }

    void Compute(OpKernelContext* context) override {
        const Tensor& input = context->input(0);
        const Tensor& means = context->input(1);
        
        unsigned long int eid_ = (eid_high_ << 32) + eid_low_;
        
        auto output_shape = input.shape();
        if (act_mode_ == "bnrelupool") {
            output_shape = TensorShape({
             input.shape().dim_sizes()[0],
             input.shape().dim_sizes()[1]/2-1,
             input.shape().dim_sizes()[2]/2-1,
             input.shape().dim_sizes()[3]
             });
        }

        Tensor* output = nullptr;
        OP_REQUIRES_OK(context, context->allocate_output(0, output_shape, &output));
        typedef void(* res_act) (unsigned long int, float*, float*, float*);
        dlerror();
        res_act act_fwd = (res_act) dlsym(lib_, "resnet_activation_fwd");
        const char *dlsym_error = dlerror();
        OP_REQUIRES(context, !dlsym_error, errors::Unknown("loading of relu failed: ", dlsym_error));

        act_fwd(eid_,
               (float*) input.flat<T>().data(),
               (float*) output->flat<T>().data(),
               (float*) means.flat<T>().data()
               );
    }
  private:
    void* lib_;
    std::string act_mode_;
    int64 eid_low_;
    int64 eid_high_;

};  // class ResnetActivationOp

typedef Eigen::ThreadPoolDevice CPUDevice;

REGISTER_KERNEL_BUILDER(Name("ResnetActivation").Device(DEVICE_CPU), ResnetActivationOp<CPUDevice, float>);

REGISTER_OP("ResnetActivation")
    .Attr("eid_low: int")
    .Attr("eid_high: int")
    .Attr("mode: string")
    .Input("src: float")
    .Input("means: float")
    .Output("dst: float")
    .SetShapeFn([](::tensorflow::shape_inference::InferenceContext* c) {
      std::string act_mode;
      c->GetAttr("mode", &act_mode);
      if (act_mode == "bnrelupool") {

        auto N = c->Dim(c->input(0), 0);
        auto H = c->Dim(c->input(0), 1);
        auto W = c->Dim(c->input(0), 2);
        auto C = c->Dim(c->input(0), 3); 

        H = c->MakeDim(c->Value(H) / 2 - 1);
        W = c->MakeDim(c->Value(W) / 2 - 1);
        c->set_output(0, c->MakeShape({N, H, W, C}));
        return Status::OK();
      }
      c->set_output(0, c->input(0));
      return Status::OK();
      
    });
    