import torch
import torch.nn as nn
from torchvision.models.utils import load_state_dict_from_url
from transform import group_detach, unblind, blind


__all__ = ['ResNet', 'resnet18', 'resnet34', 'resnet50', 'resnet101',
           'resnet152', 'resnext50_32x4d', 'resnext101_32x8d',
           'wide_resnet50_2', 'wide_resnet101_2']


model_urls = {
    'resnet18': 'https://download.pytorch.org/models/resnet18-5c106cde.pth',
    'resnet34': 'https://download.pytorch.org/models/resnet34-333f7ec4.pth',
    'resnet50': 'https://download.pytorch.org/models/resnet50-19c8e357.pth',
    'resnet101': 'https://download.pytorch.org/models/resnet101-5d3b4d8f.pth',
    'resnet152': 'https://download.pytorch.org/models/resnet152-b121ed2d.pth',
    'resnext50_32x4d': 'https://download.pytorch.org/models/resnext50_32x4d-7cdf4587.pth',
    'resnext101_32x8d': 'https://download.pytorch.org/models/resnext101_32x8d-8ba56ff5.pth',
    'wide_resnet50_2': 'https://download.pytorch.org/models/wide_resnet50_2-95faca4d.pth',
    'wide_resnet101_2': 'https://download.pytorch.org/models/wide_resnet101_2-32ee1156.pth',
}

def forward_single(transformed_list, blind_list, blind_mat, unblind_mat, is_blind, image):

    node_list = []
    loss_list = []
    grad_list = []
    prev_val = image

    if is_blind:
        prev_val = group_detach(blind(image, blind_mat))

    list_range = len(blind_list)

    for idx, id1 in enumerate(range(list_range)):
        if idx != 0:
            prev_val = group_detach(node_list[idx-1])        

        if blind_list[idx] == 1 and is_blind:
            # unblind
            prev_val = group_detach(unblind(prev_val, unblind_mat))

        for val in prev_val:
            val.requires_grad = True
        
        grad_list.append(prev_val)                    
    
        local_res = []

        for bdx in range(len(transformed_list)):
            #print(transformed_list[bdx][idx])
            res = transformed_list[bdx][idx](prev_val[bdx])
            local_res.append(res)
        loss_list.append(local_res)
        final_res = local_res
        if blind_list[idx] == 1 and is_blind:
            final_res = group_detach(blind(local_res, blind_mat))
        final_res = group_detach(final_res)
        node_list.append(final_res)

    return final_res, node_list, loss_list, grad_list

def backward_single(gradient, node_list, loss_list, grad_list):
    res_grad = []
    for bptr, grad in enumerate(gradient):
        prev_grad = grad
            
        for idx in range(len(loss_list)):
            newidx = len(node_list) - idx - 1
            
            loss = loss_list[newidx][bptr]
            loss.backward(gradient=prev_grad)
            # cleaning
            if idx == 0:
                del grad
            else:
                del grad_list[newidx+1][bptr].grad
            
            prev_grad = grad_list[newidx][bptr].grad

        res_grad.append(prev_grad)
    return res_grad

def conv3x3(in_planes, out_planes, stride=1, groups=1, dilation=1):
    """3x3 convolution with padding"""
    return nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride,
                     padding=dilation, groups=groups, bias=False, dilation=dilation)


def conv1x1(in_planes, out_planes, stride=1):
    """1x1 convolution"""
    return nn.Conv2d(in_planes, out_planes, kernel_size=1, stride=stride, bias=False)


class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, inplanes, planes, stride=1, downsample=None, groups=1,
                 base_width=64, dilation=1, norm_layer=None):
        super(BasicBlock, self).__init__()
        if norm_layer is None:
            norm_layer = nn.BatchNorm2d
        if groups != 1 or base_width != 64:
            raise ValueError('BasicBlock only supports groups=1 and base_width=64')
        if dilation > 1:
            raise NotImplementedError("Dilation > 1 not supported in BasicBlock")
        # Both self.conv1 and self.downsample layers downsample the input when stride != 1
        self.conv1 = conv3x3(inplanes, planes, stride)
        self.bn1 = norm_layer(planes)
        self.relu = nn.ReLU(inplace=False)
        self.conv2 = conv3x3(planes, planes)
        self.bn2 = norm_layer(planes)
        self.downsample = downsample
        self.stride = stride

    def forward(self, x):
        identity = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)

        if self.downsample is not None:
            identity = self.downsample(x)

        out += identity
        out = self.relu(out)

        return out


class Bottleneck(nn.Module):
    # Bottleneck in torchvision places the stride for downsampling at 3x3 convolution(self.conv2)
    # while original implementation places the stride at the first 1x1 convolution(self.conv1)
    # according to "Deep residual learning for image recognition"https://arxiv.org/abs/1512.03385.
    # This variant is also known as ResNet V1.5 and improves accuracy according to
    # https://ngc.nvidia.com/catalog/model-scripts/nvidia:resnet_50_v1_5_for_pytorch.

    expansion = 4

    def __init__(self, inplanes, planes, stride=1, downsample=None, groups=1,
                 base_width=64, dilation=1, norm_layer=None, blind_mat=None, unblind_mat=None, is_blind=False):
        super(Bottleneck, self).__init__()
        if norm_layer is None:
            norm_layer = nn.BatchNorm2d
        width = int(planes * (base_width / 64.)) * groups
        # Both self.conv2 and self.downsample layers downsample the input when stride != 1
        self.conv1 = conv1x1(inplanes, width)
        self.bn1 = norm_layer(width)
        self.conv2 = conv3x3(width, width, stride, groups, dilation)
        self.bn2 = norm_layer(width)
        self.conv3 = conv1x1(width, planes * self.expansion)
        self.bn3 = norm_layer(planes * self.expansion)
        self.relu = nn.ReLU(inplace=False)
        self.downsample = downsample
        self.stride = stride
        self.unblind_mat = unblind_mat
        self.blind_mat   = blind_mat
        self.blind_list = []
        self.seq_list = [[]]
        self.skip_list = [[]]
        self.skip_blind = []
        self.down1_res = []
        self.is_blind = is_blind
        self.output_loss = None
        self.skip_loss_list = None
        self.skip_grad_list = None
        self.skip_node_list = None

    def build_list(self):
        self.seq_list[0].append(nn.Sequential(self.conv1))
        self.blind_list.append(0)
        self.seq_list[0].append(nn.Sequential(self.bn1, nn.ReLU(inplace=False)))
        self.blind_list.append(1)
        self.seq_list[0].append(nn.Sequential(self.conv2))
        self.blind_list.append(0)
        self.seq_list[0].append(nn.Sequential(self.bn2, nn.ReLU(inplace=False)))
        self.blind_list.append(1)
        self.seq_list[0].append(nn.Sequential(self.conv3))
        self.blind_list.append(0)
        self.seq_list[0].append(nn.Sequential(self.bn3))
        self.blind_list.append(1)

        if self.downsample is not None:
            self.skip_list[0].append(nn.Sequential(self.downsample[0]))
            self.skip_blind.append(0)
            self.skip_list[0].append(nn.Sequential(self.downsample[1]))
            self.skip_blind.append(1)

    def merge_list(self, model):
        self.seq_list.append(model.seq_list[0])
        self.skip_list.append(model.skip_list[0])


    def custom_forward(self, list_x):
        self.node_list = []
        self.loss_list = []
        self.grad_list = []
        self.skip_loss_list = []
        self.skip_node_list = []
        self.skip_grad_list = []
        identity = group_detach(list_x)
        for iden in identity:
            iden.requires_grad = True
        # identity
        if self.downsample is not None:
            self.skip_grad_list.append(identity)
            self.down0_res = []
            for bdx in range(len(self.skip_list)):
                res = self.skip_list[bdx][0](identity[bdx])
                self.down0_res.append(res)
                
            self.skip_loss_list.append(self.down0_res)
            self.skip_node_list.append(self.down0_res)

            if self.is_blind:
                self.down1_in = group_detach(unblind(self.down0_res, self.unblind_mat))
            else:
                self.down1_in = group_detach(self.down0_res)

            self.skip_grad_list.append(self.down1_in)
            self.down1_res = []
            for bdx in range(len(self.skip_list)):
                self.down1_in[bdx].requires_grad = True
                res = self.skip_list[bdx][1](self.down1_in[bdx])
                
                self.down1_res.append(res)
            self.skip_loss_list.append(self.down1_res)
            self.skip_node_list.append(self.down1_res)
        else:
            self.skip_grad_list.append(identity)
            self.down1_res = identity
            self.skip_loss_list.append(self.down1_res)
            self.skip_node_list.append(self.down1_res)

        self.down1_res = group_detach(self.down1_res)

        self.normal_res, self.node_list, self.loss_list, self.grad_list = \
            forward_single(self.seq_list, self.blind_list, self.blind_mat, self.unblind_mat, self.is_blind, identity)

        if self.is_blind:
            self.final_unblinded = group_detach(unblind(self.normal_res, self.unblind_mat))
        else:
            self.final_unblinded = group_detach(self.normal_res)

        out_put = []
        for bdx in range(len(self.skip_list)):
            self.final_unblinded[bdx].requires_grad=True
            self.down1_res[bdx].requires_grad=True
            to_relu = self.final_unblinded[bdx] + self.down1_res[bdx]
            out_put.append(self.relu(to_relu))

        self.output_loss = out_put
        
        if self.is_blind:
            out_put = group_detach(unblind(out_put, self.unblind_mat))

        out_put = group_detach(out_put)
        return out_put

    def custom_backward(self, gradient):
        next_grad_normal = []
        next_grad_skip   = []
        for bdx, grad in enumerate(gradient):
            self.output_loss[bdx].backward(grad)
            next_grad_skip.append(self.down1_res[bdx].grad)
            next_grad_normal.append(self.final_unblinded[bdx].grad)

        # backwards skip
        ret_skip = backward_single(next_grad_skip, self.skip_node_list, self.skip_loss_list, self.skip_grad_list)

        # backwards normal
        ret_grad = backward_single(next_grad_normal, self.node_list, self.loss_list, self.grad_list)
        

        return ret_grad

    def forward(self, x):
        identity = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)
        out = self.relu(out)

        out = self.conv3(out)
        out = self.bn3(out)

        if self.downsample is not None:
            identity = self.downsample(x)

        out += identity
        out = self.relu(out)

        return out


class ResNet(nn.Module):

    def __init__(self, block, layers, num_classes=1000, zero_init_residual=False,
                 groups=1, width_per_group=64, replace_stride_with_dilation=None,
                 norm_layer=None, blind_mat=None, unblind_mat=None, is_blind=False):
        super(ResNet, self).__init__()
        if norm_layer is None:
            norm_layer = nn.BatchNorm2d
        self._norm_layer = norm_layer

        self.inplanes = 64
        self.dilation = 1
        if replace_stride_with_dilation is None:
            # each element in the tuple indicates if we should replace
            # the 2x2 stride with a dilated convolution instead
            replace_stride_with_dilation = [False, False, False]
        if len(replace_stride_with_dilation) != 3:
            raise ValueError("replace_stride_with_dilation should be None "
                             "or a 3-element tuple, got {}".format(replace_stride_with_dilation))
        self.groups = groups
        self.base_width = width_per_group
        self.conv1 = nn.Conv2d(3, self.inplanes, kernel_size=7, stride=2, padding=3,
                               bias=False)
        self.bn1 = norm_layer(self.inplanes)
        self.relu = nn.ReLU(inplace=False)
        self.maxpool = nn.MaxPool2d(kernel_size=3, stride=2, padding=1)
        self.layer1 = self._make_layer(block, 64, layers[0], unblind_mat=unblind_mat, blind_mat=blind_mat)
        self.layer2 = self._make_layer(block, 128, layers[1], stride=2,
                                       dilate=replace_stride_with_dilation[0], unblind_mat=unblind_mat, blind_mat=blind_mat)
        self.layer3 = self._make_layer(block, 256, layers[2], stride=2,
                                       dilate=replace_stride_with_dilation[1], unblind_mat=unblind_mat, blind_mat=blind_mat)
        self.layer4 = self._make_layer(block, 512, layers[3], stride=2,
                                       dilate=replace_stride_with_dilation[2], unblind_mat=unblind_mat, blind_mat=blind_mat)
        self.avgpool = nn.AdaptiveAvgPool2d((1, 1))
        self.fc = nn.Linear(512 * block.expansion, num_classes)
        self.unblind_mat = unblind_mat
        self.blind_mat   = blind_mat
        self.is_blind = is_blind

        self.node_list1 = None
        self.loss_list1 = None
        self.grad_list1 = None

        self.node_list0 = None
        self.loss_list0 = None
        self.grad_list0 = None

        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
            elif isinstance(m, (nn.BatchNorm2d, nn.GroupNorm)):
                nn.init.constant_(m.weight, 1)
                nn.init.constant_(m.bias, 0)

        # Zero-initialize the last BN in each residual branch,
        # so that the residual branch starts with zeros, and each residual block behaves like an identity.
        # This improves the model by 0.2~0.3% according to https://arxiv.org/abs/1706.02677
        if zero_init_residual:
            for m in self.modules():
                if isinstance(m, Bottleneck):
                    nn.init.constant_(m.bn3.weight, 0)
                elif isinstance(m, BasicBlock):
                    nn.init.constant_(m.bn2.weight, 0)

    def _make_layer(self, block, planes, blocks, stride=1, dilate=False, unblind_mat=None, blind_mat=None):
        norm_layer = self._norm_layer
        downsample = None
        previous_dilation = self.dilation
        if dilate:
            self.dilation *= stride
            stride = 1
        if stride != 1 or self.inplanes != planes * block.expansion:
            downsample = nn.Sequential(
                conv1x1(self.inplanes, planes * block.expansion, stride),
                norm_layer(planes * block.expansion),
            )

        layers = []
        layers.append(block(self.inplanes, planes, stride, downsample, self.groups,
                            self.base_width, previous_dilation, norm_layer, unblind_mat=unblind_mat, blind_mat=blind_mat))
        self.inplanes = planes * block.expansion
        for _ in range(1, blocks):
            layers.append(block(self.inplanes, planes, groups=self.groups,
                                base_width=self.base_width, dilation=self.dilation,
                                norm_layer=norm_layer, unblind_mat=unblind_mat, blind_mat=blind_mat))

        return nn.Sequential(*layers)

    def _forward_impl(self, x):
        # See note [TorchScript super()]
        x = self.conv1(x)
        x = self.bn1(x)
        x = self.relu(x)
        x = self.maxpool(x)

        x = self.layer1(x)
        x = self.layer2(x)
        x = self.layer3(x)
        x = self.layer4(x)

        x = self.avgpool(x)
        x = torch.flatten(x, 1)
        x = self.fc(x)

        return x

    def forward(self, x):
        return self._forward_impl(x)

    def custom_forward(self, x):
        res0, self.node_list0, self.loss_list0, self.grad_list0 = forward_single(self.seq_list, self.blind_list, self.blind_mat, self.unblind_mat, self.is_blind, x)

        res1 = res0        
        for bottle in self.layer1:
            res1 = bottle.custom_forward(res1)
        

        for bottle in self.layer2:
            res1 = bottle.custom_forward(res1)
        for bottle in self.layer3:
            res1 = bottle.custom_forward(res1)
        for bottle in self.layer4:
            res1 = bottle.custom_forward(res1)

        final_res, self.node_list1, self.loss_list1, self.grad_list1 = forward_single(self.seq_list1, self.blind_list1, self.blind_mat, self.unblind_mat, self.is_blind, res1)

        return final_res

    def custom_backward(self, gradient):
        
        prev_grad = backward_single(gradient, self.node_list1, self.loss_list1, self.grad_list1)

        for idx in range(len(self.layer4)):
            prev_grad = self.layer4[len(self.layer4)- idx - 1].custom_backward(gradient=prev_grad)

        for idx in range(len(self.layer3)):
            prev_grad = self.layer3[len(self.layer3)- idx - 1].custom_backward(gradient=prev_grad)

        for idx in range(len(self.layer2)):
            prev_grad = self.layer2[len(self.layer2)- idx - 1].custom_backward(gradient=prev_grad)

        for idx in range(len(self.layer1)):
            prev_grad = self.layer1[len(self.layer1)- idx - 1].custom_backward(gradient=prev_grad)

        backward_single(prev_grad, self.node_list0, self.loss_list0, self.grad_list0)


    def build_list(self):
        self.seq_list = [[]]
        self.blind_list = []
        self.seq_list[0].append(nn.Sequential(self.conv1))
        self.blind_list.append(0)
        self.seq_list[0].append(nn.Sequential(self.bn1, nn.ReLU(inplace=False), self.maxpool))
        self.blind_list.append(1)

        for bottle in self.layer1:
            bottle.build_list()
        for bottle in self.layer2:
            bottle.build_list()
        for bottle in self.layer3:
            bottle.build_list()
        for bottle in self.layer4:
            bottle.build_list()
                
        self.seq_list1 = [[]]
        self.blind_list1 = []
        self.seq_list1[0].append(nn.Sequential(self.avgpool))
        self.blind_list1.append(1)
        self.seq_list1[0].append(nn.Sequential(nn.Flatten(), self.fc))
        self.blind_list1.append(0)


    def merge_list(self, other):
        for model in other:
            self.seq_list.append(model.seq_list[0])

            for idx, bottle in enumerate(self.layer1):
                bottle.merge_list(model.layer1[idx])
            for idx, bottle in enumerate(self.layer2):
                bottle.merge_list(model.layer2[idx])
            for idx, bottle in enumerate(self.layer3):
                bottle.merge_list(model.layer3[idx])
            for idx, bottle in enumerate(self.layer4):
                bottle.merge_list(model.layer4[idx])



            self.seq_list1.append(model.seq_list1[0])


def _resnet(arch, block, layers, pretrained, progress, unblind_mat=None, blind_mat=None, **kwargs):
    model = ResNet(block, layers, unblind_mat=unblind_mat, blind_mat=blind_mat, **kwargs)
    if pretrained:
        state_dict = load_state_dict_from_url(model_urls[arch],
                                              progress=progress)
        model.load_state_dict(state_dict)
    return model


def resnet18(pretrained=False, progress=True, **kwargs):
    r"""ResNet-18 model from
    `"Deep Residual Learning for Image Recognition" <https://arxiv.org/pdf/1512.03385.pdf>`_
    Args:
        pretrained (bool): If True, returns a model pre-trained on ImageNet
        progress (bool): If True, displays a progress bar of the download to stderr
    """
    return _resnet('resnet18', BasicBlock, [2, 2, 2, 2], pretrained, progress,
                   **kwargs)


def resnet34(pretrained=False, progress=True, **kwargs):
    r"""ResNet-34 model from
    `"Deep Residual Learning for Image Recognition" <https://arxiv.org/pdf/1512.03385.pdf>`_
    Args:
        pretrained (bool): If True, returns a model pre-trained on ImageNet
        progress (bool): If True, displays a progress bar of the download to stderr
    """
    return _resnet('resnet34', BasicBlock, [3, 4, 6, 3], pretrained, progress,
                   **kwargs)


def resnet50(pretrained=False, progress=True, unblind_mat=None, blind_mat=None, **kwargs):
    r"""ResNet-50 model from
    `"Deep Residual Learning for Image Recognition" <https://arxiv.org/pdf/1512.03385.pdf>`_
    Args:
        pretrained (bool): If True, returns a model pre-trained on ImageNet
        progress (bool): If True, displays a progress bar of the download to stderr
    """
    return _resnet('resnet50', Bottleneck, [3, 4, 6, 3], pretrained, progress, unblind_mat=unblind_mat, blind_mat=blind_mat,
                   **kwargs)


def resnet101(pretrained=False, progress=True, **kwargs):
    r"""ResNet-101 model from
    `"Deep Residual Learning for Image Recognition" <https://arxiv.org/pdf/1512.03385.pdf>`_
    Args:
        pretrained (bool): If True, returns a model pre-trained on ImageNet
        progress (bool): If True, displays a progress bar of the download to stderr
    """
    return _resnet('resnet101', Bottleneck, [3, 4, 23, 3], pretrained, progress,
                   **kwargs)


def resnet152(pretrained=False, progress=True, **kwargs):
    r"""ResNet-152 model from
    `"Deep Residual Learning for Image Recognition" <https://arxiv.org/pdf/1512.03385.pdf>`_
    Args:
        pretrained (bool): If True, returns a model pre-trained on ImageNet
        progress (bool): If True, displays a progress bar of the download to stderr
    """
    return _resnet('resnet152', Bottleneck, [3, 8, 36, 3], pretrained, progress,
                   **kwargs)


def resnext50_32x4d(pretrained=False, progress=True, **kwargs):
    r"""ResNeXt-50 32x4d model from
    `"Aggregated Residual Transformation for Deep Neural Networks" <https://arxiv.org/pdf/1611.05431.pdf>`_
    Args:
        pretrained (bool): If True, returns a model pre-trained on ImageNet
        progress (bool): If True, displays a progress bar of the download to stderr
    """
    kwargs['groups'] = 32
    kwargs['width_per_group'] = 4
    return _resnet('resnext50_32x4d', Bottleneck, [3, 4, 6, 3],
                   pretrained, progress, **kwargs)


def resnext101_32x8d(pretrained=False, progress=True, **kwargs):
    r"""ResNeXt-101 32x8d model from
    `"Aggregated Residual Transformation for Deep Neural Networks" <https://arxiv.org/pdf/1611.05431.pdf>`_
    Args:
        pretrained (bool): If True, returns a model pre-trained on ImageNet
        progress (bool): If True, displays a progress bar of the download to stderr
    """
    kwargs['groups'] = 32
    kwargs['width_per_group'] = 8
    return _resnet('resnext101_32x8d', Bottleneck, [3, 4, 23, 3],
                   pretrained, progress, **kwargs)


def wide_resnet50_2(pretrained=False, progress=True, **kwargs):
    r"""Wide ResNet-50-2 model from
    `"Wide Residual Networks" <https://arxiv.org/pdf/1605.07146.pdf>`_
    The model is the same as ResNet except for the bottleneck number of channels
    which is twice larger in every block. The number of channels in outer 1x1
    convolutions is the same, e.g. last block in ResNet-50 has 2048-512-2048
    channels, and in Wide ResNet-50-2 has 2048-1024-2048.
    Args:
        pretrained (bool): If True, returns a model pre-trained on ImageNet
        progress (bool): If True, displays a progress bar of the download to stderr
    """
    kwargs['width_per_group'] = 64 * 2
    return _resnet('wide_resnet50_2', Bottleneck, [3, 4, 6, 3],
                   pretrained, progress, **kwargs)


def wide_resnet101_2(pretrained=False, progress=True, **kwargs):
    r"""Wide ResNet-101-2 model from
    `"Wide Residual Networks" <https://arxiv.org/pdf/1605.07146.pdf>`_
    The model is the same as ResNet except for the bottleneck number of channels
    which is twice larger in every block. The number of channels in outer 1x1
    convolutions is the same, e.g. last block in ResNet-50 has 2048-512-2048
    channels, and in Wide ResNet-50-2 has 2048-1024-2048.
    Args:
        pretrained (bool): If True, returns a model pre-trained on ImageNet
        progress (bool): If True, displays a progress bar of the download to stderr
    """
    kwargs['width_per_group'] = 64 * 2
    return _resnet('wide_resnet101_2', Bottleneck, [3, 4, 23, 3],
                   pretrained, progress, **kwargs)